package gotel

import (
	"io"
	"net/http"
)

// WrapResponseWriter is a proxy around an http.ResponseWriter that allows you to hook
// into various parts of the response process.
// The original work was derived from Chi's middleware, source:
// https://github.com/go-chi/chi/blob/master/middleware/wrap_writer.go
type WrapResponseWriter interface {
	http.ResponseWriter
	// Status returns the HTTP status of the request, or 0 if one has not
	// yet been sent.
	Status() int
	// BytesWritten returns the total number of bytes sent to the client.
	BytesWritten() int
	// Tee causes the response body to be written to the given io.Writer in
	// addition to proxying the writes through. Only one io.Writer can be
	// tee'd to at once: setting a second one will overwrite the first.
	// Writes will be sent to the proxy before being written to this
	// io.Writer. It is illegal for the tee'd writer to be modified
	// concurrently with writes.
	Tee(w io.Writer)
	// Unwrap returns the original proxied target.
	Unwrap() http.ResponseWriter
	// Discard causes all writes to the original ResponseWriter be discarded,
	// instead writing only to the tee'd writer if it's set.
	// The caller is responsible for calling WriteHeader and Write on the
	// original ResponseWriter once the processing is done.
	Discard()
}

// NewWrapResponseWriterFunc abstracts an interface for the http.ResponseWriter wrapper constructor.
type NewWrapResponseWriterFunc func(w http.ResponseWriter, protoMajor int) WrapResponseWriter

// basicWriter wraps a http.ResponseWriter that implements the minimal
// http.ResponseWriter interface.
type basicWriter struct {
	http.ResponseWriter

	tee         io.Writer
	code        int
	bytes       int
	wroteHeader bool
	discard     bool
}

func (b *basicWriter) WriteHeader(code int) {
	if code >= 100 && code <= 199 && code != http.StatusSwitchingProtocols {
		if !b.discard {
			b.ResponseWriter.WriteHeader(code)
		}
	} else if !b.wroteHeader {
		b.code = code

		b.wroteHeader = true
		if !b.discard {
			b.ResponseWriter.WriteHeader(code)
		}
	}
}

func (b *basicWriter) Write(buf []byte) (int, error) {
	b.maybeWriteHeader()

	var (
		n   int
		err error
	)

	switch {
	case !b.discard:
		n, err = b.ResponseWriter.Write(buf)
		if b.tee != nil {
			_, err2 := b.tee.Write(buf[:n])
			// Prefer errors generated by the proxied writer.
			if err == nil {
				err = err2
			}
		}
	case b.tee != nil:
		n, err = b.tee.Write(buf)
	default:
		n, err = io.Discard.Write(buf)
	}

	b.bytes += n

	return n, err
}

func (b *basicWriter) Status() int {
	return b.code
}

func (b *basicWriter) BytesWritten() int {
	return b.bytes
}

func (b *basicWriter) Tee(w io.Writer) {
	b.tee = w
}

func (b *basicWriter) Unwrap() http.ResponseWriter {
	return b.ResponseWriter
}

func (b *basicWriter) Discard() {
	b.discard = true
}

func (b *basicWriter) maybeWriteHeader() {
	if !b.wroteHeader {
		b.WriteHeader(http.StatusOK)
	}
}
