import type { Moment } from 'moment';

import type { OpenTelemetry } from '../features/hasura-metadata-types';
import type { KeyValuePair } from '../components/Common/ConfigureTransformation/stateDefaults';

import type { Driver } from '../dataSources';
import type { Nullable } from './../components/Common/utils/tsUtils';
import type { PermissionsType } from '../components/Services/RemoteSchema/Permissions/types';

export type DataSource = {
  name: string;
  url: string | { from_env: string };
  driver: Driver;
  connection_pool_settings?: ConnectionPoolSettings;
  read_replicas?: Omit<SourceConnectionInfo, 'connection_string'>[];
};

// GENERATED

export type PGColumn = string;
export type ComputedFieldName = string;
export type RoleName = string;
export type TriggerName = string;
export type RemoteRelationshipName = string;
export type RemoteSchemaName = string;
export type CollectionName = string;
export type GraphQLName = string;
export type GraphQLType = string;
export type RelationshipName = string;
export type ActionName = string;

/**
 * A String value which supports templating environment variables enclosed in {{ and }}.
 * Template example: https://{{ACTION_API_DOMAIN}}/create-user
 */
export type WebhookURL = string;

// //////////////////////////////
// #region TABLES
// /////////////////////////////

export type TableName = string | QualifiedTable;

export interface QualifiedTable {
  name: string;
  schema: string;
  dataset?: string;
}

export interface QualifiedTableBigQuery {
  name: string;
  dataset: string;
}

/**
 * Configuration for the table/view
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/table-view.html#table-config
 */
export interface TableConfig {
  /** Customise the table name */
  custom_name: string;
  /** Customise the root fields */
  custom_root_fields?: CustomRootFields;
  /** Customise the column configuration */
  column_config?: ColumnConfig;
}

/**
 * Representation of a table in metadata, 'tables.yaml' and 'metadata.json'
 */
export interface TableEntry {
  table: QualifiedTable;
  is_enum?: boolean;
  apollo_federation_config?: { enable: 'v1' };
  configuration?: TableConfig;
  event_triggers?: EventTrigger[];
  computed_fields?: ComputedField[];
  object_relationships?: ObjectRelationship[];
  array_relationships?: ArrayRelationship[];
  remote_relationships?: RemoteRelationship[];
  insert_permissions?: InsertPermissionEntry[];
  select_permissions?: SelectPermissionEntry[];
  update_permissions?: UpdatePermissionEntry[];
  delete_permissions?: DeletePermissionEntry[];
}

/**
 * Customise the root fields
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/table-view.html#custom-root-fields
 */
export interface CustomRootFields {
  /** Customise the `<table-name>` root field */
  select?: string | CustomRootField | null;
  /** Customise the `<table-name>_by_pk` root field */
  select_by_pk?: string | CustomRootField | null;
  /** Customise the `<table-name>_aggregate` root field */
  select_aggregate?: string | CustomRootField | null;
  /** Customise the `<table-name>_stream` root field */
  select_stream?: string | CustomRootField | null;
  /** Customise the `insert_<table-name>` root field */
  insert?: string | CustomRootField | null;
  /** Customise the `insert_<table-name>_one` root field */
  insert_one?: string | CustomRootField | null;
  /** Customise the `update_<table-name>` root field */
  update?: string | CustomRootField | null;
  /** Customise the `update_<table-name>_by_pk` root field */
  update_by_pk?: string | CustomRootField | null;
  /** Customise the `delete_<table-name>` root field */
  delete?: string | CustomRootField | null;
  /** Customise the `delete_<table-name>_by_pk` root field */
  delete_by_pk?: string | CustomRootField | null;
}

export interface CustomRootField {
  name?: string | null;
  comment?: string | null;
}

export interface ColumnConfig {
  [key: string]: ColumnConfigValue;
}

export interface ColumnConfigValue {
  custom_name?: string | null;
  comment?: string | null;
}

// ////////////////////////////
// #endregion TABLES
// /////////////////////////////

// //////////////////////////////
// #region CUSTOM FUNCTIONS
// /////////////////////////////

export type FunctionName = string | QualifiedFunction;

export interface QualifiedFunction {
  name: string;
  schema: string;
}

/**
 * A custom SQL function to add to the GraphQL schema with configuration.
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-functions.html#args-syntax
 */
export interface Function {
  /** Name of the SQL function */
  function: FunctionName;
  /** Configuration for the SQL function */
  configuration?: FunctionConfiguration;
}

export interface FunctionDefinition {
  name: string;
  schema: string;
}

/**
 * Configuration for a CustomFunction
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-functions.html#function-configuration
 */
export interface FunctionConfiguration {
  /**
   * Function argument which accepts session info JSON
   * Currently, only functions which satisfy the following constraints can be exposed over the GraphQL API (terminology from Postgres docs):
   * - Function behaviour: ONLY `STABLE` or `IMMUTABLE`
   * - Return type: MUST be `SETOF <table-name>`
   * - Argument modes: ONLY `IN`
   */
  session_argument?: string;
}

export interface FunctionPermission {
  role: string;
  definition?: Record<string, any>;
}

// ////////////////////////////
// #endregion CUSTOM FUNCTIONS
// /////////////////////////////

// //////////////////////////////
// #region RELATIONSHIPS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#args-syntax
 */
export interface ObjectRelationship {
  /** Name of the new relationship */
  name: string;
  /** Use one of the available ways to define an object relationship */
  using: ObjRelUsing;
  /** Comment */
  comment?: string;
}

/**
 * Use one of the available ways to define an object relationship
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#objrelusing
 */
export interface ObjRelUsing {
  /** The column with foreign key constraint */
  foreign_key_constraint_on?: PGColumn;
  /** Manual mapping of table and columns */
  manual_configuration?: ObjRelUsingManualMapping;
}

/**
 * Manual mapping of table and columns
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#objrelusingmanualmapping
 */
export interface ObjRelUsingManualMapping {
  /** The table to which the relationship has to be established */
  remote_table: TableName;
  /** Mapping of columns from current table to remote table */
  column_mapping: { [key: string]: string };
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#create-array-relationship-syntax
 */
export interface ArrayRelationship {
  /** Name of the new relationship */
  name: string;
  /** Use one of the available ways to define an array relationship */
  using: ArrRelUsing;
  /** Comment */
  comment?: string;
}

/**
 * Use one of the available ways to define an object relationship
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#arrrelusing
 */
export interface ArrRelUsing {
  /** The column with foreign key constraint */
  foreign_key_constraint_on?: ArrRelUsingFKeyOn;
  /** Manual mapping of table and columns */
  manual_configuration?: ArrRelUsingManualMapping;
}

/**
 * The column with foreign key constraint
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#arrrelusingfkeyon
 */
export interface ArrRelUsingFKeyOn {
  column: PGColumn;
  table: TableName;
}

/**
 * Manual mapping of table and columns
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/relationship.html#arrrelusingmanualmapping
 */
export interface ArrRelUsingManualMapping {
  /** The table to which the relationship has to be established */
  remote_table: TableName;
  /** Mapping of columns from current table to remote table */
  column_mapping: { [key: string]: string };
}

/**
 * Preset values for columns that can be sourced from session variables or static values.
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/syntax-defs.html#columnpresetexp
 */
export interface ColumnPresetsExpression {
  [key: string]: string;
}

// //////////////////////////////
// #endregion RELATIONSHIPS
// /////////////////////////////

// //////////////////////////////
// #region PERMISSIONS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#args-syntax
 */

export interface ValidationInput {
  type: 'http';
  definition: {
    url: string;
    timeout: number;
    headers: { key: string; value: string }[];
    forward_client_headers: boolean;
  };
}
export interface InsertPermissionEntry {
  /** Role */
  role: RoleName;
  /** The permission definition */
  permission: InsertPermission;
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#insertpermission
 */
export interface InsertPermission {
  /** This expression has to hold true for every new row that is inserted */
  check?: { [key: string]: Record<string, any> | string | number };
  /** Preset values for columns that can be sourced from session variables or static values */
  set?: ColumnPresetsExpression;
  /** Can insert into only these columns (or all when '*' is specified) */
  columns: PGColumn[] | '*';
  /**
   * When set to true the mutation is accessible only if x-hasura-use-backend-only-permissions session variable exists
   * and is set to true and request is made with x-hasura-admin-secret set if any auth is configured
   */
  backend_only?: boolean;
  validation_input: ValidationInput;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#create-select-permission-syntax
 */
export interface SelectPermissionEntry {
  /** Role */
  role: RoleName;
  /** The permission definition */
  permission: SelectPermission;
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#selectpermission
 */
export interface SelectPermission {
  /** Only these columns are selectable (or all when '*' is specified) */
  columns: PGColumn[] | '*';
  /** Only these computed fields are selectable */
  computed_fields?: ComputedFieldName[];
  /**
   * The maximum number of rows that can be returned
   * @TJS-type integer
   */
  limit?: number;
  /** Toggle allowing aggregate queries */
  allow_aggregations?: boolean;
  /** Only the rows where this precondition holds true are selectable */
  filter?: { [key: string]: Record<string, any> | string | number };
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#create-update-permission-syntax
 */
export interface UpdatePermissionEntry {
  /** Role */
  role: RoleName;
  /** The permission definition */
  permission: UpdatePermission;
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#updatepermission
 */
export interface UpdatePermission {
  /** Postcondition which must be satisfied by rows which have been updated */
  check?: { [key: string]: Record<string, any> | string | number };
  /** Preset values for columns that can be sourced from session variables or static values */
  set?: ColumnPresetsExpression;
  /** Only these columns are selectable (or all when '*' is specified) */
  columns: PGColumn[] | '*';
  /** Only the rows where this precondition holds true are updatable */
  filter?: { [key: string]: Record<string, any> | string | number };
  validation_input: ValidationInput;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#create-delete-permission-syntax
 */
export interface DeletePermissionEntry {
  /** Role */
  role: RoleName;
  /** The permission definition */
  permission: DeletePermission;
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/permission.html#deletepermission
 */
export interface DeletePermission {
  /** Only the rows where this precondition holds true are updatable */
  filter?: { [key: string]: Record<string, any> | string | number };
  validation_input: ValidationInput;
}

// //////////////////////////////
//  #endregion PERMISSIONS
// /////////////////////////////

// //////////////////////////////
// #region COMPUTED FIELDS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/computed-field.html#args-syntax
 */
export interface ComputedField {
  /** Name of the new computed field */
  name: ComputedFieldName;
  /** The computed field definition */
  definition: ComputedFieldDefinition;
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/computed-field.html#computedfielddefinition
 */
export interface ComputedFieldDefinition {
  /** The SQL function */
  function: FunctionName;
  /** Name of the argument which accepts a table row type. If omitted, the first argument is considered a table argument */
  table_argument?: string;
  /** Name of the argument which accepts the Hasura session object as a JSON/JSONB value. If omitted, the Hasura session object is not passed to the function */
  session_argument?: string;
}

// //////////////////////////////
//  #endregion COMPUTED FIELDS
// /////////////////////////////

// //////////////////////////////
// #region EVENT TRIGGERS
// /////////////////////////////

/**
 * NOTE: The metadata type doesn't QUITE match the 'create' arguments here
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/event-triggers.html#create-event-trigger
 */

export interface EventTriggerAutoCleanup {
  schedule: string;
  batch_size: number;
  clear_older_than: number;
  timeout: number;
  clean_invocation_logs: boolean;
  paused: boolean;
}
export interface EventTrigger {
  /** Name of the event trigger */
  name: TriggerName;
  /** The SQL function */
  definition: EventTriggerDefinition;
  /** The SQL function */
  retry_conf: RetryConf;
  /** The SQL function */
  webhook?: string;
  webhook_from_env?: string;
  /** The SQL function */
  headers?: ServerHeader[];
  /** Request transformation object */
  request_transform?: RequestTransform;
  /** Auto-cleanup configuration object */
  cleanup_config?: EventTriggerAutoCleanup;
}

export interface EventTriggerDefinition {
  enable_manual: boolean;
  insert?: OperationSpec;
  delete?: OperationSpec;
  update?: OperationSpec;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/event-triggers.html#eventtriggercolumns
 */
export type EventTriggerColumns = '*' | PGColumn[];

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/event-triggers.html#operationspec
 */
export interface OperationSpec {
  columns: EventTriggerColumns;
  payload?: EventTriggerColumns;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/syntax-defs.html#headerfromvalue
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/syntax-defs.html#headerfromenv
 */
export interface ServerHeader {
  /** Name of the header */
  name: string;
  /** Value of the header */
  value?: string;
  /** value obtained from env file */
  value_from_env?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/event-triggers.html#retryconf
 */
export interface RetryConf {
  /**
   * Number of times to retry delivery.
   * Default: 0
   * @TJS-type integer
   */
  num_retries?: number;
  /**
   * Number of seconds to wait between each retry.
   * Default: 10
   * @TJS-type integer
   */
  interval_sec?: number;
  /**
   * Number of seconds to wait for response before timing out.
   * Default: 60
   * @TJS-type integer
   */
  timeout_sec?: number;
}

// //////////////////////////////
//  #endregion EVENT TRIGGERS
// /////////////////////////////

// //////////////////////////////
// #region SCHEDULED TRIGGERS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/scheduled-triggers.html#create-cron-trigger
 */
export interface CronTrigger {
  /**	Name of the cron trigger */
  name: TriggerName;
  /**	URL of the webhook */
  webhook: WebhookURL;
  /**	Cron expression at which the trigger should be invoked. */
  schedule: string;
  /** Any JSON payload which will be sent when the webhook is invoked. */
  payload?: Record<string, any>;
  /** List of headers to be sent with the webhook */
  headers: ServerHeader[];
  /**	Retry configuration if scheduled invocation delivery fails */
  retry_conf?: RetryConfST;
  /**	Flag to indicate whether a trigger should be included in the metadata. When a cron trigger is included in the metadata, the user will be able to export it when the metadata of the graphql-engine is exported. */
  include_in_metadata: boolean;
  /**	Custom comment. */
  comment?: string;
  /** Rest connectors. */
  request_transform?: RequestTransform;
}

export interface ScheduledTrigger {
  /**	URL of the webhook */
  webhook: WebhookURL;
  /**	Scheduled time at which the trigger should be invoked. */
  schedule_at: string | Moment;
  /** Any JSON payload which will be sent when the webhook is invoked. */
  payload?: Record<string, any>;
  /** List of headers to be sent with the webhook */
  headers: ServerHeader[];
  /**	Retry configuration if scheduled invocation delivery fails */
  retry_conf?: RetryConfST;
  /**	Custom comment. */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/scheduled-triggers.html#retryconfst
 */
export interface RetryConfST {
  /**
   * Number of times to retry delivery.
   * Default: 0
   * @TJS-type integer
   */
  num_retries?: number;
  /**
   * Number of seconds to wait between each retry.
   * Default: 10
   * @TJS-type integer
   */
  retry_interval_seconds?: number;
  /**
   * Number of seconds to wait for response before timing out.
   * Default: 60
   * @TJS-type integer
   */
  timeout_seconds?: number;
  /**
   * Number of seconds between scheduled time and actual delivery time that is acceptable. If the time difference is more than this, then the event is dropped.
   * Default: 21600 (6 hours)
   * @TJS-type integer
   */
  tolerance_seconds?: number;
}

// //////////////////////////////
//  #endregion SCHEDULED TRIGGERS
// /////////////////////////////

// //////////////////////////////
// #region REMOTE SCHEMAS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/remote-schemas.html#add-remote-schema
 */
export interface RemoteSchema {
  /** Name of the remote schema */
  name: string;
  /** Name of the remote schema */
  definition: RemoteSchemaDef;
  /** Comment */
  comment?: string;
  permissions?: PermissionsType[];
  remote_relationships?: (rsToDbRelDef | rsToRsRelDef)[];
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/syntax-defs.html#remoteschemadef
 */
export interface RemoteSchemaDef {
  url?: string;
  url_from_env?: string;
  headers?: ServerHeader[];
  forward_client_headers?: boolean;
  timeout_seconds?: number;
}

export interface rsToDbRelDef {
  relationships: {
    definition: {
      to_source: {
        table: QualifiedTable;
        source: string;
        relationship_type: 'array' | 'object';
        field_mapping: Record<string, unknown>;
      };
    };
    name: string;
  }[];
  type_name: string;
}

export interface rsToRsRelDef {
  relationships: {
    definition: {
      to_remote_schema: {
        lhs_fields: string[];
        remote_field: Record<string, unknown>;
        remote_schema: string;
      };
    };
    name: string;
  }[];
  type_name: string;
}

// //////////////////////////////
//  #endregion REMOTE SCHEMAS
// /////////////////////////////

// //////////////////////////////
// #region REMOTE RELATIONSHIPS
// /////////////////////////////

// NOTE: RemoteRelationship Metadata shape is slightly different than 'create' arguments here

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/remote-relationships.html#args-syntax
 */
export interface RemoteDBRelationship {
  /** Name of the remote relationship */
  name: string;
  /** Definition object */
  definition: {
    to_source: {
      relationship_type: string;
      source: string;
      table: { schema: string; name: string };
      field_mapping: Record<string, string>;
    };
  };
}

export interface RemoteRelationship {
  /** Name of the remote relationship */
  name: RemoteRelationshipName;
  /** Definition object */
  definition: RemoteRelationshipDef;
  remote_relationships?: RemoteDBRelationship;
}

export interface RemoteRelationshipDef {
  /**
   * Column(s) in the table that is used for joining with remote schema field.
   * All join keys in remote_field must appear here.
   */
  hasura_fields?: PGColumn[];
  /** Name of the remote schema to join with */
  remote_schema?: RemoteSchemaName;
  /** The schema tree ending at the field in remote schema which needs to be joined with. */
  remote_field?: RemoteField;

  to_source: {
    relationship_type: string;
    source: string;
    table: QualifiedTable;
    field_mapping: Record<string, string>;
  };

  to_remote_schema?: ToRemoteSchema;
}

export interface ToRemoteSchema {
  lhs_fields: string[];
  remote_schema: string;
  remote_field: RemoteField;
}

/**
 * A recursive tree structure that points to the field in the remote schema that needs to be joined with.
 * It is recursive because the remote field maybe nested deeply in the remote schema.
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/remote-relationships.html#remotefield
 */
export interface RemoteField {
  [FieldName: string]: {
    arguments: InputArguments;
    field?: RemoteField;
  };
}

/**
 * Note: Table columns can be referred by prefixing $ e.g $id.
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/remote-relationships.html#inputarguments
 */
export interface InputArguments {
  [InputField: string]: InputArguments | string;
}

// //////////////////////////////
// #endregion REMOTE RELATIONSHIPS
// /////////////////////////////

// //////////////////////////////
// #region QUERY COLLECTIONS
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/query-collections.html#args-syntax
 */
export interface QueryCollectionEntry {
  /** Name of the query collection */
  name: CollectionName;
  /** List of queries */
  definition: {
    queries: QueryCollection[];
  };
  /** Comment */
  comment?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/syntax-defs.html#collectionquery
 */
export interface QueryCollection {
  name: string;
  query: string;
}

// //////////////////////////////
// #endregion QUERY COLLECTIONS
// /////////////////////////////

// //////////////////////////////
// #region ALLOW LIST
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/query-collections.html#add-collection-to-allowlist-syntax
 */

type AllowListScope =
  | {
      global: false;
      roles: string[];
    }
  | { global: true };
export interface AllowList {
  /** Name of a query collection to be added to the allow-list */
  collection: CollectionName;
  scope?: AllowListScope;
}

// //////////////////////////////
//  #endregion ALLOW LIST
// /////////////////////////////

// //////////////////////////////
// #region CUSTOM TYPES
// /////////////////////////////

export interface CustomTypes {
  input_objects?: InputObjectType[];
  objects?: ObjectType[];
  scalars?: ScalarType[];
  enums?: EnumType[];
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#inputobjecttype
 */
export interface InputObjectType {
  /** Name of the Input object type */
  name: GraphQLName;
  /** Description of the Input object type */
  description?: string;
  /** Fields of the Input object type */
  fields: InputObjectField[];
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#inputobjectfield
 */
export interface InputObjectField {
  /** Name of the Input object type */
  name: GraphQLName;
  /** Description of the Input object type */
  description?: string;
  /** GraphQL type of the Input object type */
  type: GraphQLType;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#objecttype
 */
export interface ObjectType {
  /** Name of the Input object type */
  name: GraphQLName;
  /** Description of the Input object type */
  description?: string;
  /** Fields of the Input object type */
  fields: InputObjectField[];
  /** Relationships of the Object type to tables */
  relationships?: CustomTypeObjectRelationship[];
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#objectfield
 */
export interface ObjectField {
  /** Name of the Input object type */
  name: GraphQLName;
  /** Description of the Input object type */
  description?: string;
  /** GraphQL type of the Input object type */
  type: GraphQLType;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#objectrelationship
 */
export interface CustomTypeObjectRelationship {
  /** Name of the relationship, shouldn’t conflict with existing field names */
  name: RelationshipName;
  /** Type of the relationship */
  type: 'object' | 'array';
  /** The table to which relationship is defined */
  remote_table: TableName;
  /** Mapping of fields of object type to columns of remote table  */
  field_mapping: {
    [ObjectFieldName: string]: string;
  };
  /** Source name, where remote_table exists */
  source?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#scalartype
 */
export interface ScalarType {
  /** Name of the Scalar type */
  name: GraphQLName;
  /** Description of the Scalar type */
  description?: string;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#enumtype
 */
export interface EnumType {
  /** Name of the Enum type */
  name: GraphQLName;
  /** Description of the Enum type */
  description?: string;
  /** Values of the Enum type */
  values: EnumValue[];
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/custom-types.html#enumvalue
 */
export interface EnumValue {
  /** Value of the Enum type */
  value: GraphQLName;
  /** Description of the Enum value */
  description?: string;
  /** If set to true, the enum value is marked as deprecated */
  is_deprecated?: boolean;
}

// //////////////////////////////
//  #endregion CUSTOM TYPES
// /////////////////////////////

// //////////////////////////////
// #region ACTIONS
// /////////////////////////////

// Note: Action Metadata type is slightly different than "create" type

/**
 *
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/actions.html#args-syntax
 */
export interface Action {
  /** Name of the action  */
  name: ActionName;
  /** Definition of the action */
  definition: ActionDefinition;
  /** Comment */
  comment?: string;
  /** Permissions of the action */
  permissions?: Array<{ role: string }>;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/syntax-defs.html#requesttransformation
 */
export type RequestTransformMethod =
  | 'POST'
  | 'GET'
  | 'PUT'
  | 'DELETE'
  | 'PATCH';

export type RequestTransformContentType =
  | 'application/json'
  | 'application/x-www-form-urlencoded';

export type RequestTransformBodyActions =
  | 'remove'
  | 'transform'
  | 'x_www_form_urlencoded';

export type RequestTransformBody = {
  action: RequestTransformBodyActions;
  template?: string;
  form_template?: Record<string, string> | string;
};

export type ResponseTransformBody = {
  action: RequestTransformBodyActions;
  template?: string;
};

export type RequestTransformHeaders = {
  add_headers?: Record<string, string>;
  remove_headers?: string[];
};

export type RequestTransformTemplateEngine = 'Kriti';

interface RequestTransformFields {
  method?: Nullable<RequestTransformMethod>;
  url?: Nullable<string>;
  content_type?: Nullable<RequestTransformContentType>;
  request_headers?: Nullable<RequestTransformHeaders>;
  query_params?: Nullable<Record<string, string>> | string;
  template_engine?: Nullable<RequestTransformTemplateEngine>;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/syntax-defs/#requesttransformation
 */
interface RequestTransformV1 extends RequestTransformFields {
  version: 1;
  body?: string;
}

interface RequestTransformV2 extends RequestTransformFields {
  version: 2;
  body?: RequestTransformBody;
}

export type RequestTransform = RequestTransformV1 | RequestTransformV2;

export type ResponseTranform = {
  version: 2;
  body?: ResponseTransformBody;
  template_engine?: Nullable<RequestTransformTemplateEngine>;
};

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/actions.html#actiondefinition
 */
export interface ActionDefinition {
  arguments?: InputArgument[];
  output_type?: string;
  kind?: 'synchronous' | 'asynchronous';
  headers?: ServerHeader[];
  forward_client_headers?: boolean;
  handler: WebhookURL;
  type?: 'mutation' | 'query';
  transform?: RequestTransform;
  responseTransform?: ResponseTranform;
}

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/actions.html#inputargument
 */
export interface InputArgument {
  name: string;
  type: GraphQLType;
}

// //////////////////////////////
//  #endregion ACTIONS
// /////////////////////////////

// /////////////////////////////
// #region REST ENDPOINT
// /////////////////////////////

/**
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api//schema-metadata-api/restified-endpoints.html
 */

export type AllowedRESTMethods = 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE';

export interface RestEndpointDefinition {
  query: {
    query_name: string;
    collection_name: string;
  };
}

export interface BigQueryServiceAccount {
  project_id?: string;
  client_email?: string;
  private_key?: string;
  from_env?: string;
}

export interface RestEndpointEntry {
  name: string;
  url: string;
  methods: AllowedRESTMethods[];
  definition: RestEndpointDefinition;
  comment?: string;
}

// //////////////////////////////
//  #endregion REST ENDPOINT
// /////////////////////////////

/**
 * Docs for type: https://hasura.io/docs/latest/graphql/core/api-reference/syntax-defs.html#pgsourceconnectioninfo
 */

export type SSLModeOptions = 'verify-ca' | 'verify-full' | 'disable';

export type IsolationLevelOptions =
  | 'read-committed'
  | 'repeatable-read'
  | 'serializable';

export type NamingConventionOptions = 'hasura-default' | 'graphql-default';

export interface SSLConfigOptions {
  sslmode?: SSLModeOptions;
  sslrootcert?: {
    from_env: string;
  };
  sslcert?: {
    from_env: string;
  };
  sslkey?: {
    from_env: string;
  };
  sslpassword?: {
    from_env: string;
  };
}

export interface ConnectionPoolSettings {
  max_connections?: number;
  total_max_connections?: number;
  idle_timeout?: number;
  retries?: number;
  pool_timeout?: number;
  connection_lifetime?: number;
}

export type ConnectionParams = {
  username: string;
  password?: string;
  database: string;
  host: string;
  port: number;
};

export type GraphQLFieldCustomization = {
  rootFields?: {
    namespace?: string;
    prefix?: string;
    suffix?: string;
  };
  typeNames?: {
    prefix?: string;
    suffix?: string;
  };
  namingConvention?: NamingConventionOptions | null;
};

export interface SourceConnectionInfo {
  // used for SQL Server
  connection_string?: string | { from_env: string };
  // used for Postgres
  database_url?:
    | string
    | { from_env: string }
    | { connection_parameters: ConnectionParams };
  use_prepared_statements?: boolean;
  isolation_level?: IsolationLevelOptions;
  pool_settings?: ConnectionPoolSettings;
  ssl_configuration?: SSLConfigOptions;
}

/**
 * Type used in exported 'metadata.json' and replace metadata endpoint
 * https://hasura.io/docs/latest/graphql/core/api-reference/schema-metadata-api/manage-metadata.html#replace-metadata
 */
export interface HasuraMetadataV2 {
  version: '2';
  tables: TableEntry[];
  actions?: Action[];
  custom_types?: CustomTypes;
  functions?: Array<{
    function: { schema: string; name: string };
    configuration: Record<string, any>;
  }>;
  remote_schemas?: RemoteSchema[];
  query_collections?: QueryCollectionEntry[];
  allowlist?: AllowList[];
  cron_triggers?: CronTrigger[];
}

type GraphQLCustomizationMetadata = {
  root_fields: GraphQLFieldCustomization['rootFields'];
  type_names: GraphQLFieldCustomization['typeNames'];
  naming_convention: GraphQLFieldCustomization['namingConvention'];
};

// Used for Dynamic Connection Routing
export type ConnectionSet = {
  connection_info: SourceConnectionInfo;
  name: string;
};

export type SourceConnectionTemplate = {
  template: string | null;
};
export interface MetadataDataSource {
  name: string;
  kind:
    | 'postgres'
    | 'mssql'
    | 'mysql'
    | 'bigquery'
    | 'citus'
    | 'cockroach'
    | 'alloy';
  configuration?: {
    connection_info?: SourceConnectionInfo;
    extensions_schema?: string;
    // pro-only feature
    read_replicas?: SourceConnectionInfo[];
    connection_template?: SourceConnectionTemplate;
    connection_set?: ConnectionSet[];
    service_account?: BigQueryServiceAccount;
    global_select_limit?: number;
    project_id?: string;
    datasets?: string[];
  };
  tables: TableEntry[];
  functions?: Array<{
    function: QualifiedFunction;
    configuration?: {
      exposed_as?: 'mutation' | 'query';
      session_argument?: string;
    };
    permissions?: FunctionPermission[];
  }>;
  query_collections?: QueryCollectionEntry[];
  allowlist?: AllowList[];
  customization?: GraphQLCustomizationMetadata;
}

export interface InheritedRole {
  role_name: string;
  role_set: string[];
}

export interface DomainList {
  host: string;
  suffix?: string;
  perms?: string[];
}
export interface APILimits {
  per_role?: Record<string, number>;
  global?: number;
}

type APILimit<T> = {
  global: T;
  per_role?: Record<string, T>;
};
export interface HasuraMetadataV3 {
  version: 3;
  sources: MetadataDataSource[];
  remote_schemas?: RemoteSchema[];
  actions?: Action[];
  custom_types?: CustomTypes;
  cron_triggers?: CronTrigger[];
  query_collections?: QueryCollectionEntry[];
  allowlist?: AllowList[];
  inherited_roles: InheritedRole[];
  network?: { tls_allowlist?: DomainList[] };
  rest_endpoints?: RestEndpointEntry[];
  api_limits?: {
    disabled?: boolean;
    depth_limit?: APILimit<number>;
    node_limit?: APILimit<number>;
    batch_limit?: APILimit<number>;
    time_limit?: APILimit<number>;
    rate_limit?: APILimit<{
      unique_params: Nullable<'IP' | string[]>;
      max_reqs_per_min: number;
    }>;
  };
  graphql_schema_introspection?: {
    disabled_for_roles: string[];
  };

  /**
   * The EE Lite OpenTelemetry settings.
   *
   * ATTENTION: Both Lux and the EE Lite server allow configuring OpenTelemetry. Anyway, this only
   * represents the EE Lite one since Lux stores the OpenTelemetry settings by itself.
   */
  opentelemetry?: OpenTelemetry;
}

// Inconsistent Objects

export interface InconsistentObject {
  definition:
    | string
    | {
        comment: string;
        definition: InconsistentObjectDefinition;
      };
  reason: string;
  name: string;
  type: string;
  message:
    | string
    | {
        message: string;
        request: InconsistentObjectRequest;
      };
}

type InconsistentObjectRequest = {
  proxy: string | null;
  secure: boolean;
  path: string;
  responseTimeout: string;
  method: 'POST' | 'GET' | 'PUT' | 'DELETE' | 'PATCH' | 'OPTION';
  host: string;
  requestVersion: `${number}`;
  redirectCount: `${number}`;
  port: `${number}`;
};

type InconsistentObjectDefinition = {
  timeout_seconds: number;
  url_from_env: string;
  forward_client_headers: boolean;
};

export type QueryParams = KeyValuePair[] | string;
