# Copyright 2014-2019 the openage authors. See copying.md for legal info.

# set CODEGEN_SCU_FILE to the absolute path to SCU file
macro(get_codegen_scu_file)
	set(CODEGEN_SCU_FILE "${CMAKE_BINARY_DIR}/libopenage/codegen_scu.gen.cpp")
endmacro()

function(codegen_run)
	# this function must be called once all required assets have been created, but before the executable is finalized.

	# make sure this function gets called only once
	get_property(codegen_run GLOBAL PROPERTY SFT_CODEGEN_HAS_BEEN_RUN)
	if(codegen_run)
		message(FATAL_ERROR "codegen has already been run")
	endif()
	set_property(GLOBAL PROPERTY SFT_CODEGEN_HAS_BEEN_RUN 1)

	set(CODEGEN_INVOCATION
		"${PYTHON}" -m openage codegen
		"--input-dir=${CMAKE_SOURCE_DIR}"
		"--output-dir=${CMAKE_BINARY_DIR}"
		"--generated-list-file=${CMAKE_BINARY_DIR}/codegen_generated_files"
		"--depend-list-file=${CMAKE_BINARY_DIR}/codegen_depends")

	execute_process(COMMAND
		${CODEGEN_INVOCATION} --mode=dryrun
		WORKING_DIRECTORY "${CMAKE_BINARY_DIR}"
		RESULT_VARIABLE COMMAND_RESULT
	)

	if(NOT ${COMMAND_RESULT} EQUAL 0)
		message(FATAL_ERROR "failed to get target list from codegen invocation")
	endif()

	FILE(READ "${CMAKE_BINARY_DIR}/codegen_generated_files" CODEGEN_TARGET_FILES)
	FILE(READ "${CMAKE_BINARY_DIR}/codegen_depends" CODEGEN_DEPENDS)
	STRING(REGEX REPLACE "\n" ";" CODEGEN_TARGET_FILES ${CODEGEN_TARGET_FILES})
	STRING(REGEX REPLACE "\n" ";" CODEGEN_DEPENDS ${CODEGEN_DEPENDS})

	# as the codegen creates all files at once,
	# let the buildsystem only depend on this single dummy file.
	# otherwise the codegen invocation would be done for each generated source.
	set(CODEGEN_TIMEFILE "${CMAKE_BINARY_DIR}/codegen_timefile")

	add_custom_command(
		OUTPUT "${CODEGEN_TIMEFILE}"
		BYPRODUCTS ${CODEGEN_TARGET_FILES}
		COMMAND ${CODEGEN_INVOCATION} --mode=codegen "--touch-file-on-cache-change=${CMAKE_CURRENT_LIST_FILE}" --force-rerun-on-generated-list-change
		COMMAND "${CMAKE_COMMAND}" -E touch "${CODEGEN_TIMEFILE}"
		WORKING_DIRECTORY "${CMAKE_BINARY_DIR}"
		DEPENDS ${CODEGEN_DEPENDS}
		COMMENT "openage.codegen: generating c++ code"
	)

	add_custom_target(codegen
		DEPENDS "${CODEGEN_TIMEFILE}"
	)

	add_custom_target(cleancodegen
		COMMAND ${CODEGEN_INVOCATION} --mode=clean
		COMMAND "${CMAKE_COMMAND}" -E remove "${CODEGEN_TIMEFILE}"
		WORKING_DIRECTORY "${CMAKE_BINARY_DIR}"
	)

	get_codegen_scu_file()
	get_filename_component(CODEGEN_SCU_DIR ${CODEGEN_SCU_FILE} DIRECTORY)

	set(CODEGEN_SCU_CONTENT "// Generated by codegen.\n\n")
	list(SORT CODEGEN_TARGET_FILES)
	foreach(target ${CODEGEN_TARGET_FILES})
		if("${target}" MATCHES "\\.cpp$")
			file(RELATIVE_PATH target_rel ${CODEGEN_SCU_DIR} ${target})
			string(APPEND CODEGEN_SCU_CONTENT "#include \"${target_rel}\"\n")
		endif()
	endforeach()

	write_on_change(${CODEGEN_SCU_FILE} "${CODEGEN_SCU_CONTENT}")
endfunction()
