/*
 * Soft:        Keepalived is a failover program for the LVS project
 *              <www.linuxvirtualserver.org>. It monitor & manipulate
 *              a loadbalanced server pool using multi-layer checks.
 *
 * Part:        vrrp_track.c include file.
 *
 * Author:      Alexandre Cassen, <acassen@linux-vs.org>
 *
 *              This program is distributed in the hope that it will be useful,
 *              but WITHOUT ANY WARRANTY; without even the implied warranty of
 *              MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *              See the GNU General Public License for more details.
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *
 * Copyright (C) 2001-2017 Alexandre Cassen, <acassen@gmail.com>
 */

#ifndef _VRRP_TRACK_H
#define _VRRP_TRACK_H

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <string.h>
#include <syslog.h>
#include <stdbool.h>
#include <sys/types.h>

/* local includes */
#include "vector.h"
#include "list.h"
#include "notify.h"

/* Macro definition */
#define TRACK_ISUP(L)	(vrrp_tracked_up((L)))
#define SCRIPT_ISUP(L)	(vrrp_script_up((L)))

/* VRRP script tracking defaults */
#define VRRP_SCRIPT_DI 1	/* external script track interval (in sec) */
#define VRRP_SCRIPT_DT 0	/* external script track timeout (in sec) */
#define VRRP_SCRIPT_DW 0	/* external script default weight */

/* VRRP script tracking results.
 * The result is an integer between 0 and rise-1 to indicate a DOWN state,
 * or between rise-1 and rise+fall-1 to indicate an UP state. Upon failure,
 * we decrease result and set it to zero when we pass below rise. Upon
 * success, we increase result and set it to rise+fall-1 when we pass above
 * rise-1.
 */

/* external script we call to track local processes */
typedef struct _vrrp_script {
	char			*sname;		/* instance name */
	char			*script;	/* the command to be called */
	unsigned long		interval;	/* interval between script calls */
	unsigned long		timeout;	/* microseconds before script timeout */
	int			weight;		/* weight associated to this script */
	int			result;		/* result of last call to this script: 0..R-1 = KO, R..R+F-1 = OK */
	int			inuse;		/* how many users have weight>0 ? */
	int			rise;		/* R: how many successes before OK */
	int			fall;		/* F: how many failures before KO */
	script_state_t		state;		/* current state of script */
	script_init_state_t	init_state;	/* current initialisation state of script */
	uid_t			uid;		/* uid to run script as */
	gid_t			gid;		/* gid to run script as */
	bool			insecure;	/* Set if script is run by root, but is non-root modifiable */
} vrrp_script_t;

/* Tracked script structure definition */
typedef struct _tracked_sc {
	int			weight;		/* tracking weight when non-zero */
	vrrp_script_t		*scr;		/* script pointer, cannot be NULL */
} tracked_sc_t;

/* prototypes */
extern void dump_track(void *);
extern void alloc_track(list, vector_t *);
extern void dump_track_script(void *);
extern void alloc_track_script(list, vector_t *, const char *);
extern int vrrp_tracked_up(list);
extern void vrrp_log_tracked_down(list);
extern int vrrp_tracked_weight(list);
extern int vrrp_script_up(list);
extern int vrrp_script_weight(list);
extern vrrp_script_t *find_script_by_name(char *);

#endif
