package dynamodb_test

import (
	"context"
	"fmt"
	"strconv"
	"testing"
	"time"

	"github.com/RichardKnop/machinery/v2/backends/dynamodb"
	"github.com/RichardKnop/machinery/v2/log"
	"github.com/RichardKnop/machinery/v2/tasks"
	awsdynamodb "github.com/aws/aws-sdk-go-v2/service/dynamodb"
	"github.com/aws/aws-sdk-go-v2/service/dynamodb/types"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestNew(t *testing.T) {
	// should call t.Skip if not connected to internet
	backend, err := dynamodb.New(dynamodb.TestCnf)
	require.NoError(t, err)
	assert.IsType(t, new(dynamodb.Backend), backend)
}

func TestInitGroup(t *testing.T) {
	groupUUID := "testGroupUUID"
	taskUUIDs := []string{"testTaskUUID1", "testTaskUUID2", "testTaskUUID3"}
	log.INFO.Println(dynamodb.TestDynamoDBBackend.GetConfig())

	err := dynamodb.TestDynamoDBBackend.InitGroup(groupUUID, taskUUIDs)
	assert.Nil(t, err)

	err = dynamodb.TestErrDynamoDBBackend.InitGroup(groupUUID, taskUUIDs)
	assert.NotNil(t, err)

	// assert proper TTL value is set in InitGroup()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 3 * 3600 // results should expire after 3 hours
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	// Override DynamoDB PutItem() behavior
	var isPutItemCalled bool
	client.PutItemOverride = func(ctx context.Context, input *awsdynamodb.PutItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.PutItemOutput, error) {
		isPutItemCalled = true
		assert.NotNil(t, input)

		actualTTLStr := input.Item["TTL"].(*types.AttributeValueMemberN).Value
		expectedTTLTime := time.Now().Add(3 * time.Hour)
		assertTTLValue(t, expectedTTLTime, actualTTLStr)

		return &awsdynamodb.PutItemOutput{}, nil
	}
	err = dynamodb.TestDynamoDBBackend.InitGroup(groupUUID, taskUUIDs)
	assert.Nil(t, err)
	assert.True(t, isPutItemCalled)
	client.ResetOverrides()
}

func assertTTLValue(t *testing.T, expectedTTLTime time.Time, actualEncodedTTLValue string) {
	actualTTLTimestamp, err := strconv.ParseInt(actualEncodedTTLValue, 10, 64)
	assert.Nil(t, err)
	actualTTLTime := time.Unix(actualTTLTimestamp, 0)
	assert.WithinDuration(t, expectedTTLTime, actualTTLTime, time.Second)
}

func TestGroupCompleted(t *testing.T) {
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	tableName := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	// Override DynamoDB BatchGetItem() behavior
	var isBatchGetItemCalled bool
	client.BatchGetItemOverride = func(ctx context.Context, input *awsdynamodb.BatchGetItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		isBatchGetItemCalled = true
		assert.NotNil(t, input)
		assert.Nil(t, validateBatchGetItemInput(input))

		return &awsdynamodb.BatchGetItemOutput{
			Responses: map[string][]map[string]types.AttributeValue{
				tableName: {
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StateFailure}},
				},
			},
		}, nil
	}
	groupUUID := "testGroupUUID"
	isCompleted, err := dynamodb.TestDynamoDBBackend.GroupCompleted(groupUUID, 3)
	assert.Nil(t, err)
	assert.True(t, isCompleted)
	assert.True(t, isBatchGetItemCalled)
	client.ResetOverrides()
}
func TestGroupCompletedReturnsError(t *testing.T) {
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	client.BatchGetItemOverride = func(ctx context.Context, input *awsdynamodb.BatchGetItemInput, ops ...func(options *awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		return nil, fmt.Errorf("Simulating error from AWS")
	}
	isCompleted, err := dynamodb.TestDynamoDBBackend.GroupCompleted("test", 3)
	assert.NotNil(t, err)
	assert.False(t, isCompleted)
	client.ResetOverrides()
}

// TestGroupCompletedReturnsFalse tests that the GroupCompleted() returns false when some tasks have not yet finished.
func TestGroupCompletedReturnsFalse(t *testing.T) {
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	tableName := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	// Override DynamoDB BatchGetItem() behavior
	client.BatchGetItemOverride = func(ctx context.Context, _ *awsdynamodb.BatchGetItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		return &awsdynamodb.BatchGetItemOutput{
			Responses: map[string][]map[string]types.AttributeValue{
				tableName: {
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StateFailure}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StatePending}},
				},
			},
		}, nil
	}
	isCompleted, err := dynamodb.TestDynamoDBBackend.GroupCompleted("testGroup", 3)
	assert.Nil(t, err)
	assert.False(t, isCompleted)
	client.ResetOverrides()
}

// TestGroupCompletedReturnsFalse tests that the GroupCompleted() retries the the request until MaxFetchAttempts before returning an error
func TestGroupCompletedRetries(t *testing.T) {
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	tableName := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	// Override DynamoDB BatchGetItem() behavior
	var countBatchGetItemAPICalls int
	client.BatchGetItemOverride = func(ctx context.Context, _ *awsdynamodb.BatchGetItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		countBatchGetItemAPICalls++

		return &awsdynamodb.BatchGetItemOutput{
			Responses: map[string][]map[string]types.AttributeValue{
				tableName: {
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
				},
			},
			UnprocessedKeys: map[string]types.KeysAndAttributes{
				tableName: {
					Keys: []map[string]types.AttributeValue{
						{"TaskUUID": &types.AttributeValueMemberS{Value: "unfetchedTaskUUID1"}},
						{"TaskUUID": &types.AttributeValueMemberS{Value: "unfetchedTaskUUID2"}},
					},
				},
			},
		}, nil
	}
	_, err := dynamodb.TestDynamoDBBackend.GroupCompleted("testGroup", 3)
	assert.NotNil(t, err)
	assert.Equal(t, dynamodb.MaxFetchAttempts, countBatchGetItemAPICalls)
	client.ResetOverrides()
}

// TestGroupCompletedReturnsFalse tests that the GroupCompleted() retries the the request and returns success if all keys are fetched on retries.
func TestGroupCompletedRetrieSuccess(t *testing.T) {
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	tableName := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	// Override DynamoDB BatchGetItem() behavior
	var countBatchGetItemAPICalls int
	client.BatchGetItemOverride = func(ctx context.Context, _ *awsdynamodb.BatchGetItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		countBatchGetItemAPICalls++

		// simulate unfetched keys on 1st attempt.
		if countBatchGetItemAPICalls == 1 {
			return &awsdynamodb.BatchGetItemOutput{
				Responses: map[string][]map[string]types.AttributeValue{
					tableName: {}, // no keys returned in this attempt.
				},
				UnprocessedKeys: map[string]types.KeysAndAttributes{
					tableName: {
						Keys: []map[string]types.AttributeValue{
							{"TaskUUID": &types.AttributeValueMemberS{Value: "unfetchedTaskUUID1"}},
							{"TaskUUID": &types.AttributeValueMemberS{Value: "unfetchedTaskUUID2"}},
							{"TaskUUID": &types.AttributeValueMemberS{Value: "unfetchedTaskUUID3"}},
						},
					},
				},
			}, nil

		}

		// Return all keys in subsequent attempts.
		return &awsdynamodb.BatchGetItemOutput{
			Responses: map[string][]map[string]types.AttributeValue{
				tableName: {
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
					{"State": &types.AttributeValueMemberS{Value: tasks.StateSuccess}},
				},
			},
		}, nil
	}
	isCompleted, err := dynamodb.TestDynamoDBBackend.GroupCompleted("testGroup", 3)
	assert.Nil(t, err)
	assert.True(t, isCompleted)
	assert.Equal(t, 2, countBatchGetItemAPICalls)
	client.ResetOverrides()
}

func TestPrivateFuncGetGroupMeta(t *testing.T) {
	groupUUID := "testGroupUUID"
	meta, err := dynamodb.TestDynamoDBBackend.GetGroupMetaForTest(groupUUID)
	item := tasks.GroupMeta{
		GroupUUID:      "testGroupUUID",
		Lock:           false,
		ChordTriggered: false,
		TaskUUIDs: []string{
			"testTaskUUID1",
			"testTaskUUID2",
			"testTaskUUID3",
		},
	}
	assert.Nil(t, err)
	assert.EqualValues(t, item, *meta)
	_, err = dynamodb.TestErrDynamoDBBackend.GetGroupMetaForTest(groupUUID)
	assert.NotNil(t, err)
}

func TestPrivateFuncUnmarshalTaskStateGetItemResult(t *testing.T) {
	result := awsdynamodb.GetItemOutput{
		Item: map[string]types.AttributeValue{
			"Error": &types.AttributeValueMemberNULL{
				Value: true,
			},
			"State": &types.AttributeValueMemberS{
				Value: tasks.StatePending,
			},
			"TaskUUID": &types.AttributeValueMemberS{
				Value: "testTaskUUID1",
			},
			"Results:": &types.AttributeValueMemberNULL{
				Value: true,
			},
		},
	}

	invalidResult := awsdynamodb.GetItemOutput{
		Item: map[string]types.AttributeValue{
			"Error": &types.AttributeValueMemberBOOL{
				Value: true,
			},
			"State": &types.AttributeValueMemberS{
				Value: tasks.StatePending,
			},
			"TaskUUID": &types.AttributeValueMemberS{
				Value: "testTaskUUID1",
			},
			"Results:": &types.AttributeValueMemberBOOL{
				Value: true,
			},
		},
	}

	item := tasks.TaskState{
		TaskUUID: "testTaskUUID1",
		Results:  nil,
		State:    tasks.StatePending,
		Error:    "",
	}
	state, err := dynamodb.TestErrDynamoDBBackend.UnmarshalTaskStateGetItemResultForTest(&result)
	assert.Nil(t, err)
	assert.EqualValues(t, item, *state)

	_, err = dynamodb.TestDynamoDBBackend.UnmarshalTaskStateGetItemResultForTest(nil)
	assert.NotNil(t, err)

	_, err = dynamodb.TestDynamoDBBackend.UnmarshalTaskStateGetItemResultForTest(&invalidResult)
	assert.NotNil(t, err)

}

func TestPrivateFuncUnmarshalGroupMetaGetItemResult(t *testing.T) {
	result := awsdynamodb.GetItemOutput{
		Item: map[string]types.AttributeValue{
			"TaskUUIDs": &types.AttributeValueMemberL{
				Value: []types.AttributeValue{
					&types.AttributeValueMemberS{
						Value: "testTaskUUID1",
					},
					&types.AttributeValueMemberS{
						Value: "testTaskUUID2",
					},
					&types.AttributeValueMemberS{
						Value: "testTaskUUID3",
					},
				},
			},
			"ChordTriggered": &types.AttributeValueMemberBOOL{
				Value: false,
			},
			"GroupUUID": &types.AttributeValueMemberS{
				Value: "testGroupUUID",
			},
			"Lock": &types.AttributeValueMemberBOOL{
				Value: false,
			},
		},
	}

	invalidResult := awsdynamodb.GetItemOutput{
		Item: map[string]types.AttributeValue{
			"TaskUUIDs": &types.AttributeValueMemberL{
				Value: []types.AttributeValue{
					&types.AttributeValueMemberS{
						Value: "testTaskUUID1",
					},
					&types.AttributeValueMemberS{
						Value: "testTaskUUID2",
					},
					&types.AttributeValueMemberS{
						Value: "testTaskUUID3",
					},
				},
			},
			"ChordTriggered": &types.AttributeValueMemberS{
				Value: "false", // this attribute is invalid
			},
			"GroupUUID": &types.AttributeValueMemberS{
				Value: "testGroupUUID",
			},
			"Lock": &types.AttributeValueMemberBOOL{
				Value: false,
			},
		},
	}

	item := tasks.GroupMeta{
		GroupUUID:      "testGroupUUID",
		Lock:           false,
		ChordTriggered: false,
		TaskUUIDs: []string{
			"testTaskUUID1",
			"testTaskUUID2",
			"testTaskUUID3",
		},
	}
	meta, err := dynamodb.TestErrDynamoDBBackend.UnmarshalGroupMetaGetItemResultForTest(&result)
	assert.Nil(t, err)
	assert.EqualValues(t, item, *meta)
	_, err = dynamodb.TestErrDynamoDBBackend.UnmarshalGroupMetaGetItemResultForTest(nil)
	assert.NotNil(t, err)

	_, err = dynamodb.TestErrDynamoDBBackend.UnmarshalGroupMetaGetItemResultForTest(&invalidResult)
	assert.NotNil(t, err)

}

func TestPrivateFuncSetTaskState(t *testing.T) {
	signature := &tasks.Signature{
		Name: "Test",
		Args: []tasks.Arg{
			{
				Type:  "int64",
				Value: 1,
			},
		},
	}
	state := tasks.NewPendingTaskState(signature)
	err := dynamodb.TestErrDynamoDBBackend.SetTaskStateForTest(state)
	assert.NotNil(t, err)
	err = dynamodb.TestDynamoDBBackend.SetTaskStateForTest(state)
	assert.Nil(t, err)
}

// verifyUpdateInput is a helper function to verify valid dynamoDB update input.
func verifyUpdateInput(t *testing.T, input *awsdynamodb.UpdateItemInput, expectedTaskID string, expectedState string, expectedTTLTime time.Time) {
	assert.NotNil(t, input)

	// verify task ID
	assert.Equal(t, expectedTaskID, input.Key["TaskUUID"].(*types.AttributeValueMemberS).Value)

	// verify task state
	assert.Equal(t, expectedState, input.ExpressionAttributeValues[":s"].(*types.AttributeValueMemberS).Value)

	// Verify TTL
	if !expectedTTLTime.IsZero() {
		actualTTLStr := input.ExpressionAttributeValues[":t"].(*types.AttributeValueMemberN).Value
		assertTTLValue(t, expectedTTLTime, actualTTLStr)
	}
}

func TestSetStateSuccess(t *testing.T) {
	signature := &tasks.Signature{UUID: "testTaskUUID"}

	// assert correct task ID, state and TTL value is set in SetStateSuccess()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 3 * 3600 // results should expire after 3 hours
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	// Override DynamoDB UpdateItem() behavior
	var isUpdateItemCalled bool
	client.UpdateItemOverride = func(ctx context.Context, input *awsdynamodb.UpdateItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.UpdateItemOutput, error) {
		isUpdateItemCalled = true
		verifyUpdateInput(t, input, signature.UUID, tasks.StateSuccess, time.Now().Add(3*time.Hour))
		return &awsdynamodb.UpdateItemOutput{}, nil
	}

	err := dynamodb.TestDynamoDBBackend.SetStateSuccess(signature, nil)
	assert.Nil(t, err)
	assert.True(t, isUpdateItemCalled)
	client.ResetOverrides()
}

func TestSetStateFailure(t *testing.T) {
	signature := &tasks.Signature{UUID: "testTaskUUID"}

	// assert correct task ID, state and TTL value is set in SetStateFailure()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 2 * 3600 // results should expire after 2 hours
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	// Override DynamoDB UpdateItem() behavior
	var isUpdateItemCalled bool
	client.UpdateItemOverride = func(ctx context.Context, input *awsdynamodb.UpdateItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.UpdateItemOutput, error) {
		isUpdateItemCalled = true
		verifyUpdateInput(t, input, signature.UUID, tasks.StateFailure, time.Now().Add(2*time.Hour))
		return &awsdynamodb.UpdateItemOutput{}, nil
	}

	err := dynamodb.TestDynamoDBBackend.SetStateFailure(signature, "Some error occurred")
	assert.Nil(t, err)
	assert.True(t, isUpdateItemCalled)
	client.ResetOverrides()
}

func TestSetStateReceived(t *testing.T) {
	signature := &tasks.Signature{UUID: "testTaskUUID"}

	// assert correct task ID, state and *no* TTL value is set in SetStateReceived()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 2 * 3600 // results should expire after 2 hours (ignored for this state)
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	var isUpdateItemCalled bool
	client.UpdateItemOverride = func(ctx context.Context, input *awsdynamodb.UpdateItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.UpdateItemOutput, error) {
		isUpdateItemCalled = true
		verifyUpdateInput(t, input, signature.UUID, tasks.StateReceived, time.Time{})
		return &awsdynamodb.UpdateItemOutput{}, nil
	}

	err := dynamodb.TestDynamoDBBackend.SetStateReceived(signature)
	assert.Nil(t, err)
	assert.True(t, isUpdateItemCalled)
	client.ResetOverrides()
}

func TestSetStateStarted(t *testing.T) {
	signature := &tasks.Signature{UUID: "testTaskUUID"}

	// assert correct task ID, state and *no* TTL value is set in SetStateStarted()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 2 * 3600 // results should expire after 2 hours (ignored for this state)
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	var isUpdateItemCalled bool
	client.UpdateItemOverride = func(ctx context.Context, input *awsdynamodb.UpdateItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.UpdateItemOutput, error) {
		isUpdateItemCalled = true
		verifyUpdateInput(t, input, signature.UUID, tasks.StateStarted, time.Time{})
		return &awsdynamodb.UpdateItemOutput{}, nil
	}

	err := dynamodb.TestDynamoDBBackend.SetStateStarted(signature)
	assert.Nil(t, err)
	assert.True(t, isUpdateItemCalled)
	client.ResetOverrides()
}

func TestSetStateRetry(t *testing.T) {
	signature := &tasks.Signature{UUID: "testTaskUUID"}

	// assert correct task ID, state and *no* TTL value is set in SetStateStarted()
	dynamodb.TestDynamoDBBackend.GetConfig().ResultsExpireIn = 2 * 3600 // results should expire after 2 hours (ignored for this state)
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	var isUpdateItemCalled bool
	client.UpdateItemOverride = func(ctx context.Context, input *awsdynamodb.UpdateItemInput, ops ...func(options *awsdynamodb.Options)) (*awsdynamodb.UpdateItemOutput, error) {
		isUpdateItemCalled = true
		verifyUpdateInput(t, input, signature.UUID, tasks.StateRetry, time.Time{})
		return &awsdynamodb.UpdateItemOutput{}, nil
	}

	err := dynamodb.TestDynamoDBBackend.SetStateRetry(signature)
	assert.Nil(t, err)
	assert.True(t, isUpdateItemCalled)
	client.ResetOverrides()
}

func TestGroupTaskStates(t *testing.T) {
	expectedStates := map[string]*tasks.TaskState{
		"testTaskUUID1": {
			TaskUUID: "testTaskUUID1",
			Results:  nil,
			State:    tasks.StatePending,
			Error:    "",
		},
		"testTaskUUID2": {
			TaskUUID: "testTaskUUID2",
			Results:  nil,
			State:    tasks.StateStarted,
			Error:    "",
		},
		"testTaskUUID3": {
			TaskUUID: "testTaskUUID3",
			Results:  nil,
			State:    tasks.StateSuccess,
			Error:    "",
		},
	}
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	tableName := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	client.BatchGetItemOverride = func(ctx context.Context, input *awsdynamodb.BatchGetItemInput, ops ...func(options *awsdynamodb.Options)) (*awsdynamodb.BatchGetItemOutput, error) {
		assert.Nil(t, validateBatchGetItemInput(input))

		return &awsdynamodb.BatchGetItemOutput{
			Responses: map[string][]map[string]types.AttributeValue{
				tableName: {
					{
						"TaskUUID": &types.AttributeValueMemberS{Value: "testTaskUUID1"},
						"Results:": &types.AttributeValueMemberNULL{Value: true},
						"State":    &types.AttributeValueMemberS{Value: tasks.StatePending},
						"Error":    &types.AttributeValueMemberNULL{Value: true},
					},
					{
						"TaskUUID": &types.AttributeValueMemberS{Value: "testTaskUUID2"},
						"Results:": &types.AttributeValueMemberNULL{Value: true},
						"State":    &types.AttributeValueMemberS{Value: tasks.StateStarted},
						"Error":    &types.AttributeValueMemberNULL{Value: true},
					},
					{
						"TaskUUID": &types.AttributeValueMemberS{Value: "testTaskUUID3"},
						"Results:": &types.AttributeValueMemberNULL{Value: true},
						"State":    &types.AttributeValueMemberS{Value: tasks.StateSuccess},
						"Error":    &types.AttributeValueMemberNULL{Value: true},
					},
				},
			},
		}, nil
	}
	defer client.ResetOverrides()

	states, err := dynamodb.TestDynamoDBBackend.GroupTaskStates("testGroupUUID", 3)
	assert.Nil(t, err)
	for _, s := range states {
		assert.EqualValues(t, *s, *expectedStates[s.TaskUUID])
	}
}

func TestTriggerChord(t *testing.T) {
	groupUUID := "testGroupUUID"
	triggered, err := dynamodb.TestDynamoDBBackend.TriggerChord(groupUUID)
	assert.Nil(t, err)
	assert.True(t, triggered)
}

func TestGetState(t *testing.T) {
	taskUUID := "testTaskUUID1"
	expectedState := &tasks.TaskState{
		TaskUUID: "testTaskUUID1",
		Results:  nil,
		State:    tasks.StatePending,
		Error:    "",
	}
	client := dynamodb.TestDynamoDBBackend.GetClient().(*dynamodb.TestDynamoDBClient)
	client.GetItemOverride = func(ctx context.Context, input *awsdynamodb.GetItemInput, ops ...func(*awsdynamodb.Options)) (*awsdynamodb.GetItemOutput, error) {
		return &awsdynamodb.GetItemOutput{
			Item: map[string]types.AttributeValue{
				"TaskUUID": &types.AttributeValueMemberS{Value: "testTaskUUID1"},
				"Results:": &types.AttributeValueMemberNULL{Value: true},
				"State":    &types.AttributeValueMemberS{Value: tasks.StatePending},
				"Error":    &types.AttributeValueMemberNULL{Value: false},
			},
		}, nil
	}
	defer client.ResetOverrides()

	state, err := dynamodb.TestDynamoDBBackend.GetState(taskUUID)
	assert.Nil(t, err)
	assert.EqualValues(t, expectedState, state)
}

func TestPurgeState(t *testing.T) {
	taskUUID := "testTaskUUID1"
	err := dynamodb.TestDynamoDBBackend.PurgeState(taskUUID)
	assert.Nil(t, err)

	err = dynamodb.TestErrDynamoDBBackend.PurgeState(taskUUID)
	assert.NotNil(t, err)
}

func TestPurgeGroupMeta(t *testing.T) {
	groupUUID := "GroupUUID"
	err := dynamodb.TestDynamoDBBackend.PurgeGroupMeta(groupUUID)
	assert.Nil(t, err)

	err = dynamodb.TestErrDynamoDBBackend.PurgeGroupMeta(groupUUID)
	assert.NotNil(t, err)
}

func TestPrivateFuncLockGroupMeta(t *testing.T) {
	groupUUID := "GroupUUID"
	err := dynamodb.TestDynamoDBBackend.LockGroupMetaForTest(groupUUID)
	assert.Nil(t, err)
	err = dynamodb.TestErrDynamoDBBackend.LockGroupMetaForTest(groupUUID)
	assert.NotNil(t, err)
}

func TestPrivateFuncUnLockGroupMeta(t *testing.T) {
	groupUUID := "GroupUUID"
	err := dynamodb.TestDynamoDBBackend.UnlockGroupMetaForTest(groupUUID)
	assert.Nil(t, err)
	err = dynamodb.TestErrDynamoDBBackend.UnlockGroupMetaForTest(groupUUID)
	assert.NotNil(t, err)
}

func TestPrivateFuncChordTriggered(t *testing.T) {
	groupUUID := "GroupUUID"
	err := dynamodb.TestDynamoDBBackend.ChordTriggeredForTest(groupUUID)
	assert.Nil(t, err)
	err = dynamodb.TestErrDynamoDBBackend.ChordTriggeredForTest(groupUUID)
	assert.NotNil(t, err)
}

func TestDynamoDBPrivateFuncUpdateGroupMetaLock(t *testing.T) {
	groupUUID := "GroupUUID"
	err := dynamodb.TestDynamoDBBackend.UpdateGroupMetaLockForTest(groupUUID, true)
	assert.Nil(t, err)
	err = dynamodb.TestErrDynamoDBBackend.UpdateGroupMetaLockForTest(groupUUID, true)
	assert.NotNil(t, err)
}

func TestPrivateFuncUpdateToFailureStateWithError(t *testing.T) {
	signature := &tasks.Signature{
		Name: "Test",
		Args: []tasks.Arg{
			{
				Type:  "int64",
				Value: 1,
			},
		},
	}

	state := tasks.NewFailureTaskState(signature, "This is an error")
	err := dynamodb.TestDynamoDBBackend.UpdateToFailureStateWithErrorForTest(state)
	assert.Nil(t, err)
}

func TestPrivateFuncTableExistsForTest(t *testing.T) {
	tables := []string{"foo"}
	assert.False(t, dynamodb.TestDynamoDBBackend.TableExistsForTest("bar", tables))
	assert.True(t, dynamodb.TestDynamoDBBackend.TableExistsForTest("foo", tables))
}

func TestPrivateFuncCheckRequiredTablesIfExistForTest(t *testing.T) {
	err := dynamodb.TestDynamoDBBackend.CheckRequiredTablesIfExistForTest()
	assert.Nil(t, err)
	taskTable := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable
	groupTable := dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.GroupMetasTable
	err = dynamodb.TestErrDynamoDBBackend.CheckRequiredTablesIfExistForTest()
	assert.NotNil(t, err)
	dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable = "foo"
	err = dynamodb.TestDynamoDBBackend.CheckRequiredTablesIfExistForTest()
	assert.NotNil(t, err)
	dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.TaskStatesTable = taskTable
	dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.GroupMetasTable = "foo"
	err = dynamodb.TestDynamoDBBackend.CheckRequiredTablesIfExistForTest()
	assert.NotNil(t, err)
	dynamodb.TestDynamoDBBackend.GetConfig().DynamoDB.GroupMetasTable = groupTable
}

func validateBatchGetItemInput(input *awsdynamodb.BatchGetItemInput) error {
	if input == nil {
		return fmt.Errorf("input is nil")
	}
	if len(input.RequestItems) == 0 {
		return fmt.Errorf("RequestItems cannot be empty")
	}
	for tableName, keysAndAttributes := range input.RequestItems {
		if tableName == "" {
			return fmt.Errorf("table name cannot be empty")
		}
		if len(keysAndAttributes.Keys) == 0 {
			return fmt.Errorf("Keys for table %s cannot be empty", tableName)
		}
	}
	return nil
}
