<?php declare(strict_types = 0);
/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/


use PHPUnit\Framework\TestCase;

class CHistFunctionValidatorTest extends TestCase {

	/**
	 * An array of history functions, options and the expected results.
	 */
	public function dataProvider() {
		return [
			['foo(/host/key)', [], ['rc' => false, 'error' => 'unknown function "foo"']],
			['avg_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "avg_foreach"']],
			['count_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "count_foreach"']],
			['exists_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "exists_foreach"']],
			['last_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "last_foreach"']],
			['max_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "max_foreach"']],
			['min_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "min_foreach"']],
			['sum_foreach(/host/key)', [], ['rc' => false, 'error' => 'unknown function "sum_foreach"']],

			['avg(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "avg"']],
			['avg(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, {$PERIOD})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, {$PERIOD}:now-5h)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, {$PERIOD}:now/h-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, {{$PERIOD}.regsub("^([0-9]+)", \1)}:now/{{$OFFSET}.regsub("^([0-9]+)", \1)}-{{$TIMESHIFT}.regsub("^([0-9]+)", \1)})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, "{$PERIOD}")', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, "{$PERIOD}d")', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, "{#PERIOD}")', ['usermacros' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, "{#PERIOD}")', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, "{#PERIOD}w")', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['avg(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 35791394m)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 596523h)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 24855d)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 3550w)', [], ['rc' => true, 'error' => null]],
			['avg(/host/key, 2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, 35791395m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, 596524h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, 24856d)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, 3551w)', [], ['rc' => false, 'error' => 'invalid second parameter in function "avg"']],
			['avg(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "avg"']],

			['bucket_percentile(/host/key)', ['calculated' => true], ['rc' => false, 'error' => 'mandatory parameter is missing in function "bucket_percentile"']],
			['bucket_percentile(/host/key, 1h)', ['calculated' => true], ['rc' => false, 'error' => 'mandatory parameter is missing in function "bucket_percentile"']],
			['bucket_percentile(/host/key, 1h, 101)', ['calculated' => true], ['rc' => false, 'error' => 'invalid third parameter in function "bucket_percentile"']],
			['bucket_percentile(/host/key, 1h, 50)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['bucket_percentile(/host/key, 1h:now-24h, 50)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "bucket_percentile"']],

			['bucket_rate_foreach(/host/key[*])', ['calculated' => true], ['rc' => false, 'error' => 'mandatory parameter is missing in function "bucket_rate_foreach"']],
			['bucket_rate_foreach(/host/key[*], 1h, 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid third parameter in function "bucket_rate_foreach"']],
			['bucket_rate_foreach(/host/key[*], 1h)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['bucket_rate_foreach(/host/key[*], 1h, 1)', ['calculated' => true], ['rc' => true, 'error' => null]],

			['change(/host/key)', [], ['rc' => true, 'error' => null]],
			['change(/host/key,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "change"']],

			['changecount(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "changecount"']],
			['changecount(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "changecount"']],
			['changecount(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "changecount"']],
			['changecount(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "changecount"']],
			['changecount(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "changecount"']],
			['changecount(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #256, "")', [], ['rc' => false, 'error' => 'invalid third parameter in function "changecount"']],
			['changecount(/host/key, #256, "test")', [], ['rc' => false, 'error' => 'invalid third parameter in function "changecount"']],
			['changecount(/host/key, #256, "inc")', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #256, "dec")', [], ['rc' => true, 'error' => null]],
			['changecount(/host/key, #256, "all")', [], ['rc' => true, 'error' => null]],

			['count(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "count"']],
			['count(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "count"']],
			['count(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "count"']],
			['count(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "count"']],
			['count(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['count(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['count(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "count"']],
			['count(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "str")', [], ['rc' => false, 'error' => 'invalid third parameter in function "count"']],
			['count(/host/key, #256, 10)', [], ['rc' => false, 'error' => 'invalid third parameter in function "count"']],
			['count(/host/key, #256, 10K)', [], ['rc' => false, 'error' => 'invalid third parameter in function "count"']],
			['count(/host/key, #256, "eq")', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "ne")', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "like")', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "bitand")', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp",)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp", 100)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp", 1s)', [], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp", {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp", {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['count(/host/key, #256, "regexp",,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "count"']],

			['countunique(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "countunique"']],
			['countunique(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "countunique"']],
			['countunique(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "countunique"']],
			['countunique(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "countunique"']],
			['countunique(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "countunique"']],
			['countunique(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "str")', [], ['rc' => false, 'error' => 'invalid third parameter in function "countunique"']],
			['countunique(/host/key, #256, 10)', [], ['rc' => false, 'error' => 'invalid third parameter in function "countunique"']],
			['countunique(/host/key, #256, 10K)', [], ['rc' => false, 'error' => 'invalid third parameter in function "countunique"']],
			['countunique(/host/key, #256, "eq")', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "ne")', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "like")', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "bitand")', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp",)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp", 100)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp", 1s)', [], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp", {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp", {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['countunique(/host/key, #256, "regexp",,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "countunique"']],

			['find(/host/key)', [], ['rc' => true, 'error' => null]],
			['find(/host/key,)', [], ['rc' => true, 'error' => null]],
			['find(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "find"']],
			['find(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "find"']],
			['find(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['find(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['find(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "find"']],
			['find(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "str")', [], ['rc' => false, 'error' => 'invalid third parameter in function "find"']],
			['find(/host/key, #256, 10)', [], ['rc' => false, 'error' => 'invalid third parameter in function "find"']],
			['find(/host/key, #256, 10K)', [], ['rc' => false, 'error' => 'invalid third parameter in function "find"']],
			['find(/host/key, #256, "eq")', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "ne")', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "like")', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "bitand")', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp",)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp", 100)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp", 1s)', [], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp", {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp", {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['find(/host/key, #256, "regexp",,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "find"']],

			['first(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "first"']],
			['first(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['first(/host/key,#1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['first(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['first(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['first(/host/key, #3:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key, #5:now/M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key, #2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "first"']],
			['first(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['first(/host/key, 1,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "first"']],

			['firstclock(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "firstclock"']],
			['firstclock(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['firstclock(/host/key,#1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['firstclock(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['firstclock(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['firstclock(/host/key, #3:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key, #5:now/M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key, #2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "firstclock"']],
			['firstclock(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['firstclock(/host/key, 1,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "firstclock"']],
			['firstclock(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "firstclock"']],
			['firstclock(/host/key, 1h, 1)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "firstclock"']],

			['forecast(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "forecast"']],
			['forecast(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "forecast"']],
			['forecast(/host/key,,10h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "forecast"']],
			['forecast(/host/key,#1)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "forecast"']],
			['forecast(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "forecast"']],
			['forecast(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "forecast"']],
			['forecast(/host/key,1,5s)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key,#1,10h)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key,1s,-7d)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key,1s, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key,1s, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key,1s, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key, 1m,)', [], ['rc' => false, 'error' => 'invalid third parameter in function "forecast"']],
			['forecast(/host/key, 1M,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "forecast"']],
			['forecast(/host/key, 1m:now/h-1h, 1d)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #3:now/h-1h, 1m)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #5:now/M, 1w)', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "linear")', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "polynomial3")', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "{$M}")', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, {$M})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "foo")', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "forecast"']],
			['forecast(/host/key, #256, 30d, 25d)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "forecast"']],
			['forecast(/host/key, #256, 30d, "exponential", "delta")', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "exponential", "avg")', [], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "exponential", "bar")', [], ['rc' => false, 'error' => 'invalid fifth parameter in function "forecast"']],
			['forecast(/host/key, #256, 30d, "exponential", "{#M}")', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "exponential", {#M})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['forecast(/host/key, #256, 30d, "exponential", "min",)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "forecast"']],

			['fuzzytime(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "fuzzytime"']],
			['fuzzytime(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 1)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 1h)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, "1h")', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 35791394m)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 596523h)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 24855d)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 3550w)', [], ['rc' => true, 'error' => null]],
			['fuzzytime(/host/key, 2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 35791395m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 596524h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 24856d)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 3551w)', [], ['rc' => false, 'error' => 'invalid second parameter in function "fuzzytime"']],
			['fuzzytime(/host/key, 1h,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "fuzzytime"']],

			['item_count(/host/key)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['item_count(/host/key,)', ['calculated' => true], ['rc' => false, 'error' => 'invalid number of parameters in function "item_count"']],

			['kurtosis(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "kurtosis"']],
			['kurtosis(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "kurtosis"']],
			['kurtosis(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "kurtosis"']],
			['kurtosis(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "kurtosis"']],
			['kurtosis(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "kurtosis"']],
			['kurtosis(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['kurtosis(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "kurtosis"']],

			['last(/host/key)', [], ['rc' => true, 'error' => null]],
			['last(/host/key,)', [], ['rc' => true, 'error' => null]],
			['last(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['last(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['last(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['last(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['last(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "last"']],
			['last(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "last"']],

			['lastclock(/host/key)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key,)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['lastclock(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "lastclock"']],
			['lastclock(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "lastclock"']],

			['logeventid(/host/key)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key,)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logeventid"']],
			['logeventid(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, "str")', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, 10)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, 10K)', [], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['logeventid(/host/key, #256, "str",)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "logeventid"']],

			['logseverity(/host/key)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key,)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['logseverity(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logseverity"']],
			['logseverity(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "logseverity"']],

			['logsource(/host/key)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key,)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logsource"']],
			['logsource(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, "str")', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, 10)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, 10K)', [], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['logsource(/host/key, #256, "str",)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "logsource"']],

			['logtimestamp(/host/key)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key,)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key,1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key,1s)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, 1m)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, 1m:now/h-1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['logtimestamp(/host/key, #2147483648)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, 2147483647)', [], ['rc' => false, 'error' => 'invalid second parameter in function "logtimestamp"']],
			['logtimestamp(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "logtimestamp"']],

			['mad(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "mad"']],
			['mad(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "mad"']],
			['mad(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "mad"']],
			['mad(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "mad"']],
			['mad(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "mad"']],
			['mad(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['mad(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "mad"']],

			['max(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "max"']],
			['max(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "max"']],
			['max(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "max"']],
			['max(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "max"']],
			['max(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['max(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['max(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "max"']],
			['max(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['max(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "max"']],

			['min(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "min"']],
			['min(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "min"']],
			['min(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "min"']],
			['min(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "min"']],
			['min(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['min(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['min(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "min"']],
			['min(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['min(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "min"']],

			['monodec(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "monodec"']],
			['monodec(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monodec"']],
			['monodec(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monodec"']],
			['monodec(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monodec"']],
			['monodec(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monodec"']],
			['monodec(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #256, "")', [], ['rc' => false, 'error' => 'invalid third parameter in function "monodec"']],
			['monodec(/host/key, #256, "test")', [], ['rc' => false, 'error' => 'invalid third parameter in function "monodec"']],
			['monodec(/host/key, #256, "weak")', [], ['rc' => true, 'error' => null]],
			['monodec(/host/key, #256, "strict")', [], ['rc' => true, 'error' => null]],

			['monoinc(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "monoinc"']],
			['monoinc(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monoinc"']],
			['monoinc(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monoinc"']],
			['monoinc(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monoinc"']],
			['monoinc(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "monoinc"']],
			['monoinc(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #256,)', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #256, "")', [], ['rc' => false, 'error' => 'invalid third parameter in function "monoinc"']],
			['monoinc(/host/key, #256, "test")', [], ['rc' => false, 'error' => 'invalid third parameter in function "monoinc"']],
			['monoinc(/host/key, #256, "weak")', [], ['rc' => true, 'error' => null]],
			['monoinc(/host/key, #256, "strict")', [], ['rc' => true, 'error' => null]],

			['nodata(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "nodata"']],
			['nodata(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, 1)', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h)', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, "{$INVALID_MACRO")', ['usermacros' => true], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, "{#INVALID_LLDMACRO")', ['lldmacros' => true], ['rc' => false, 'error' => 'invalid second parameter in function "nodata"']],
			['nodata(/host/key, "text{$MACRO}text")', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, "1h")', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h,)', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h,)', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h, "strict")', [], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['nodata(/host/key, 1h, "food")', [], ['rc' => false, 'error' => 'invalid third parameter in function "nodata"']],
			['nodata(/host/key, 1h, "strict",)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "nodata"']],

			['percentile(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "percentile"']],
			['percentile(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "percentile"']],
			['percentile(/host/key, #1)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "percentile"']],
			['percentile(/host/key, #1,)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "percentile"']],
			['percentile(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "percentile"']],
			['percentile(/host/key, 1h:now/h-1h, "abc")', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1h:now/h-1h, 5s)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1h:now/h-1h, 1m)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1h:now/h-1h, -1)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1h:now/h-1h, 101)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1h, 0.12345)', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1, 0)', [], ['rc' => true, 'error' => null]],
			['percentile(/host/key, 1h, 0.1234)', [], ['rc' => true, 'error' => null]],
			['percentile(/host/key, 1h, 99.9999)', [], ['rc' => true, 'error' => null]],
			['percentile(/host/key, #1, 100)', [], ['rc' => true, 'error' => null]],
			['percentile(/host/key, 1s, "abc")', [], ['rc' => false, 'error' => 'invalid third parameter in function "percentile"']],
			['percentile(/host/key, 1s, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['percentile(/host/key, 1s, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['percentile(/host/key, 1s, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['percentile(/host/key, #256, 5,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "percentile"']],

			['rate(/host/key)=1', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "rate"']],
			['rate(/host/key, 1h)=1', [], ['rc' => true, 'error' => null]],
			['rate(/host/key, 1h:now/h-1h)=1', [], ['rc' => true, 'error' => null]],
			['rate(/host/key, 1h, 1)=1', [], ['rc' => false, 'error' => 'invalid number of parameters in function "rate"']],

			['skewness(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "skewness"']],
			['skewness(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "skewness"']],
			['skewness(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "skewness"']],
			['skewness(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "skewness"']],
			['skewness(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "skewness"']],
			['skewness(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['skewness(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "skewness"']],

			['stddevpop(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "stddevpop"']],
			['stddevpop(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevpop"']],
			['stddevpop(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevpop"']],
			['stddevpop(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevpop"']],
			['stddevpop(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevpop"']],
			['stddevpop(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['stddevpop(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "stddevpop"']],

			['stddevsamp(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "stddevsamp"']],
			['stddevsamp(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevsamp"']],
			['stddevsamp(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevsamp"']],
			['stddevsamp(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevsamp"']],
			['stddevsamp(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "stddevsamp"']],
			['stddevsamp(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['stddevsamp(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "stddevsamp"']],

			['sum(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "sum"']],
			['sum(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sum"']],
			['sum(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sum"']],
			['sum(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sum"']],
			['sum(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sum"']],
			['sum(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['sum(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "sum"']],

			['sumofsquares(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "sumofsquares"']],
			['sumofsquares(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sumofsquares"']],
			['sumofsquares(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sumofsquares"']],
			['sumofsquares(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sumofsquares"']],
			['sumofsquares(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "sumofsquares"']],
			['sumofsquares(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['sumofsquares(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "sumofsquares"']],

			['timeleft(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "timeleft"']],
			['timeleft(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "timeleft"']],
			['timeleft(/host/key,#1)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "timeleft"']],
			['timeleft(/host/key,#1,)', [], ['rc' => false, 'error' => 'invalid third parameter in function "timeleft"']],
			['timeleft(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "timeleft"']],
			['timeleft(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "timeleft"']],
			['timeleft(/host/key,1,5s)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key,#1,10h)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key,1s,-7d)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key,1s, "abc")', [], ['rc' => false, 'error' => 'invalid third parameter in function "timeleft"']],
			['timeleft(/host/key,1s, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['timeleft(/host/key,1s, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['timeleft(/host/key,1s, "{$MACRO}{#LLDMACRO}")', ['usermacros' => true, 'lldmacros' => true], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, 1m:now/h-1h, 1d)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #3:now/h-1h, 1m)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #5:now/M, 1w)', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #256, 30d, "linear")', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #256, 30d, "polynomial3")', [], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #256, 30d, "{$M}")', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #256, 30d, {$M})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['timeleft(/host/key, #256, 30d, "foo")', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "timeleft"']],
			['timeleft(/host/key, #256, 30d, 25d)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "timeleft"']],
			['timeleft(/host/key, #256, 30d, "exponential",)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "timeleft"']],

			['trendavg(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendavg"']],
			['trendavg(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendavg"']],
			['trendavg(/host/key, 1h:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, 1d:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, 1y:now/M-1h)', [], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendavg(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "trendavg"']],

			['trendcount(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendcount"']],
			['trendcount(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendcount"']],
			['trendcount(/host/key, 1h:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, 1d:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, 1y:now/M-1h)', [], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendcount(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "trendcount"']],

			['trendmax(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendmax"']],
			['trendmax(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmax"']],
			['trendmax(/host/key, 1h:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, 1d:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, 1y:now/M-1h)', [], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendmax(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "trendmax"']],

			['trendmin(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendmin"']],
			['trendmin(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendmin"']],
			['trendmin(/host/key, 1h:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, 1d:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, 1y:now/M-1h)', [], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendmin(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "trendmin"']],

			['trendstl(/host/key)', ['calculated' => true], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendstl"']],
			['trendstl(/host/key, 0, 1h, 1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, 1m:now/h, 1h, 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, #0, 1h, 1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, 1y, 1h, 1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1M, 1d)', [], ['rc' => false, 'error' => 'invalid third parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1y, 1d)', [], ['rc' => false, 'error' => 'invalid third parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 0, 1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid third parameter in function "trendstl"']],
			['trendstl(/host/key, 1h:now/h, 1m, 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid third parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fourth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1h)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 0.5)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 1.55555)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 2147483647.5)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 2147483648)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, -1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1w, 1h, 1d, 1, 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid sixth parameter in function "trendstl"']],
			['trendstl(/host/key, 1h:now/d, 1h, 1d, 1, "test")', ['calculated' => true], ['rc' => false, 'error' => 'invalid sixth parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1h, 1d, 1, "mad", 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid seventh parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1h, 1d, 1, "mad", 7.5)', ['calculated' => true], ['rc' => false, 'error' => 'invalid seventh parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1h, 1d, 1, "mad", 2147483648)', ['calculated' => true], ['rc' => false, 'error' => 'invalid seventh parameter in function "trendstl"']],
			['trendstl(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendstl"']],
			['trendstl(/host/key, 0, 1h, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, #0, 1h, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, 1y, 1h, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 0, 1)', [], ['rc' => false, 'error' => 'invalid third parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 0)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 0)', [], ['rc' => false, 'error' => 'invalid fifth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 2h, 1, 0)', [], ['rc' => false, 'error' => 'invalid sixth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1d, 1, "test")', [], ['rc' => false, 'error' => 'invalid sixth parameter in function "trendstl"']],
			['trendstl(/host/key, 1d:now/M-1h, 1h, 1w, 1, "mad", 0)', [], ['rc' => false, 'error' => 'invalid seventh parameter in function "trendstl"']],
			['trendstl(/host/key, 240h:now/h, 1h, 1d)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 1y:now/h, 1h, 1d)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 4w, 1d)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 365d, 1d)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 365d, 1d, 1.53)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 365d, 1d, 2147483647)', [], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, {$PERIOD}:{$TIMESHIFT}, 1h, 1d)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 1h, 1d, 1, "mad", 7)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, 240h:now/h, 1h, 1d, 1, "mad", 2147483647)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, {$PARAM2}, {$MACRO3}, {$MACRO4}, {$MACRO5}, {$MACRO6}, {$MACRO7})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, {#PARAM2}, {#PARAM3}, {#PARAM4}), {#PARAM5}), {#PARAM6}), {#PARAM7})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendstl(/host/key, {$PARAM2}, {#PARAM3}, {$PARAM4}), {#PARAM5}), {$PARAM6}), {#PARAM7})', ['lldmacros' => true, 'usermacros' => true], ['rc' => true, 'error' => null]],

			['trendsum(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "trendsum"']],
			['trendsum(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "trendsum"']],
			['trendsum(/host/key, 1h:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, 1d:now-1h)', [], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, 1y:now/M-1h)', [], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['trendsum(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "trendsum"']],

			['varpop(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "varpop"']],
			['varpop(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varpop"']],
			['varpop(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varpop"']],
			['varpop(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varpop"']],
			['varpop(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varpop"']],
			['varpop(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['varpop(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "varpop"']],

			['varsamp(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "varsamp"']],
			['varsamp(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varsamp"']],
			['varsamp(/host/key,0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varsamp"']],
			['varsamp(/host/key,#0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varsamp"']],
			['varsamp(/host/key,1)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key,#1)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key,1s)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, 1m)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, 1M)', [], ['rc' => false, 'error' => 'invalid second parameter in function "varsamp"']],
			['varsamp(/host/key, 1m:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, #3:now/h-1h)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, #5:now/M)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, #2147483647)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, 2147483647)', [], ['rc' => true, 'error' => null]],
			['varsamp(/host/key, #256,)', [], ['rc' => false, 'error' => 'invalid number of parameters in function "varsamp"']],

			['avg_foreach(/host/key)', ['calculated' => true], ['rc' => false, 'error' => 'mandatory parameter is missing in function "avg_foreach"']],
			['avg_foreach(/host/key,)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, #0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, #1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, 1)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['avg_foreach(/host/key, 1h)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['avg_foreach(/host/key, 1d)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['avg_foreach(/host/key, 1d:now/d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "avg_foreach"']],
			['avg_foreach(/host/key, {$MACRO})', ['usermacros' => true, 'calculated' => true], ['rc' => true, 'error' => null]],
			['avg_foreach(/host/key, {#LLDMACRO})', ['lldmacros' => true, 'calculated' => true], ['rc' => true, 'error' => null]],
			['avg_foreach(/host/key, 1d,)', ['calculated' => true], ['rc' => false, 'error' => 'invalid number of parameters in function "avg_foreach"']],

			['exists_foreach(/host/key)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['exists_foreach(/host/key,)', ['calculated' => true], ['rc' => false, 'error' => 'invalid number of parameters in function "exists_foreach"']],

			['last_foreach(/host/key)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key,)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, 0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, #0)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, #1)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, 1)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, 1h)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, 1d)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, 1d:now/d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, {$PERIOD}:{$TIMESHIFT})', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, {$PERIOD}:now-1d)', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, {$PERIOD}:now-{$TIMESHIFT})', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, {{$PERIOD}.regsub("^([0-9]+)", \1)}:now-{{$TIMESHIFT}.regsub("^([0-9]+)", \1)})', ['usermacros' => true, 'calculated' => true], ['rc' => false, 'error' => 'invalid second parameter in function "last_foreach"']],
			['last_foreach(/host/key, {$MACRO})', ['usermacros' => true, 'calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, {#LLDMACRO})', ['lldmacros' => true, 'calculated' => true], ['rc' => true, 'error' => null]],
			['last_foreach(/host/key, 1d,)', ['calculated' => true], ['rc' => false, 'error' => 'invalid number of parameters in function "last_foreach"']],

			// Wildcards in non-aggregating functions.
			['sum(/*/key[p1,p2], 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum"']],
			['sum(/host/*, 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum"']],
			['sum(/host/key[*,p2], 1d)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['sum(/host/key[p1,*], 1d)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['sum(/host/key[*,*], 1d)', ['calculated' => true], ['rc' => true, 'error' => null]],
			['sum(/*/*, 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum"']],
			['sum(/*/key[*,p2], 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum"']],

			// Wildcards in aggregating functions.
			['sum_foreach(/*/key[p1,p2], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],
			['sum_foreach(/host/*, 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],
			['sum_foreach(/host/key[*,p2], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],
			['sum_foreach(/host/key[p1,*], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],
			['sum_foreach(/host/key[*,*], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],
			['sum_foreach(/*/*, 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum_foreach"']],
			['sum_foreach(/*/key[*,p2], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],

			// Filters.
			['sum(/host/key?[tag="foo"], 1d)', ['calculated' => true], ['rc' => false, 'error' => 'invalid first parameter in function "sum"']],
			['sum_foreach(/host/key?[tag="foo"], 1d)', ['calculated' => true, 'aggregating' => true], ['rc' => true, 'error' => null]],

			['baselinedev(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinedev"']],
			['baselinedev(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now-1h)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinedev"']],
			['baselinedev(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid third parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now-1h, 1M)', [], ['rc' => false, 'error' => 'invalid third parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now-1h, "M")', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now-1h, "M", "M")', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now-1h, "M", 2)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1d:now-1d, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1d:now-1d, "y", -1)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 0)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5.0)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5.00000000000000000000000000000000000000000000000000001)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5.2)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5e-1)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5e3d)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now-1d, "y", 5e3)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1M:now/M+2M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1h:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1y:now/M, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1M:now/w, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1w:now/d, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1d:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1s:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1m:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinedev"']],
			['baselinedev(/host/key, 1h:now/h, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1h:now/d, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1h:now/w, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1h:now/M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1d:now/w, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1w:now/M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, 1M:now/y, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {$PERIOD}:{$TIMESHIFT}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {$PERIOD}:now-{$TIMESHIFT}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {$MACRO}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {$MACRO}, {$MACRO}, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {#LLDMACRO}, {#LLDMACRO}, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['baselinedev(/host/key, {#LLDMACRO}, {$MACRO}, {#LLDMACRO})', ['lldmacros' => true], ['rc' => false, 'error' => 'invalid third parameter in function "baselinedev"']],
			['baselinedev(/host/key, {#LLDMACRO}, {$MACRO}, {#LLDMACRO})', ['lldmacros' => true, 'usermacros' => true], ['rc' => true, 'error' => null]],

			['baselinewma(/host/key)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinewma"']],
			['baselinewma(/host/key,)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, #0)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, #1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now-1h)', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinewma"']],
			['baselinewma(/host/key, 1y:now/y,)', [], ['rc' => false, 'error' => 'invalid third parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now-1h, 1M)', [], ['rc' => false, 'error' => 'invalid third parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now-1h, "M")', [], ['rc' => false, 'error' => 'mandatory parameter is missing in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now-1h, "M", "M")', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now-1h, "M", 2)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1d:now-1d, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1d:now-1d, "y", -1)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 0)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5.0)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5.00000000000000000000000000000000000000000000000000001)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5.2)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5e-1)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5e3d)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now-1d, "y", 5e3)', [], ['rc' => false, 'error' => 'invalid fourth parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1M:now/M+2M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1h:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1y:now/M, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1M:now/w, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1w:now/d, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1d:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1s:now/m, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1m:now/h, "y", 1)', [], ['rc' => false, 'error' => 'invalid second parameter in function "baselinewma"']],
			['baselinewma(/host/key, 1h:now/h, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1h:now/d, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1h:now/w, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1h:now/M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1d:now/w, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1w:now/M, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, 1M:now/y, "y", 1)', [], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {$PERIOD}:{$TIMESHIFT}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {$PERIOD}:now-{$TIMESHIFT}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {{$PERIOD}.func()}:now-{{$TIMESHIFT: context}.func()}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {$MACRO}, "y", 2)', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {$MACRO}, {$MACRO}, {$MACRO})', ['usermacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {#LLDMACRO}, {#LLDMACRO}, {#LLDMACRO})', ['lldmacros' => true], ['rc' => true, 'error' => null]],
			['baselinewma(/host/key, {#LLDMACRO}, {$MACRO}, {#LLDMACRO})', ['lldmacros' => true], ['rc' => false, 'error' => 'invalid third parameter in function "baselinewma"']],
			['baselinewma(/host/key, {#LLDMACRO}, {$MACRO}, {#LLDMACRO})', ['lldmacros' => true, 'usermacros' => true], ['rc' => true, 'error' => null]]
		];
	}

	/**
	 * @dataProvider dataProvider
	 */
	public function testHistFunctionValidator(string $source, array $options, array $expected) {
		$expression_parser = new CExpressionParser([
			'usermacros' => true,
			'lldmacros' => true
		] + $options);
		$hist_function_validator = new CHistFunctionValidator([
			'parameters' => (new CHistFunctionData($options))->getParameters()
		] + $options);
		$expression_parser->parse($source);
		$this->assertNotNull($expression_parser->getResult(), $expression_parser->getError());
		$tokens = $expression_parser->getResult()->getTokens();

		$this->assertSame(CExpressionParserResult::TOKEN_TYPE_HIST_FUNCTION, $tokens[0]['type']);
		$this->assertSame($expected, [
			'rc' => $hist_function_validator->validate($tokens[0]),
			'error' => $hist_function_validator->getError()
		]);
	}
}
