<?php
/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/


use PHPUnit\Framework\TestCase;

class C10TriggerExpressionTest extends TestCase {

	public static function provider() {
		return [
			['', null, false],
			['+', null, false],
			['1+1', null, true],
			['abc', null, false],
			['{#LLD}', null, false, ['lldmacros' => false]],
			['{#LLD}', null, true, ['lldmacros' => true]],
			['{#LLD}', null, true],

			['.5', null, true],
			['5.', null, true],
			['..5', null, false],
			['5..', null, false],

			['1', null, true],
			['1s', null, true],
			['1m', null, true],
			['1h', null, true],
			['1d', null, true],
			['1w', null, true],
			['1K', null, true],
			['1M', null, true],
			['1G', null, true],
			['1T', null, true],
			['1.5', null, true],
			['1.5s', null, true],
			['1.5m', null, true],
			['1.5h', null, true],
			['1.5d', null, true],
			['1.5w', null, true],
			['1.5K', null, true],
			['1.5M', null, true],
			['1.5G', null, true],
			['1.5T', null, true],
			['-1.5', null, true],
			['-1.5s', null, true],
			['-1.5m', null, true],
			['-1.5h', null, true],
			['-1.5d', null, true],
			['-1.5w', null, true],
			['-1.5K', null, true],
			['-1.5M', null, true],
			['-1.5G', null, true],
			['-1.5T', null, true],

			['{TRIGGER.VALUE}', null, true],
			['{$USERMACRO}', null, true],
			['{TRIGGER.VALUE}=1', null, true],
			['{$USERMACRO}=1', null, true],

			['{host}', null, false],
			['{host:key}', null, false],
			['{host:key.str}', null, false],

			['{host:key.diff()} and {TRIGGER.VALUE}', null, true],
			['{host:key.diff()}and {TRIGGER.VALUE}', null, false],
			['{host:key.diff()}  and{TRIGGER.VALUE}', null, false],
			['{host:key.diff()}and{TRIGGER.VALUE}', null, false],
			['{host:key.diff()}and+{TRIGGER.VALUE}', null, false],
			['{host:key.diff()}and-{TRIGGER.VALUE}', null, false],
			['{host:key.diff()} and + {TRIGGER.VALUE}', null, false],
			['{host:key.diff()} and - {TRIGGER.VALUE}', null, true],

			['{host:key.diff()} and {$USERMACRO}', null, true],
			['{host:key.diff()}and {$USERMACRO}', null, false],
			['{host:key.diff()} and{$USERMACRO}', null, false],
			['{host:key.diff()}and{$USERMACRO}', null, false],
			['{host:key.diff()}and+{$USERMACRO}', null, false],
			['{host:key.diff()}and-{$USERMACRO}', null, false],
			['{host:key.diff()} and + {$USERMACRO}', null, false],
			['{host:key.diff()} and - {$USERMACRO}', null, true],

			// basic "not" support
			['not 1', null, true],
			['not (1)', null, true],
			['not -1', null, true],
			['not (-1)', null, true],
			['not -(1)', null, true],
			['not {TRIGGER.VALUE}', null, true],
			['not ({TRIGGER.VALUE})', null, true],
			['not -{TRIGGER.VALUE}', null, true],
			['not (-{TRIGGER.VALUE})', null, true],
			['not -({TRIGGER.VALUE})', null, true],
			['not {$USERMACRO}', null, true],
			['not -{$USERMACRO}', null, true],
			['not (-{$USERMACRO})', null, true],
			['not -({$USERMACRO})', null, true],
			['not {host:key.diff()}', null, true],
			['not ({host:key.diff()})', null, true],
			['not (-{host:key.diff()})', null, true],
			['not -({host:key.diff()})', null, true],

			['not1', null, false],
			['not(1)', null, true],
			['not-1', null, false],
			['not(-1)', null, true],
			['not-(1)', null, false],
			['not{TRIGGER.VALUE}', null, false],
			['not({TRIGGER.VALUE})', null, true],
			['not-{TRIGGER.VALUE}', null, false],
			['not(-{TRIGGER.VALUE})', null, true],
			['not-({TRIGGER.VALUE})', null, false],
			['not{$USERMACRO}', null, false],
			['not-{$USERMACRO}', null, false],
			['not(-{$USERMACRO})', null, true],
			['not-({$USERMACRO})', null, false],
			['not{host:key.diff()}', null, false],
			['not({host:key.diff()})', null, true],
			['not(-{host:key.diff()})', null, true],
			['not-({host:key.diff()})', null, false],

			['- not 1', null, false],
			['-not 1', null, false],
			['- not1', null, false],
			['-not1', null, false],
			['1 not 1', null, false],
			['(1) not 1', null, false],
			['1not1', null, false],
			['(1)not1', null, false],

			// operator cases
			['Not 1', null, false],
			['NOT 1', null, false],
			['1 Or 1', null, false],
			['1 OR 1', null, false],
			['1 And 1', null, false],
			['1 AND 1', null, false],

			['{host:key.diff()}=00', null, true],
			['{host:key.diff()}=0 0', null, false],
			['{host:key.diff()}=0 0={host:key.diff()}', null, false],
			['{host:key.diff()} = 00 = {host:key.diff()}', null, true],

			['{host:key.str(ГУГЛ)}=0', null, true],
			['{host:key.str("ГУГЛ")}=0', null, true],
			['{host:key[ГУГЛ].str(ГУГЛ)}=0', null, true],
			['{host:key["ГУГЛ"].str("ГУГЛ")}=0', null, true],
			['{host:key.str("こんにちは、世界")}', null, true],
			['{host:key.str(こんにちは、世界)}', null, true],
			['{host:key["こんにちは、世界"].str("こんにちは、世界")}', null, true],
			['{host:key[こんにちは、世界].str(こんにちは、世界)}', null, true],

			['{host:key[a,,"b",,[c,d,,"e",],,[f]].count(1,,"b",3)}', null, true],
			['{host:key[a,,"b",,[[c,d,,"e"],[]],,[f]].count(1,,"b",3)}', null, false],
			['{host:key[a,,"b",,[c,d,,"e",,,[f]].count(1,,"b",3)}', null, false],
			['{host:key[a,,"b",,[c,d,,"e",],,f]].count(1,,"b",3)}', null, false],

			['{abcdefghijklmnopqrstuvwxyz. _-ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890:key.diff()}', null, true],
			['{host:abcdefghijklmnopqrstuvwxyz._-ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890.diff()}', null, true],
			['{host:key.abcdefghijklmnopqrstuvwxyz()}', null, true],

			['{host:,.diff()}', null, false],
			['{host:;.diff()}', null, false],
			['{host::.diff()}', null, false],

			['{:key.diff()}', null, false],
			['{host:.diff()}', null, false],
			['{host:key.()}', null, false],

			[' {host:key.diff()} ', null, true],
			['({host:key.diff()})', null, true],
			['{(host:key.diff()})', null, false],
			['({host:key.diff())}', null, false],
			['{(host:key.diff())}', null, false],
			['{(host:key.diff()})=0', null, false],
			['({host:key.diff())}=0', null, false],
			['{(host:key.diff())}=0', null, false],
			['({host:key.diff()}=)0', null, false],
			['({host:key.diff()})0', null, false],
			['0(={host:key.diff()})', null, false],
			['0({host:key.diff()})', null, false],
			['( {host:key.diff()} )', null, true],
			[' ( {host:key.diff()} ) ', null, true],
			['(( {host:key.diff()} ))', null, true],
			[' ( ( {host:key.diff()} ) ) ', null, true],
			['((( {host:key.diff()} )))', null, true],
			[' ( ( ( {host:key.diff()} ) ) ) ', null, true],
			['()0={host:key.diff()}', null, false],
			['0()={host:key.diff()}', null, false],
			['0=()={host:key.diff()}', null, false],
			['0=(){host:key.diff()}', null, false],
			['0={()host:key.diff()}', null, false],
			['0={host:key.diff()()}', null, false],
			['0={host:key.diff()}()', null, false],
			['0={host:key.diff()}+()()()()5', null, false],
			['0={host:key.diff()}+((((()))))5', null, false],
			['(0)={host:key.diff()}', null, true],
			['(0+)={host:key.diff()}', null, false],
			['(0=){host:key.diff()}', null, false],
			['({host:key.diff)()}', null, false],
			['({host:key.)diff()}', null, false],
			['({host:key).diff()}', null, false],
			['(-5)={host:key.diff()}', null, true],
			['(15 - 5.25 - 1)={host:key.diff()}', null, true],
			['{host:key.diff()} = -5', null, true],
			['{0:0.last(0)}=0', null, true],

			['(({host:key.diff()})=0)', null, true],
			['( ( {host:key.diff()} ) = 0 )', null, true],
			['(({host:key.diff()}) * 100) / 95', null, true],
			['(({host:key.diff()}) * 5.25K) / 95.0', null, true],
			['(({host:key.diff()}) * 1w) / 1d', null, true],
			['(({host:key.diff()}) * 1w) / 1Ks', null, false],
			['(({host:key.diff()}) * 1w) / (1d * ({host:key.diff()})', null, false],
			['(({host:key.diff()}) * 1w) / (1d * host:key.diff()}))', null, false],
			['(({host:key.diff()}) * 1w) / (1d * ({host:key.diff()}))', null, true],

			['{host:key.last(1)} * (-1)', null, true],
			['{host:key.last(1)} / (-1)', null, true],
			['{host:key.last(1)} + (-1)', null, true],
			['{host:key.last(1)} - (-1)', null, true],
			['{host:key.last(1)} = (-1)', null, true],
			['{host:key.last(1)} <> (-1)', null, true],
			['{host:key.last(1)} < (-1)', null, true],
			['{host:key.last(1)} > (-1)', null, true],
			['{host:key.last(1)} and (-1)', null, true],
			['{host:key.last(1)} or (-1)', null, true],

			['{host:key.last(1)} * not (-1)', null, true],
			['{host:key.last(1)} / not (-1)', null, true],
			['{host:key.last(1)} + not (-1)', null, true],
			['{host:key.last(1)} - not (-1)', null, true],
			['{host:key.last(1)} = not (-1)', null, true],
			['{host:key.last(1)} <> not (-1)', null, true],
			['{host:key.last(1)} < not (-1)', null, true],
			['{host:key.last(1)} > not (-1)', null, true],
			['{host:key.last(1)} and not (-1)', null, true],
			['{host:key.last(1)} or not (-1)', null, true],

			['{host:key.last(1)}* (-1)', null, true],
			['{host:key.last(1)}/ (-1)', null, true],
			['{host:key.last(1)}+ (-1)', null, true],
			['{host:key.last(1)}- (-1)', null, true],
			['{host:key.last(1)}= (-1)', null, true],
			['{host:key.last(1)}<> (-1)', null, true],
			['{host:key.last(1)}< (-1)', null, true],
			['{host:key.last(1)}> (-1)', null, true],
			['{host:key.last(1)}and (-1)', null, false],
			['{host:key.last(1)}or (-1)', null, false],

			['{host:key.last(1)}* not (-1)', null, true],
			['{host:key.last(1)}/ not (-1)', null, true],
			['{host:key.last(1)}+ not (-1)', null, true],
			['{host:key.last(1)}- not (-1)', null, true],
			['{host:key.last(1)}= not (-1)', null, true],
			['{host:key.last(1)}<> not (-1)', null, true],
			['{host:key.last(1)}< not (-1)', null, true],
			['{host:key.last(1)}> not (-1)', null, true],
			['{host:key.last(1)}and not (-1)', null, false],
			['{host:key.last(1)}or not (-1)', null, false],

			['{host:key.last(1)} *(-1)', null, true],
			['{host:key.last(1)} /(-1)', null, true],
			['{host:key.last(1)} +(-1)', null, true],
			['{host:key.last(1)} -(-1)', null, true],
			['{host:key.last(1)} =(-1)', null, true],
			['{host:key.last(1)} <>(-1)', null, true],
			['{host:key.last(1)} <(-1)', null, true],
			['{host:key.last(1)} >(-1)', null, true],
			['{host:key.last(1)} and(-1)', null, true],
			['{host:key.last(1)} or(-1)', null, true],

			['{host:key.last(1)} * not(-1)', null, true],
			['{host:key.last(1)} / not(-1)', null, true],
			['{host:key.last(1)} + not(-1)', null, true],
			['{host:key.last(1)} - not(-1)', null, true],
			['{host:key.last(1)} = not(-1)', null, true],
			['{host:key.last(1)} <> not(-1)', null, true],
			['{host:key.last(1)} < not(-1)', null, true],
			['{host:key.last(1)} > not(-1)', null, true],
			['{host:key.last(1)} and not(-1)', null, true],
			['{host:key.last(1)} or not(-1)', null, true],

			['{host:key.last(1)}*(-1)', null, true],
			['{host:key.last(1)}/(-1)', null, true],
			['{host:key.last(1)}+(-1)', null, true],
			['{host:key.last(1)}-(-1)', null, true],
			['{host:key.last(1)}=(-1)', null, true],
			['{host:key.last(1)}<>(-1)', null, true],
			['{host:key.last(1)}<(-1)', null, true],
			['{host:key.last(1)}>(-1)', null, true],
			['{host:key.last(1)}and(-1)', null, false],
			['{host:key.last(1)}or(-1)', null, false],

			['{host:key.last(1)}*not(-1)', null, false],
			['{host:key.last(1)}/not(-1)', null, false],
			['{host:key.last(1)}+not(-1)', null, false],
			['{host:key.last(1)}-not(-1)', null, false],
			['{host:key.last(1)}=not(-1)', null, false],
			['{host:key.last(1)}<>not(-1)', null, false],
			['{host:key.last(1)}<not(-1)', null, false],
			['{host:key.last(1)}>not(-1)', null, false],
			['{host:key.last(1)}andnot(-1)', null, false],
			['{host:key.last(1)}ornot(-1)', null, false],

			['{host:key.last(1)} * -1', null, true],
			['{host:key.last(1)} / -1', null, true],
			['{host:key.last(1)} + -1', null, true],
			['{host:key.last(1)} - -1', null, true],
			['{host:key.last(1)} = -1', null, true],
			['{host:key.last(1)} <> -1', null, true],
			['{host:key.last(1)} < -1', null, true],
			['{host:key.last(1)} > -1', null, true],
			['{host:key.last(1)} and -1', null, true],
			['{host:key.last(1)} or -1', null, true],

			['{host:key.last(1)} * not -1', null, true],
			['{host:key.last(1)} / not -1', null, true],
			['{host:key.last(1)} + not -1', null, true],
			['{host:key.last(1)} - not -1', null, true],
			['{host:key.last(1)} = not -1', null, true],
			['{host:key.last(1)} <> not -1', null, true],
			['{host:key.last(1)} < not -1', null, true],
			['{host:key.last(1)} > not -1', null, true],
			['{host:key.last(1)} and not -1', null, true],
			['{host:key.last(1)} or not -1', null, true],

			['{host:key.last(1)}* -1', null, true],
			['{host:key.last(1)}/ -1', null, true],
			['{host:key.last(1)}+ -1', null, true],
			['{host:key.last(1)}- -1', null, true],
			['{host:key.last(1)}= -1', null, true],
			['{host:key.last(1)}<> -1', null, true],
			['{host:key.last(1)}< -1', null, true],
			['{host:key.last(1)}> -1', null, true],
			['{host:key.last(1)}and -1', null, false],
			['{host:key.last(1)}or -1', null, false],

			['{host:key.last(1)}* not -1', null, true],
			['{host:key.last(1)}/ not -1', null, true],
			['{host:key.last(1)}+ not -1', null, true],
			['{host:key.last(1)}- not -1', null, true],
			['{host:key.last(1)}= not -1', null, true],
			['{host:key.last(1)}<> not -1', null, true],
			['{host:key.last(1)}< not -1', null, true],
			['{host:key.last(1)}> not -1', null, true],
			['{host:key.last(1)}and not -1', null, false],
			['{host:key.last(1)}or not -1', null, false],

			['{host:key.last(1)} *-1', null, true],
			['{host:key.last(1)} /-1', null, true],
			['{host:key.last(1)} +-1', null, true],
			['{host:key.last(1)} --1', null, true],
			['{host:key.last(1)} =-1', null, true],
			['{host:key.last(1)} <>-1', null, true],
			['{host:key.last(1)} <-1', null, true],
			['{host:key.last(1)} >-1', null, true],
			['{host:key.last(1)} and-1', null, false],
			['{host:key.last(1)} or-1', null, false],

			['{host:key.last(1)} * not-1', null, false],
			['{host:key.last(1)} / not-1', null, false],
			['{host:key.last(1)} + not-1', null, false],
			['{host:key.last(1)} - not-1', null, false],
			['{host:key.last(1)} = not-1', null, false],
			['{host:key.last(1)} <> not-1', null, false],
			['{host:key.last(1)} < not-1', null, false],
			['{host:key.last(1)} > not-1', null, false],
			['{host:key.last(1)} and not-1', null, false],
			['{host:key.last(1)} or not-1', null, false],

			['{host:key.last(1)}*-1', null, true],
			['{host:key.last(1)}/-1', null, true],
			['{host:key.last(1)}+-1', null, true],
			['{host:key.last(1)}--1', null, true],
			['{host:key.last(1)}=-1', null, true],
			['{host:key.last(1)}<>-1', null, true],
			['{host:key.last(1)}<-1', null, true],
			['{host:key.last(1)}>-1', null, true],
			['{host:key.last(1)}and-1', null, false],
			['{host:key.last(1)}or-1', null, false],

			['{host:key.last(1)}*not-1', null, false],
			['{host:key.last(1)}/not-1', null, false],
			['{host:key.last(1)}+not-1', null, false],
			['{host:key.last(1)}-not-1', null, false],
			['{host:key.last(1)}=not-1', null, false],
			['{host:key.last(1)}<>not-1', null, false],
			['{host:key.last(1)}<not-1', null, false],
			['{host:key.last(1)}>not-1', null, false],
			['{host:key.last(1)}andnot-1', null, false],
			['{host:key.last(1)}ornot-1', null, false],

			['{host:key.last(1)} * (- 1)', null, true],
			['{host:key.last(1)} / (- 1)', null, true],
			['{host:key.last(1)} + (- 1)', null, true],
			['{host:key.last(1)} - (- 1)', null, true],
			['{host:key.last(1)} = (- 1)', null, true],
			['{host:key.last(1)} <> (- 1)', null, true],
			['{host:key.last(1)} < (- 1)', null, true],
			['{host:key.last(1)} > (- 1)', null, true],
			['{host:key.last(1)} and (- 1)', null, true],
			['{host:key.last(1)} or (- 1)', null, true],

			['{host:key.last(1)} * not (- 1)', null, true],
			['{host:key.last(1)} / not (- 1)', null, true],
			['{host:key.last(1)} + not (- 1)', null, true],
			['{host:key.last(1)} - not (- 1)', null, true],
			['{host:key.last(1)} = not (- 1)', null, true],
			['{host:key.last(1)} <> not (- 1)', null, true],
			['{host:key.last(1)} < not (- 1)', null, true],
			['{host:key.last(1)} > not (- 1)', null, true],
			['{host:key.last(1)} and not (- 1)', null, true],
			['{host:key.last(1)} or not (- 1)', null, true],

			['{host:key.last(1)}* (- 1)', null, true],
			['{host:key.last(1)}/ (- 1)', null, true],
			['{host:key.last(1)}+ (- 1)', null, true],
			['{host:key.last(1)}- (- 1)', null, true],
			['{host:key.last(1)}= (- 1)', null, true],
			['{host:key.last(1)}<> (- 1)', null, true],
			['{host:key.last(1)}< (- 1)', null, true],
			['{host:key.last(1)}> (- 1)', null, true],
			['{host:key.last(1)}and (- 1)', null, false],
			['{host:key.last(1)}or (- 1)', null, false],

			['{host:key.last(1)}* not (- 1)', null, true],
			['{host:key.last(1)}/ not (- 1)', null, true],
			['{host:key.last(1)}+ not (- 1)', null, true],
			['{host:key.last(1)}- not (- 1)', null, true],
			['{host:key.last(1)}= not (- 1)', null, true],
			['{host:key.last(1)}<> not (- 1)', null, true],
			['{host:key.last(1)}< not (- 1)', null, true],
			['{host:key.last(1)}> not (- 1)', null, true],
			['{host:key.last(1)}and not (- 1)', null, false],
			['{host:key.last(1)}or not (- 1)', null, false],

			['{host:key.last(1)} *(- 1)', null, true],
			['{host:key.last(1)} /(- 1)', null, true],
			['{host:key.last(1)} +(- 1)', null, true],
			['{host:key.last(1)} -(- 1)', null, true],
			['{host:key.last(1)} =(- 1)', null, true],
			['{host:key.last(1)} <>(- 1)', null, true],
			['{host:key.last(1)} <(- 1)', null, true],
			['{host:key.last(1)} >(- 1)', null, true],
			['{host:key.last(1)} and(- 1)', null, true],
			['{host:key.last(1)} or(- 1)', null, true],

			['{host:key.last(1)} * not(- 1)', null, true],
			['{host:key.last(1)} / not(- 1)', null, true],
			['{host:key.last(1)} + not(- 1)', null, true],
			['{host:key.last(1)} - not(- 1)', null, true],
			['{host:key.last(1)} = not(- 1)', null, true],
			['{host:key.last(1)} <> not(- 1)', null, true],
			['{host:key.last(1)} < not(- 1)', null, true],
			['{host:key.last(1)} > not(- 1)', null, true],
			['{host:key.last(1)} and not(- 1)', null, true],
			['{host:key.last(1)} or not(- 1)', null, true],

			['{host:key.last(1)}*(- 1)', null, true],
			['{host:key.last(1)}/(- 1)', null, true],
			['{host:key.last(1)}+(- 1)', null, true],
			['{host:key.last(1)}-(- 1)', null, true],
			['{host:key.last(1)}=(- 1)', null, true],
			['{host:key.last(1)}<>(- 1)', null, true],
			['{host:key.last(1)}<(- 1)', null, true],
			['{host:key.last(1)}>(- 1)', null, true],
			['{host:key.last(1)}and(- 1)', null, false],
			['{host:key.last(1)}or(- 1)', null, false],

			['{host:key.last(1)}*not(- 1)', null, false],
			['{host:key.last(1)}/not(- 1)', null, false],
			['{host:key.last(1)}+not(- 1)', null, false],
			['{host:key.last(1)}-not(- 1)', null, false],
			['{host:key.last(1)}=not(- 1)', null, false],
			['{host:key.last(1)}<>not(- 1)', null, false],
			['{host:key.last(1)}<not(- 1)', null, false],
			['{host:key.last(1)}>not(- 1)', null, false],
			['{host:key.last(1)}andnot(- 1)', null, false],
			['{host:key.last(1)}ornot(- 1)', null, false],

			['{host:key.last(1)} * - 1', null, true],
			['{host:key.last(1)} / - 1', null, true],
			['{host:key.last(1)} + - 1', null, true],
			['{host:key.last(1)} - - 1', null, true],
			['{host:key.last(1)} = - 1', null, true],
			['{host:key.last(1)} <> - 1', null, true],
			['{host:key.last(1)} < - 1', null, true],
			['{host:key.last(1)} > - 1', null, true],
			['{host:key.last(1)} and - 1', null, true],
			['{host:key.last(1)} or - 1', null, true],

			['{host:key.last(1)} * not - 1', null, true],
			['{host:key.last(1)} / not - 1', null, true],
			['{host:key.last(1)} + not - 1', null, true],
			['{host:key.last(1)} - not - 1', null, true],
			['{host:key.last(1)} = not - 1', null, true],
			['{host:key.last(1)} <> not - 1', null, true],
			['{host:key.last(1)} < not - 1', null, true],
			['{host:key.last(1)} > not - 1', null, true],
			['{host:key.last(1)} and not - 1', null, true],
			['{host:key.last(1)} or not - 1', null, true],

			['{host:key.last(1)}* - 1', null, true],
			['{host:key.last(1)}/ - 1', null, true],
			['{host:key.last(1)}+ - 1', null, true],
			['{host:key.last(1)}- - 1', null, true],
			['{host:key.last(1)}= - 1', null, true],
			['{host:key.last(1)}<> - 1', null, true],
			['{host:key.last(1)}< - 1', null, true],
			['{host:key.last(1)}> - 1', null, true],
			['{host:key.last(1)}and - 1', null, false],
			['{host:key.last(1)}or - 1', null, false],

			['{host:key.last(1)}* not - 1', null, true],
			['{host:key.last(1)}/ not - 1', null, true],
			['{host:key.last(1)}+ not - 1', null, true],
			['{host:key.last(1)}- not - 1', null, true],
			['{host:key.last(1)}= not - 1', null, true],
			['{host:key.last(1)}<> not - 1', null, true],
			['{host:key.last(1)}< not - 1', null, true],
			['{host:key.last(1)}> not - 1', null, true],
			['{host:key.last(1)}and not - 1', null, false],
			['{host:key.last(1)}or not - 1', null, false],

			['{host:key.last(1)} *- 1', null, true],
			['{host:key.last(1)} /- 1', null, true],
			['{host:key.last(1)} +- 1', null, true],
			['{host:key.last(1)} -- 1', null, true],
			['{host:key.last(1)} =- 1', null, true],
			['{host:key.last(1)} <>- 1', null, true],
			['{host:key.last(1)} <- 1', null, true],
			['{host:key.last(1)} >- 1', null, true],
			['{host:key.last(1)} and- 1', null, false],
			['{host:key.last(1)} or- 1', null, false],

			['{host:key.last(1)} *not- 1', null, false],
			['{host:key.last(1)} /not- 1', null, false],
			['{host:key.last(1)} +not- 1', null, false],
			['{host:key.last(1)} -not- 1', null, false],
			['{host:key.last(1)} =not- 1', null, false],
			['{host:key.last(1)} <>not- 1', null, false],
			['{host:key.last(1)} <not- 1', null, false],
			['{host:key.last(1)} >not- 1', null, false],
			['{host:key.last(1)} andnot- 1', null, false],
			['{host:key.last(1)} ornot- 1', null, false],

			['{host:key.last(1)}*- 1', null, true],
			['{host:key.last(1)}/- 1', null, true],
			['{host:key.last(1)}+- 1', null, true],
			['{host:key.last(1)}-- 1', null, true],
			['{host:key.last(1)}=- 1', null, true],
			['{host:key.last(1)}<>- 1', null, true],
			['{host:key.last(1)}<- 1', null, true],
			['{host:key.last(1)}>- 1', null, true],
			['{host:key.last(1)}and- 1', null, false],
			['{host:key.last(1)}or- 1', null, false],

			['{host:key.last(1)}*not- 1', null, false],
			['{host:key.last(1)}/not- 1', null, false],
			['{host:key.last(1)}+not- 1', null, false],
			['{host:key.last(1)}-not- 1', null, false],
			['{host:key.last(1)}=not- 1', null, false],
			['{host:key.last(1)}<>not- 1', null, false],
			['{host:key.last(1)}<not- 1', null, false],
			['{host:key.last(1)}>not- 1', null, false],
			['{host:key.last(1)}andnot- 1', null, false],
			['{host:key.last(1)}ornot- 1', null, false],

			['{host:key.last(1)} * (+1)', null, false],
			['{host:key.last(1)} / (+1)', null, false],
			['{host:key.last(1)} + (+1)', null, false],
			['{host:key.last(1)} - (+1)', null, false],
			['{host:key.last(1)} = (+1)', null, false],
			['{host:key.last(1)} <> (+1)', null, false],
			['{host:key.last(1)} < (+1)', null, false],
			['{host:key.last(1)} > (+1)', null, false],
			['{host:key.last(1)} and (+1)', null, false],
			['{host:key.last(1)} or (+1)', null, false],

			['{host:key.last(1)} * not (+1)', null, false],
			['{host:key.last(1)} / not (+1)', null, false],
			['{host:key.last(1)} + not (+1)', null, false],
			['{host:key.last(1)} - not (+1)', null, false],
			['{host:key.last(1)} = not (+1)', null, false],
			['{host:key.last(1)} <> not (+1)', null, false],
			['{host:key.last(1)} < not (+1)', null, false],
			['{host:key.last(1)} > not (+1)', null, false],
			['{host:key.last(1)} and not (+1)', null, false],
			['{host:key.last(1)} or not (+1)', null, false],

			['{host:key.last(1)}* (+1)', null, false],
			['{host:key.last(1)}/ (+1)', null, false],
			['{host:key.last(1)}+ (+1)', null, false],
			['{host:key.last(1)}- (+1)', null, false],
			['{host:key.last(1)}= (+1)', null, false],
			['{host:key.last(1)}<> (+1)', null, false],
			['{host:key.last(1)}< (+1)', null, false],
			['{host:key.last(1)}> (+1)', null, false],
			['{host:key.last(1)}and (+1)', null, false],
			['{host:key.last(1)}or (+1)', null, false],

			['{host:key.last(1)}*not (+1)', null, false],
			['{host:key.last(1)}/not (+1)', null, false],
			['{host:key.last(1)}+not (+1)', null, false],
			['{host:key.last(1)}-not (+1)', null, false],
			['{host:key.last(1)}=not (+1)', null, false],
			['{host:key.last(1)}<>not (+1)', null, false],
			['{host:key.last(1)}<not (+1)', null, false],
			['{host:key.last(1)}>not (+1)', null, false],
			['{host:key.last(1)}andnot (+1)', null, false],
			['{host:key.last(1)}ornot (+1)', null, false],

			['{host:key.last(1)} *(+1)', null, false],
			['{host:key.last(1)} /(+1)', null, false],
			['{host:key.last(1)} +(+1)', null, false],
			['{host:key.last(1)} -(+1)', null, false],
			['{host:key.last(1)} =(+1)', null, false],
			['{host:key.last(1)} <>(+1)', null, false],
			['{host:key.last(1)} <(+1)', null, false],
			['{host:key.last(1)} >(+1)', null, false],
			['{host:key.last(1)} and(+1)', null, false],
			['{host:key.last(1)} or(+1)', null, false],

			['{host:key.last(1)} * not(+1)', null, false],
			['{host:key.last(1)} / not(+1)', null, false],
			['{host:key.last(1)} + not(+1)', null, false],
			['{host:key.last(1)} - not(+1)', null, false],
			['{host:key.last(1)} = not(+1)', null, false],
			['{host:key.last(1)} <> not(+1)', null, false],
			['{host:key.last(1)} < not(+1)', null, false],
			['{host:key.last(1)} > not(+1)', null, false],
			['{host:key.last(1)} and not(+1)', null, false],
			['{host:key.last(1)} or not(+1)', null, false],

			['{host:key.last(1)}*(+1)', null, false],
			['{host:key.last(1)}/(+1)', null, false],
			['{host:key.last(1)}+(+1)', null, false],
			['{host:key.last(1)}-(+1)', null, false],
			['{host:key.last(1)}=(+1)', null, false],
			['{host:key.last(1)}<>(+1)', null, false],
			['{host:key.last(1)}<(+1)', null, false],
			['{host:key.last(1)}>(+1)', null, false],
			['{host:key.last(1)}and(+1)', null, false],
			['{host:key.last(1)}or(+1)', null, false],

			['{host:key.last(1)}*not(+1)', null, false],
			['{host:key.last(1)}/not(+1)', null, false],
			['{host:key.last(1)}+not(+1)', null, false],
			['{host:key.last(1)}-not(+1)', null, false],
			['{host:key.last(1)}=not(+1)', null, false],
			['{host:key.last(1)}<>not(+1)', null, false],
			['{host:key.last(1)}<not(+1)', null, false],
			['{host:key.last(1)}>not(+1)', null, false],
			['{host:key.last(1)}andnot(+1)', null, false],
			['{host:key.last(1)}ornot(+1)', null, false],

			['{host:key.last(1)} * +1', null, false],
			['{host:key.last(1)} / +1', null, false],
			['{host:key.last(1)} + +1', null, false],
			['{host:key.last(1)} - +1', null, false],
			['{host:key.last(1)} = +1', null, false],
			['{host:key.last(1)} <> +1', null, false],
			['{host:key.last(1)} < +1', null, false],
			['{host:key.last(1)} > +1', null, false],
			['{host:key.last(1)} and +1', null, false],
			['{host:key.last(1)} or +1', null, false],

			['{host:key.last(1)} * not +1', null, false],
			['{host:key.last(1)} / not +1', null, false],
			['{host:key.last(1)} + not +1', null, false],
			['{host:key.last(1)} - not +1', null, false],
			['{host:key.last(1)} = not +1', null, false],
			['{host:key.last(1)} <> not +1', null, false],
			['{host:key.last(1)} < not +1', null, false],
			['{host:key.last(1)} > not +1', null, false],
			['{host:key.last(1)} and not +1', null, false],
			['{host:key.last(1)} or not +1', null, false],

			['{host:key.last(1)}* +1', null, false],
			['{host:key.last(1)}/ +1', null, false],
			['{host:key.last(1)}+ +1', null, false],
			['{host:key.last(1)}- +1', null, false],
			['{host:key.last(1)}= +1', null, false],
			['{host:key.last(1)}<> +1', null, false],
			['{host:key.last(1)}< +1', null, false],
			['{host:key.last(1)}> +1', null, false],
			['{host:key.last(1)}and +1', null, false],
			['{host:key.last(1)}or +1', null, false],

			['{host:key.last(1)}* not +1', null, false],
			['{host:key.last(1)}/ not +1', null, false],
			['{host:key.last(1)}+ not +1', null, false],
			['{host:key.last(1)}- not +1', null, false],
			['{host:key.last(1)}= not +1', null, false],
			['{host:key.last(1)}<> not +1', null, false],
			['{host:key.last(1)}< not +1', null, false],
			['{host:key.last(1)}> not +1', null, false],
			['{host:key.last(1)}and not +1', null, false],
			['{host:key.last(1)}or not +1', null, false],

			['{host:key.last(1)} *+1', null, false],
			['{host:key.last(1)} /+1', null, false],
			['{host:key.last(1)} ++1', null, false],
			['{host:key.last(1)} -+1', null, false],
			['{host:key.last(1)} =+1', null, false],
			['{host:key.last(1)} <>+1', null, false],
			['{host:key.last(1)} <+1', null, false],
			['{host:key.last(1)} >+1', null, false],
			['{host:key.last(1)} and+1', null, false],
			['{host:key.last(1)} or+1', null, false],

			['{host:key.last(1)} *not+1', null, false],
			['{host:key.last(1)} /not+1', null, false],
			['{host:key.last(1)} +not+1', null, false],
			['{host:key.last(1)} -not+1', null, false],
			['{host:key.last(1)} =not+1', null, false],
			['{host:key.last(1)} <>not+1', null, false],
			['{host:key.last(1)} <not+1', null, false],
			['{host:key.last(1)} >not+1', null, false],
			['{host:key.last(1)} andnot+1', null, false],
			['{host:key.last(1)} ornot+1', null, false],

			['{host:key.last(1)}*+1', null, false],
			['{host:key.last(1)}/+1', null, false],
			['{host:key.last(1)}++1', null, false],
			['{host:key.last(1)}-+1', null, false],
			['{host:key.last(1)}=+1', null, false],
			['{host:key.last(1)}<>+1', null, false],
			['{host:key.last(1)}<+1', null, false],
			['{host:key.last(1)}>+1', null, false],
			['{host:key.last(1)}and+1', null, false],
			['{host:key.last(1)}or+1', null, false],

			['{host:key.last(1)}*not+1', null, false],
			['{host:key.last(1)}/not+1', null, false],
			['{host:key.last(1)}+not+1', null, false],
			['{host:key.last(1)}-not+1', null, false],
			['{host:key.last(1)}=not+1', null, false],
			['{host:key.last(1)}<>not+1', null, false],
			['{host:key.last(1)}<not+1', null, false],
			['{host:key.last(1)}>not+1', null, false],
			['{host:key.last(1)}andnot+1', null, false],
			['{host:key.last(1)}ornot+1', null, false],

			['{host:key.last(1)} * (+ 1)', null, false],
			['{host:key.last(1)} / (+ 1)', null, false],
			['{host:key.last(1)} + (+ 1)', null, false],
			['{host:key.last(1)} - (+ 1)', null, false],
			['{host:key.last(1)} = (+ 1)', null, false],
			['{host:key.last(1)} <> (+ 1)', null, false],
			['{host:key.last(1)} < (+ 1)', null, false],
			['{host:key.last(1)} > (+ 1)', null, false],
			['{host:key.last(1)} and (+ 1)', null, false],
			['{host:key.last(1)} or (+ 1)', null, false],

			['{host:key.last(1)} * not (+ 1)', null, false],
			['{host:key.last(1)} / not (+ 1)', null, false],
			['{host:key.last(1)} + not (+ 1)', null, false],
			['{host:key.last(1)} - not (+ 1)', null, false],
			['{host:key.last(1)} = not (+ 1)', null, false],
			['{host:key.last(1)} <> not (+ 1)', null, false],
			['{host:key.last(1)} < not (+ 1)', null, false],
			['{host:key.last(1)} > not (+ 1)', null, false],
			['{host:key.last(1)} and not (+ 1)', null, false],
			['{host:key.last(1)} or not (+ 1)', null, false],

			['{host:key.last(1)}* (+ 1)', null, false],
			['{host:key.last(1)}/ (+ 1)', null, false],
			['{host:key.last(1)}+ (+ 1)', null, false],
			['{host:key.last(1)}- (+ 1)', null, false],
			['{host:key.last(1)}= (+ 1)', null, false],
			['{host:key.last(1)}<> (+ 1)', null, false],
			['{host:key.last(1)}< (+ 1)', null, false],
			['{host:key.last(1)}> (+ 1)', null, false],
			['{host:key.last(1)}and (+ 1)', null, false],
			['{host:key.last(1)}or (+ 1)', null, false],

			['{host:key.last(1)}* not (+ 1)', null, false],
			['{host:key.last(1)}/ not (+ 1)', null, false],
			['{host:key.last(1)}+ not (+ 1)', null, false],
			['{host:key.last(1)}- not (+ 1)', null, false],
			['{host:key.last(1)}= not (+ 1)', null, false],
			['{host:key.last(1)}<> not (+ 1)', null, false],
			['{host:key.last(1)}< not (+ 1)', null, false],
			['{host:key.last(1)}> not (+ 1)', null, false],
			['{host:key.last(1)}and not (+ 1)', null, false],
			['{host:key.last(1)}or not (+ 1)', null, false],

			['{host:key.last(1)} *(+ 1)', null, false],
			['{host:key.last(1)} /(+ 1)', null, false],
			['{host:key.last(1)} +(+ 1)', null, false],
			['{host:key.last(1)} -(+ 1)', null, false],
			['{host:key.last(1)} =(+ 1)', null, false],
			['{host:key.last(1)} <>(+ 1)', null, false],
			['{host:key.last(1)} <(+ 1)', null, false],
			['{host:key.last(1)} >(+ 1)', null, false],
			['{host:key.last(1)} and(+ 1)', null, false],
			['{host:key.last(1)} or(+ 1)', null, false],

			['{host:key.last(1)} *not(+ 1)', null, false],
			['{host:key.last(1)} /not(+ 1)', null, false],
			['{host:key.last(1)} +not(+ 1)', null, false],
			['{host:key.last(1)} -not(+ 1)', null, false],
			['{host:key.last(1)} =not(+ 1)', null, false],
			['{host:key.last(1)} <>not(+ 1)', null, false],
			['{host:key.last(1)} <not(+ 1)', null, false],
			['{host:key.last(1)} >not(+ 1)', null, false],
			['{host:key.last(1)} andnot(+ 1)', null, false],
			['{host:key.last(1)} ornot(+ 1)', null, false],

			['{host:key.last(1)}*(+ 1)', null, false],
			['{host:key.last(1)}/(+ 1)', null, false],
			['{host:key.last(1)}+(+ 1)', null, false],
			['{host:key.last(1)}-(+ 1)', null, false],
			['{host:key.last(1)}=(+ 1)', null, false],
			['{host:key.last(1)}<>(+ 1)', null, false],
			['{host:key.last(1)}<(+ 1)', null, false],
			['{host:key.last(1)}>(+ 1)', null, false],
			['{host:key.last(1)}and(+ 1)', null, false],
			['{host:key.last(1)}or(+ 1)', null, false],

			['{host:key.last(1)}*not(+ 1)', null, false],
			['{host:key.last(1)}/not(+ 1)', null, false],
			['{host:key.last(1)}+not(+ 1)', null, false],
			['{host:key.last(1)}-not(+ 1)', null, false],
			['{host:key.last(1)}=not(+ 1)', null, false],
			['{host:key.last(1)}<>not(+ 1)', null, false],
			['{host:key.last(1)}<not(+ 1)', null, false],
			['{host:key.last(1)}>not(+ 1)', null, false],
			['{host:key.last(1)}andnot(+ 1)', null, false],
			['{host:key.last(1)}ornot(+ 1)', null, false],

			['{host:key.last(1)} * + 1', null, false],
			['{host:key.last(1)} / + 1', null, false],
			['{host:key.last(1)} + + 1', null, false],
			['{host:key.last(1)} - + 1', null, false],
			['{host:key.last(1)} = + 1', null, false],
			['{host:key.last(1)} <> + 1', null, false],
			['{host:key.last(1)} < + 1', null, false],
			['{host:key.last(1)} > + 1', null, false],
			['{host:key.last(1)} and + 1', null, false],
			['{host:key.last(1)} or + 1', null, false],

			['{host:key.last(1)} * not + 1', null, false],
			['{host:key.last(1)} / not + 1', null, false],
			['{host:key.last(1)} + not + 1', null, false],
			['{host:key.last(1)} - not + 1', null, false],
			['{host:key.last(1)} = not + 1', null, false],
			['{host:key.last(1)} <> not + 1', null, false],
			['{host:key.last(1)} < not + 1', null, false],
			['{host:key.last(1)} > not + 1', null, false],
			['{host:key.last(1)} and not + 1', null, false],
			['{host:key.last(1)} or not + 1', null, false],

			['{host:key.last(1)}* + 1', null, false],
			['{host:key.last(1)}/ + 1', null, false],
			['{host:key.last(1)}+ + 1', null, false],
			['{host:key.last(1)}- + 1', null, false],
			['{host:key.last(1)}= + 1', null, false],
			['{host:key.last(1)}<> + 1', null, false],
			['{host:key.last(1)}< + 1', null, false],
			['{host:key.last(1)}> + 1', null, false],
			['{host:key.last(1)}and + 1', null, false],
			['{host:key.last(1)}or + 1', null, false],

			['{host:key.last(1)}* not + 1', null, false],
			['{host:key.last(1)}/ not + 1', null, false],
			['{host:key.last(1)}+ not + 1', null, false],
			['{host:key.last(1)}- not + 1', null, false],
			['{host:key.last(1)}= not + 1', null, false],
			['{host:key.last(1)}<> not + 1', null, false],
			['{host:key.last(1)}< not + 1', null, false],
			['{host:key.last(1)}> not + 1', null, false],
			['{host:key.last(1)}and not + 1', null, false],
			['{host:key.last(1)}or not + 1', null, false],

			['{host:key.last(1)} *+ 1', null, false],
			['{host:key.last(1)} /+ 1', null, false],
			['{host:key.last(1)} ++ 1', null, false],
			['{host:key.last(1)} -+ 1', null, false],
			['{host:key.last(1)} =+ 1', null, false],
			['{host:key.last(1)} <>+ 1', null, false],
			['{host:key.last(1)} <+ 1', null, false],
			['{host:key.last(1)} >+ 1', null, false],
			['{host:key.last(1)} and+ 1', null, false],
			['{host:key.last(1)} or+ 1', null, false],

			['{host:key.last(1)} * not+ 1', null, false],
			['{host:key.last(1)} / not+ 1', null, false],
			['{host:key.last(1)} + not+ 1', null, false],
			['{host:key.last(1)} - not+ 1', null, false],
			['{host:key.last(1)} = not+ 1', null, false],
			['{host:key.last(1)} <> not+ 1', null, false],
			['{host:key.last(1)} < not+ 1', null, false],
			['{host:key.last(1)} > not+ 1', null, false],
			['{host:key.last(1)} and not+ 1', null, false],
			['{host:key.last(1)} or not+ 1', null, false],

			['{host:key.last(1)}*+ 1', null, false],
			['{host:key.last(1)}/+ 1', null, false],
			['{host:key.last(1)}++ 1', null, false],
			['{host:key.last(1)}-+ 1', null, false],
			['{host:key.last(1)}=+ 1', null, false],
			['{host:key.last(1)}<>+ 1', null, false],
			['{host:key.last(1)}<+ 1', null, false],
			['{host:key.last(1)}>+ 1', null, false],
			['{host:key.last(1)}and+ 1', null, false],
			['{host:key.last(1)}or+ 1', null, false],

			['{host:key.last(1)}*not+ 1', null, false],
			['{host:key.last(1)}/not+ 1', null, false],
			['{host:key.last(1)}+not+ 1', null, false],
			['{host:key.last(1)}-not+ 1', null, false],
			['{host:key.last(1)}=not+ 1', null, false],
			['{host:key.last(1)}<>not+ 1', null, false],
			['{host:key.last(1)}<not+ 1', null, false],
			['{host:key.last(1)}>not+ 1', null, false],
			['{host:key.last(1)}andnot+ 1', null, false],
			['{host:key.last(1)}ornot+ 1', null, false],

			['{host:key.last(1)} * (not1)', null, false],
			['{host:key.last(1)} / (not1)', null, false],
			['{host:key.last(1)} + (not1)', null, false],
			['{host:key.last(1)} - (not1)', null, false],
			['{host:key.last(1)} = (not1)', null, false],
			['{host:key.last(1)} <> (not1)', null, false],
			['{host:key.last(1)} < (not1)', null, false],
			['{host:key.last(1)} > (not1)', null, false],
			['{host:key.last(1)} and (not1)', null, false],
			['{host:key.last(1)} or (not1)', null, false],

			['{host:key.last(1)} * not (not1)', null, false],
			['{host:key.last(1)} / not (not1)', null, false],
			['{host:key.last(1)} + not (not1)', null, false],
			['{host:key.last(1)} - not (not1)', null, false],
			['{host:key.last(1)} = not (not1)', null, false],
			['{host:key.last(1)} <> not (not1)', null, false],
			['{host:key.last(1)} < not (not1)', null, false],
			['{host:key.last(1)} > not (not1)', null, false],
			['{host:key.last(1)} and not (not1)', null, false],
			['{host:key.last(1)} or not (not1)', null, false],

			['{host:key.last(1)}* (not1)', null, false],
			['{host:key.last(1)}/ (not1)', null, false],
			['{host:key.last(1)}+ (not1)', null, false],
			['{host:key.last(1)}- (not1)', null, false],
			['{host:key.last(1)}= (not1)', null, false],
			['{host:key.last(1)}<> (not1)', null, false],
			['{host:key.last(1)}< (not1)', null, false],
			['{host:key.last(1)}> (not1)', null, false],
			['{host:key.last(1)}and (not1)', null, false],
			['{host:key.last(1)}or (not1)', null, false],

			['{host:key.last(1)}*not (not1)', null, false],
			['{host:key.last(1)}/not (not1)', null, false],
			['{host:key.last(1)}+not (not1)', null, false],
			['{host:key.last(1)}-not (not1)', null, false],
			['{host:key.last(1)}=not (not1)', null, false],
			['{host:key.last(1)}<>not (not1)', null, false],
			['{host:key.last(1)}<not (not1)', null, false],
			['{host:key.last(1)}>not (not1)', null, false],
			['{host:key.last(1)}andnot (not1)', null, false],
			['{host:key.last(1)}ornot (not1)', null, false],

			['{host:key.last(1)} *(not1)', null, false],
			['{host:key.last(1)} /(not1)', null, false],
			['{host:key.last(1)} +(not1)', null, false],
			['{host:key.last(1)} -(not1)', null, false],
			['{host:key.last(1)} =(not1)', null, false],
			['{host:key.last(1)} <>(not1)', null, false],
			['{host:key.last(1)} <(not1)', null, false],
			['{host:key.last(1)} >(not1)', null, false],
			['{host:key.last(1)} and(not1)', null, false],
			['{host:key.last(1)} or(not1)', null, false],

			['{host:key.last(1)} * not(not1)', null, false],
			['{host:key.last(1)} / not(not1)', null, false],
			['{host:key.last(1)} + not(not1)', null, false],
			['{host:key.last(1)} - not(not1)', null, false],
			['{host:key.last(1)} = not(not1)', null, false],
			['{host:key.last(1)} <> not(not1)', null, false],
			['{host:key.last(1)} < not(not1)', null, false],
			['{host:key.last(1)} > not(not1)', null, false],
			['{host:key.last(1)} and not(not1)', null, false],
			['{host:key.last(1)} or not(not1)', null, false],

			['{host:key.last(1)}*(not1)', null, false],
			['{host:key.last(1)}/(not1)', null, false],
			['{host:key.last(1)}+(not1)', null, false],
			['{host:key.last(1)}-(not1)', null, false],
			['{host:key.last(1)}=(not1)', null, false],
			['{host:key.last(1)}<>(not1)', null, false],
			['{host:key.last(1)}<(not1)', null, false],
			['{host:key.last(1)}>(not1)', null, false],
			['{host:key.last(1)}and(not1)', null, false],
			['{host:key.last(1)}or(not1)', null, false],

			['{host:key.last(1)}*not(not1)', null, false],
			['{host:key.last(1)}/not(not1)', null, false],
			['{host:key.last(1)}+not(not1)', null, false],
			['{host:key.last(1)}-not(not1)', null, false],
			['{host:key.last(1)}=not(not1)', null, false],
			['{host:key.last(1)}<>not(not1)', null, false],
			['{host:key.last(1)}<not(not1)', null, false],
			['{host:key.last(1)}>not(not1)', null, false],
			['{host:key.last(1)}andnot(not1)', null, false],
			['{host:key.last(1)}ornot(not1)', null, false],

			['{host:key.last(1)} * not1', null, false],
			['{host:key.last(1)} / not1', null, false],
			['{host:key.last(1)} + not1', null, false],
			['{host:key.last(1)} - not1', null, false],
			['{host:key.last(1)} = not1', null, false],
			['{host:key.last(1)} <> not1', null, false],
			['{host:key.last(1)} < not1', null, false],
			['{host:key.last(1)} > not1', null, false],
			['{host:key.last(1)} and not1', null, false],
			['{host:key.last(1)} or not1', null, false],

			['{host:key.last(1)} * not not1', null, false],
			['{host:key.last(1)} / not not1', null, false],
			['{host:key.last(1)} + not not1', null, false],
			['{host:key.last(1)} - not not1', null, false],
			['{host:key.last(1)} = not not1', null, false],
			['{host:key.last(1)} <> not not1', null, false],
			['{host:key.last(1)} < not not1', null, false],
			['{host:key.last(1)} > not not1', null, false],
			['{host:key.last(1)} and not not1', null, false],
			['{host:key.last(1)} or not not1', null, false],

			['{host:key.last(1)}* not1', null, false],
			['{host:key.last(1)}/ not1', null, false],
			['{host:key.last(1)}+ not1', null, false],
			['{host:key.last(1)}- not1', null, false],
			['{host:key.last(1)}= not1', null, false],
			['{host:key.last(1)}<> not1', null, false],
			['{host:key.last(1)}< not1', null, false],
			['{host:key.last(1)}> not1', null, false],
			['{host:key.last(1)}and not1', null, false],
			['{host:key.last(1)}or not1', null, false],

			['{host:key.last(1)}* not not1', null, false],
			['{host:key.last(1)}/ not not1', null, false],
			['{host:key.last(1)}+ not not1', null, false],
			['{host:key.last(1)}- not not1', null, false],
			['{host:key.last(1)}= not not1', null, false],
			['{host:key.last(1)}<> not not1', null, false],
			['{host:key.last(1)}< not not1', null, false],
			['{host:key.last(1)}> not not1', null, false],
			['{host:key.last(1)}and not not1', null, false],
			['{host:key.last(1)}or not not1', null, false],

			['{host:key.last(1)} *not1', null, false],
			['{host:key.last(1)} /not1', null, false],
			['{host:key.last(1)} +not1', null, false],
			['{host:key.last(1)} -not1', null, false],
			['{host:key.last(1)} =not1', null, false],
			['{host:key.last(1)} <>not1', null, false],
			['{host:key.last(1)} <not1', null, false],
			['{host:key.last(1)} >not1', null, false],
			['{host:key.last(1)} andnot1', null, false],
			['{host:key.last(1)} ornot1', null, false],

			['{host:key.last(1)} *notnot1', null, false],
			['{host:key.last(1)} /notnot1', null, false],
			['{host:key.last(1)} +notnot1', null, false],
			['{host:key.last(1)} -notnot1', null, false],
			['{host:key.last(1)} =notnot1', null, false],
			['{host:key.last(1)} <>notnot1', null, false],
			['{host:key.last(1)} <notnot1', null, false],
			['{host:key.last(1)} >notnot1', null, false],
			['{host:key.last(1)} andnotnot1', null, false],
			['{host:key.last(1)} ornotnot1', null, false],

			['{host:key.last(1)}*not1', null, false],
			['{host:key.last(1)}/not1', null, false],
			['{host:key.last(1)}+not1', null, false],
			['{host:key.last(1)}-not1', null, false],
			['{host:key.last(1)}=not1', null, false],
			['{host:key.last(1)}<>not1', null, false],
			['{host:key.last(1)}<not1', null, false],
			['{host:key.last(1)}>not1', null, false],
			['{host:key.last(1)}andnot1', null, false],
			['{host:key.last(1)}ornot1', null, false],

			['{host:key.last(1)}*notnot1', null, false],
			['{host:key.last(1)}/notnot1', null, false],
			['{host:key.last(1)}+notnot1', null, false],
			['{host:key.last(1)}-notnot1', null, false],
			['{host:key.last(1)}=notnot1', null, false],
			['{host:key.last(1)}<>notnot1', null, false],
			['{host:key.last(1)}<notnot1', null, false],
			['{host:key.last(1)}>notnot1', null, false],
			['{host:key.last(1)}andnotnot1', null, false],
			['{host:key.last(1)}ornotnot1', null, false],

			['{host:key.last(1)} * (not 1)', null, true],
			['{host:key.last(1)} / (not 1)', null, true],
			['{host:key.last(1)} + (not 1)', null, true],
			['{host:key.last(1)} - (not 1)', null, true],
			['{host:key.last(1)} = (not 1)', null, true],
			['{host:key.last(1)} <> (not 1)', null, true],
			['{host:key.last(1)} < (not 1)', null, true],
			['{host:key.last(1)} > (not 1)', null, true],
			['{host:key.last(1)} and (not 1)', null, true],
			['{host:key.last(1)} or (not 1)', null, true],

			['{host:key.last(1)} * not (not 1)', null, true],
			['{host:key.last(1)} / not (not 1)', null, true],
			['{host:key.last(1)} + not (not 1)', null, true],
			['{host:key.last(1)} - not (not 1)', null, true],
			['{host:key.last(1)} = not (not 1)', null, true],
			['{host:key.last(1)} <> not (not 1)', null, true],
			['{host:key.last(1)} < not (not 1)', null, true],
			['{host:key.last(1)} > not (not 1)', null, true],
			['{host:key.last(1)} and not (not 1)', null, true],
			['{host:key.last(1)} or not (not 1)', null, true],

			['{host:key.last(1)}* (not 1)', null, true],
			['{host:key.last(1)}/ (not 1)', null, true],
			['{host:key.last(1)}+ (not 1)', null, true],
			['{host:key.last(1)}- (not 1)', null, true],
			['{host:key.last(1)}= (not 1)', null, true],
			['{host:key.last(1)}<> (not 1)', null, true],
			['{host:key.last(1)}< (not 1)', null, true],
			['{host:key.last(1)}> (not 1)', null, true],
			['{host:key.last(1)}and (not 1)', null, false],
			['{host:key.last(1)}or (not 1)', null, false],

			['{host:key.last(1)}* not (not 1)', null, true],
			['{host:key.last(1)}/ not (not 1)', null, true],
			['{host:key.last(1)}+ not (not 1)', null, true],
			['{host:key.last(1)}- not (not 1)', null, true],
			['{host:key.last(1)}= not (not 1)', null, true],
			['{host:key.last(1)}<> not (not 1)', null, true],
			['{host:key.last(1)}< not (not 1)', null, true],
			['{host:key.last(1)}> not (not 1)', null, true],
			['{host:key.last(1)}and not (not 1)', null, false],
			['{host:key.last(1)}or not (not 1)', null, false],

			['{host:key.last(1)} *(not 1)', null, true],
			['{host:key.last(1)} /(not 1)', null, true],
			['{host:key.last(1)} +(not 1)', null, true],
			['{host:key.last(1)} -(not 1)', null, true],
			['{host:key.last(1)} =(not 1)', null, true],
			['{host:key.last(1)} <>(not 1)', null, true],
			['{host:key.last(1)} <(not 1)', null, true],
			['{host:key.last(1)} >(not 1)', null, true],
			['{host:key.last(1)} and(not 1)', null, true],
			['{host:key.last(1)} or(not 1)', null, true],

			['{host:key.last(1)} *not(not 1)', null, false],
			['{host:key.last(1)} /not(not 1)', null, false],
			['{host:key.last(1)} +not(not 1)', null, false],
			['{host:key.last(1)} -not(not 1)', null, false],
			['{host:key.last(1)} =not(not 1)', null, false],
			['{host:key.last(1)} <>not(not 1)', null, false],
			['{host:key.last(1)} <not(not 1)', null, false],
			['{host:key.last(1)} >not(not 1)', null, false],
			['{host:key.last(1)} andnot(not 1)', null, false],
			['{host:key.last(1)} ornot(not 1)', null, false],

			['{host:key.last(1)}*(not 1)', null, true],
			['{host:key.last(1)}/(not 1)', null, true],
			['{host:key.last(1)}+(not 1)', null, true],
			['{host:key.last(1)}-(not 1)', null, true],
			['{host:key.last(1)}=(not 1)', null, true],
			['{host:key.last(1)}<>(not 1)', null, true],
			['{host:key.last(1)}<(not 1)', null, true],
			['{host:key.last(1)}>(not 1)', null, true],
			['{host:key.last(1)}and(not 1)', null, false],
			['{host:key.last(1)}or(not 1)', null, false],

			['{host:key.last(1)}*not(not 1)', null, false],
			['{host:key.last(1)}/not(not 1)', null, false],
			['{host:key.last(1)}+not(not 1)', null, false],
			['{host:key.last(1)}-not(not 1)', null, false],
			['{host:key.last(1)}=not(not 1)', null, false],
			['{host:key.last(1)}<>not(not 1)', null, false],
			['{host:key.last(1)}<not(not 1)', null, false],
			['{host:key.last(1)}>not(not 1)', null, false],
			['{host:key.last(1)}andnot(not 1)', null, false],
			['{host:key.last(1)}ornot(not 1)', null, false],

			['{host:key.last(1)} * not 1', null, true],
			['{host:key.last(1)} / not 1', null, true],
			['{host:key.last(1)} + not 1', null, true],
			['{host:key.last(1)} - not 1', null, true],
			['{host:key.last(1)} = not 1', null, true],
			['{host:key.last(1)} <> not 1', null, true],
			['{host:key.last(1)} < not 1', null, true],
			['{host:key.last(1)} > not 1', null, true],
			['{host:key.last(1)} and not 1', null, true],
			['{host:key.last(1)} or not 1', null, true],

			['{host:key.last(1)} * not not 1', null, false],
			['{host:key.last(1)} / not not 1', null, false],
			['{host:key.last(1)} + not not 1', null, false],
			['{host:key.last(1)} - not not 1', null, false],
			['{host:key.last(1)} = not not 1', null, false],
			['{host:key.last(1)} <> not not 1', null, false],
			['{host:key.last(1)} < not not 1', null, false],
			['{host:key.last(1)} > not not 1', null, false],
			['{host:key.last(1)} and not not 1', null, false],
			['{host:key.last(1)} or not not 1', null, false],

			['{host:key.last(1)}* not 1', null, true],
			['{host:key.last(1)}/ not 1', null, true],
			['{host:key.last(1)}+ not 1', null, true],
			['{host:key.last(1)}- not 1', null, true],
			['{host:key.last(1)}= not 1', null, true],
			['{host:key.last(1)}<> not 1', null, true],
			['{host:key.last(1)}< not 1', null, true],
			['{host:key.last(1)}> not 1', null, true],
			['{host:key.last(1)}and not 1', null, false],
			['{host:key.last(1)}or not 1', null, false],

			['{host:key.last(1)}* not not 1', null, false],
			['{host:key.last(1)}/ not not 1', null, false],
			['{host:key.last(1)}+ not not 1', null, false],
			['{host:key.last(1)}- not not 1', null, false],
			['{host:key.last(1)}= not not 1', null, false],
			['{host:key.last(1)}<> not not 1', null, false],
			['{host:key.last(1)}< not not 1', null, false],
			['{host:key.last(1)}> not not 1', null, false],
			['{host:key.last(1)}and not not 1', null, false],
			['{host:key.last(1)}or not not 1', null, false],

			['{host:key.last(1)} *not 1', null, false],
			['{host:key.last(1)} /not 1', null, false],
			['{host:key.last(1)} +not 1', null, false],
			['{host:key.last(1)} -not 1', null, false],
			['{host:key.last(1)} =not 1', null, false],
			['{host:key.last(1)} <>not 1', null, false],
			['{host:key.last(1)} <not 1', null, false],
			['{host:key.last(1)} >not 1', null, false],
			['{host:key.last(1)} andnot 1', null, false],
			['{host:key.last(1)} ornot 1', null, false],

			['{host:key.last(1)} * notnot 1', null, false],
			['{host:key.last(1)} / notnot 1', null, false],
			['{host:key.last(1)} + notnot 1', null, false],
			['{host:key.last(1)} - notnot 1', null, false],
			['{host:key.last(1)} = notnot 1', null, false],
			['{host:key.last(1)} <> notnot 1', null, false],
			['{host:key.last(1)} < notnot 1', null, false],
			['{host:key.last(1)} > notnot 1', null, false],
			['{host:key.last(1)} and notnot 1', null, false],
			['{host:key.last(1)} or notnot 1', null, false],

			['{host:key.last(1)}*not 1', null, false],
			['{host:key.last(1)}/not 1', null, false],
			['{host:key.last(1)}+not 1', null, false],
			['{host:key.last(1)}-not 1', null, false],
			['{host:key.last(1)}=not 1', null, false],
			['{host:key.last(1)}<>not 1', null, false],
			['{host:key.last(1)}<not 1', null, false],
			['{host:key.last(1)}>not 1', null, false],
			['{host:key.last(1)}andnot 1', null, false],
			['{host:key.last(1)}ornot 1', null, false],

			['{host:key.last(1)}*notnot 1', null, false],
			['{host:key.last(1)}/notnot 1', null, false],
			['{host:key.last(1)}+notnot 1', null, false],
			['{host:key.last(1)}-notnot 1', null, false],
			['{host:key.last(1)}=notnot 1', null, false],
			['{host:key.last(1)}<>notnot 1', null, false],
			['{host:key.last(1)}<notnot 1', null, false],
			['{host:key.last(1)}>notnot 1', null, false],
			['{host:key.last(1)}andnotnot 1', null, false],
			['{host:key.last(1)}ornotnot 1', null, false],

			['{host:key.last(1)} *not 1', null, false],
			['{host:key.last(1)} /not 1', null, false],
			['{host:key.last(1)} +not 1', null, false],
			['{host:key.last(1)} -not 1', null, false],
			['{host:key.last(1)} =not 1', null, false],
			['{host:key.last(1)} <>not 1', null, false],
			['{host:key.last(1)} <not 1', null, false],
			['{host:key.last(1)} >not 1', null, false],
			['{host:key.last(1)} andnot 1', null, false],
			['{host:key.last(1)} ornot 1', null, false],

			['{host:key.diff()} + 1', null, true],
			['{host:key.diff()} - 1', null, true],
			['{host:key.diff()} / 1', null, true],
			['{host:key.diff()} * 1', null, true],
			['{host:key.diff()} = 1', null, true],
			['{host:key.diff()} <> 1', null, true],
			['{host:key.diff()} and 1', null, true],
			['{host:key.diff()} or 1', null, true],

			['{host:key.diff()} + not 1', null, true],
			['{host:key.diff()} - not 1', null, true],
			['{host:key.diff()} / not 1', null, true],
			['{host:key.diff()} * not 1', null, true],
			['{host:key.diff()} = not 1', null, true],
			['{host:key.diff()} <> not 1', null, true],
			['{host:key.diff()} and not 1', null, true],
			['{host:key.diff()} or not 1', null, true],

			['{host:key.diff()}+ 1', null, true],
			['{host:key.diff()}- 1', null, true],
			['{host:key.diff()}/ 1', null, true],
			['{host:key.diff()}* 1', null, true],
			['{host:key.diff()}= 1', null, true],
			['{host:key.diff()}<> 1', null, true],
			['{host:key.diff()}and 1', null, false],
			['{host:key.diff()}or 1', null, false],

			['{host:key.diff()}+ not 1', null, true],
			['{host:key.diff()}- not 1', null, true],
			['{host:key.diff()}/ not 1', null, true],
			['{host:key.diff()}* not 1', null, true],
			['{host:key.diff()}= not 1', null, true],
			['{host:key.diff()}<> not 1', null, true],
			['{host:key.diff()}and not 1', null, false],
			['{host:key.diff()}or not 1', null, false],

			['{host:key.diff()} +1', null, true],
			['{host:key.diff()} -1', null, true],
			['{host:key.diff()} /1', null, true],
			['{host:key.diff()} *1', null, true],
			['{host:key.diff()} =1', null, true],
			['{host:key.diff()} <>1', null, true],
			['{host:key.diff()} and1', null, false],
			['{host:key.diff()} or1', null, false],

			['{host:key.diff()} + not1', null, false],
			['{host:key.diff()} - not1', null, false],
			['{host:key.diff()} / not1', null, false],
			['{host:key.diff()} * not1', null, false],
			['{host:key.diff()} = not1', null, false],
			['{host:key.diff()} <> not1', null, false],
			['{host:key.diff()} and not1', null, false],
			['{host:key.diff()} or not1', null, false],

			['{host:key.diff()}+1', null, true],
			['{host:key.diff()}-1', null, true],
			['{host:key.diff()}/1', null, true],
			['{host:key.diff()}*1', null, true],
			['{host:key.diff()}=1', null, true],
			['{host:key.diff()}<>1', null, true],
			['{host:key.diff()}and1', null, false],
			['{host:key.diff()}or1', null, false],

			['{host:key.diff()}+not1', null, false],
			['{host:key.diff()}-not1', null, false],
			['{host:key.diff()}/not1', null, false],
			['{host:key.diff()}*not1', null, false],
			['{host:key.diff()}=not1', null, false],
			['{host:key.diff()}<>not1', null, false],
			['{host:key.diff()}andnot1', null, false],
			['{host:key.diff()}ornot1', null, false],

			// unary operators before binary operators
			['{host:key.diff()} -* 1', null, false],
			['{host:key.diff()} -- 1', null, true],
			['{host:key.diff()} -/ 1', null, false],
			['{host:key.diff()} -* 1', null, false],
			['{host:key.diff()} -= 1', null, false],
			['{host:key.diff()} -<> 1', null, false],
			['{host:key.diff()} -and 1', null, false],
			['{host:key.diff()} -or 1', null, false],

			['{host:key.diff()}-* 1', null, false],
			['{host:key.diff()}-- 1', null, true],
			['{host:key.diff()}-/ 1', null, false],
			['{host:key.diff()}-* 1', null, false],
			['{host:key.diff()}-= 1', null, false],
			['{host:key.diff()}-<> 1', null, false],
			['{host:key.diff()}-and 1', null, false],
			['{host:key.diff()}-or 1', null, false],

			['{host:key.diff()} -*1', null, false],
			['{host:key.diff()} --1', null, true],
			['{host:key.diff()} -/1', null, false],
			['{host:key.diff()} -*1', null, false],
			['{host:key.diff()} -=1', null, false],
			['{host:key.diff()} -<>1', null, false],
			['{host:key.diff()} -and1', null, false],
			['{host:key.diff()} -or1', null, false],

			['{host:key.diff()}-*1', null, false],
			['{host:key.diff()}--1', null, true],
			['{host:key.diff()}-/1', null, false],
			['{host:key.diff()}-*1', null, false],
			['{host:key.diff()}-=1', null, false],
			['{host:key.diff()}-<>1', null, false],
			['{host:key.diff()}-and1', null, false],
			['{host:key.diff()}-or1', null, false],

			['{host:key.diff()} not* 1', null, false],
			['{host:key.diff()} not- 1', null, false],
			['{host:key.diff()} not/ 1', null, false],
			['{host:key.diff()} not* 1', null, false],
			['{host:key.diff()} not= 1', null, false],
			['{host:key.diff()} not<> 1', null, false],
			['{host:key.diff()} notand 1', null, false],
			['{host:key.diff()} notor 1', null, false],

			['{host:key.diff()}not* 1', null, false],
			['{host:key.diff()}not- 1', null, false],
			['{host:key.diff()}not/ 1', null, false],
			['{host:key.diff()}not* 1', null, false],
			['{host:key.diff()}not= 1', null, false],
			['{host:key.diff()}not<> 1', null, false],
			['{host:key.diff()}notand 1', null, false],
			['{host:key.diff()}notor 1', null, false],

			['{host:key.diff()} not*1', null, false],
			['{host:key.diff()} not-1', null, false],
			['{host:key.diff()} not/1', null, false],
			['{host:key.diff()} not*1', null, false],
			['{host:key.diff()} not=1', null, false],
			['{host:key.diff()} not<>1', null, false],
			['{host:key.diff()} notand1', null, false],
			['{host:key.diff()} notor1', null, false],

			['{host:key.diff()}-*1', null, false],
			['{host:key.diff()}--1', null, true],
			['{host:key.diff()}-/1', null, false],
			['{host:key.diff()}-*1', null, false],
			['{host:key.diff()}-=1', null, false],
			['{host:key.diff()}-<>1', null, false],
			['{host:key.diff()}-and1', null, false],
			['{host:key.diff()}-or1', null, false],

			// suffixes
			['{host:key.diff()}=1K', null, true],
			['{host:key.diff()}=1M', null, true],
			['{host:key.diff()}=1G', null, true],
			['{host:key.diff()}=1T', null, true],
			['{host:key.diff()}=1s', null, true],
			['{host:key.diff()}=1m', null, true],
			['{host:key.diff()}=1h', null, true],
			['{host:key.diff()}=1d', null, true],
			['{host:key.diff()}=1w', null, true],

			['{host:key.diff()}=1.56K', null, true],
			['{host:key.diff()}=1.56M', null, true],
			['{host:key.diff()}=1.56G', null, true],
			['{host:key.diff()}=1.56T', null, true],
			['{host:key.diff()}=1.56s', null, true],
			['{host:key.diff()}=1.56m', null, true],
			['{host:key.diff()}=1.56h', null, true],
			['{host:key.diff()}=1.56d', null, true],
			['{host:key.diff()}=1.56w', null, true],

			// text operators after suffixes
			['{host:key.diff()}=1K and 1', null, true],
			['{host:key.diff()}=1M and 1', null, true],
			['{host:key.diff()}=1G and 1', null, true],
			['{host:key.diff()}=1T and 1', null, true],
			['{host:key.diff()}=1s and 1', null, true],
			['{host:key.diff()}=1m and 1', null, true],
			['{host:key.diff()}=1h and 1', null, true],
			['{host:key.diff()}=1d and 1', null, true],
			['{host:key.diff()}=1w and 1', null, true],

			['{host:key.diff()}=1Kand 1', null, false],
			['{host:key.diff()}=1Mand 1', null, false],
			['{host:key.diff()}=1Gand 1', null, false],
			['{host:key.diff()}=1Tand 1', null, false],
			['{host:key.diff()}=1sand 1', null, false],
			['{host:key.diff()}=1mand 1', null, false],
			['{host:key.diff()}=1hand 1', null, false],
			['{host:key.diff()}=1dand 1', null, false],
			['{host:key.diff()}=1wand 1', null, false],

			['{host:key.diff()}=1.56K and 1', null, true],
			['{host:key.diff()}=1.56M and 1', null, true],
			['{host:key.diff()}=1.56G and 1', null, true],
			['{host:key.diff()}=1.56T and 1', null, true],
			['{host:key.diff()}=1.56s and 1', null, true],
			['{host:key.diff()}=1.56m and 1', null, true],
			['{host:key.diff()}=1.56h and 1', null, true],
			['{host:key.diff()}=1.56d and 1', null, true],
			['{host:key.diff()}=1.56w and 1', null, true],

			['{host:key.diff()}=1.56Kand 1', null, false],
			['{host:key.diff()}=1.56Mand 1', null, false],
			['{host:key.diff()}=1.56Gand 1', null, false],
			['{host:key.diff()}=1.56Tand 1', null, false],
			['{host:key.diff()}=1.56sand 1', null, false],
			['{host:key.diff()}=1.56mand 1', null, false],
			['{host:key.diff()}=1.56hand 1', null, false],
			['{host:key.diff()}=1.56dand 1', null, false],
			['{host:key.diff()}=1.56wand 1', null, false],

			['{host:key.diff()}=1K or 1', null, true],
			['{host:key.diff()}=1M or 1', null, true],
			['{host:key.diff()}=1G or 1', null, true],
			['{host:key.diff()}=1T or 1', null, true],
			['{host:key.diff()}=1s or 1', null, true],
			['{host:key.diff()}=1m or 1', null, true],
			['{host:key.diff()}=1h or 1', null, true],
			['{host:key.diff()}=1d or 1', null, true],
			['{host:key.diff()}=1w or 1', null, true],

			['{host:key.diff()}=1Kor 1', null, false],
			['{host:key.diff()}=1Mor 1', null, false],
			['{host:key.diff()}=1Gor 1', null, false],
			['{host:key.diff()}=1Tor 1', null, false],
			['{host:key.diff()}=1sor 1', null, false],
			['{host:key.diff()}=1mor 1', null, false],
			['{host:key.diff()}=1hor 1', null, false],
			['{host:key.diff()}=1dor 1', null, false],
			['{host:key.diff()}=1wor 1', null, false],

			['{host:key.diff()}=1.56K or 1', null, true],
			['{host:key.diff()}=1.56M or 1', null, true],
			['{host:key.diff()}=1.56G or 1', null, true],
			['{host:key.diff()}=1.56T or 1', null, true],
			['{host:key.diff()}=1.56s or 1', null, true],
			['{host:key.diff()}=1.56m or 1', null, true],
			['{host:key.diff()}=1.56h or 1', null, true],
			['{host:key.diff()}=1.56d or 1', null, true],
			['{host:key.diff()}=1.56w or 1', null, true],

			['{host:key.diff()}=1.56Kor 1', null, false],
			['{host:key.diff()}=1.56Mor 1', null, false],
			['{host:key.diff()}=1.56Gor 1', null, false],
			['{host:key.diff()}=1.56Tor 1', null, false],
			['{host:key.diff()}=1.56sor 1', null, false],
			['{host:key.diff()}=1.56mor 1', null, false],
			['{host:key.diff()}=1.56hor 1', null, false],
			['{host:key.diff()}=1.56dor 1', null, false],
			['{host:key.diff()}=1.56wor 1', null, false],

			['{host:key.diff()} + 1.173640', null, true],
			['{host:key.diff()} - 1.173640', null, true],
			['{host:key.diff()} / 1.173640', null, true],
			['{host:key.diff()} * 1.173640', null, true],
			['{host:key.diff()} = 1.173640', null, true],
			['{host:key.diff()} <> 1.173640', null, true],
			['{host:key.diff()} and 1.173640', null, true],
			['{host:key.diff()} or 1.173640', null, true],

			['{host:key.diff()}+ 1.173640', null, true],
			['{host:key.diff()}- 1.173640', null, true],
			['{host:key.diff()}/ 1.173640', null, true],
			['{host:key.diff()}* 1.173640', null, true],
			['{host:key.diff()}= 1.173640', null, true],
			['{host:key.diff()}<> 1.173640', null, true],
			['{host:key.diff()}and 1.173640', null, false],
			['{host:key.diff()}or 1.173640', null, false],

			['{host:key.diff()} +1.173640', null, true],
			['{host:key.diff()} -1.173640', null, true],
			['{host:key.diff()} /1.173640', null, true],
			['{host:key.diff()} *1.173640', null, true],
			['{host:key.diff()} =1.173640', null, true],
			['{host:key.diff()} <>1.173640', null, true],
			['{host:key.diff()} and1.173640', null, false],
			['{host:key.diff()} or1.173640', null, false],

			['{host:key.diff()}+1.173640', null, true],
			['{host:key.diff()}-1.173640', null, true],
			['{host:key.diff()}/1.173640', null, true],
			['{host:key.diff()}*1.173640', null, true],
			['{host:key.diff()}=1.173640', null, true],
			['{host:key.diff()}<>1.173640', null, true],
			['{host:key.diff()}and1.173640', null, false],
			['{host:key.diff()}or1.173640', null, false],

			['{host:key.diff()} + 1 or {host:key.diff()}', null, true],
			['{host:key.diff()} - 1 and {host:key.diff()}', null, true],
			['{host:key.diff()} / 1 <> {host:key.diff()}', null, true],
			['{host:key.diff()} * 1 = {host:key.diff()}', null, true],
			['{host:key.diff()} = 1 * {host:key.diff()}', null, true],
			['{host:key.diff()} <> 1 / {host:key.diff()}', null, true],
			['{host:key.diff()} and 1 - {host:key.diff()}', null, true],
			['{host:key.diff()} or 1 + {host:key.diff()}', null, true],

			['{host:key.diff()} + not 1 or {host:key.diff()}', null, true],
			['{host:key.diff()} - not 1 and {host:key.diff()}', null, true],
			['{host:key.diff()} / not 1 <> {host:key.diff()}', null, true],
			['{host:key.diff()} * not 1 = {host:key.diff()}', null, true],
			['{host:key.diff()} = not 1 * {host:key.diff()}', null, true],
			['{host:key.diff()} <> not 1 / {host:key.diff()}', null, true],
			['{host:key.diff()} and not 1 - {host:key.diff()}', null, true],
			['{host:key.diff()} or not 1 + {host:key.diff()}', null, true],

			['{host:key.diff()} + 1 or not {host:key.diff()}', null, true],
			['{host:key.diff()} - 1 and not {host:key.diff()}', null, true],
			['{host:key.diff()} / 1 <> not {host:key.diff()}', null, true],
			['{host:key.diff()} * 1 = not {host:key.diff()}', null, true],
			['{host:key.diff()} = 1 * not {host:key.diff()}', null, true],
			['{host:key.diff()} <> 1 / not {host:key.diff()}', null, true],
			['{host:key.diff()} and 1 - not {host:key.diff()}', null, true],
			['{host:key.diff()} or 1 + not {host:key.diff()}', null, true],

			['{host:key.diff()} -- 1', null, true],
			['{host:key.diff()} ++ 1', null, false],
			['{host:key.diff()} // 1', null, false],
			['{host:key.diff()} ** 1', null, false],
			['{host:key.diff()} == 1', null, false],
			['{host:key.diff()} <><> 1', null, false],
			['{host:key.diff()} andand 1', null, false],
			['{host:key.diff()} oror 1', null, false],

			['{host:key.diff()} +', null, false],
			['{host:key.diff()} -', null, false],
			['{host:key.diff()} /', null, false],
			['{host:key.diff()} *', null, false],
			['{host:key.diff()} =', null, false],
			['{host:key.diff()} <>', null, false],
			['{host:key.diff()} and', null, false],
			['{host:key.diff()} or', null, false],
			['{host:key.diff()} not', null, false],

			['{host:key.diff()} + not', null, false],
			['{host:key.diff()} - not', null, false],
			['{host:key.diff()} / not', null, false],
			['{host:key.diff()} * not', null, false],
			['{host:key.diff()} = not', null, false],
			['{host:key.diff()} <> not', null, false],
			['{host:key.diff()} and not', null, false],
			['{host:key.diff()} or not', null, false],

			['- {host:key.diff()}', null, true],
			['+ {host:key.diff()}', null, false],
			['/ {host:key.diff()}', null, false],
			['* {host:key.diff()}', null, false],
			['= {host:key.diff()}', null, false],
			['<> {host:key.diff()}', null, false],
			['and {host:key.diff()}', null, false],
			['or {host:key.diff()}', null, false],

			['not - {host:key.diff()}', null, true],
			['not + {host:key.diff()}', null, false],
			['not / {host:key.diff()}', null, false],
			['not * {host:key.diff()}', null, false],
			['not = {host:key.diff()}', null, false],
			['not <> {host:key.diff()}', null, false],
			['not and {host:key.diff()}', null, false],
			['not or {host:key.diff()}', null, false],

			['- not {host:key.diff()}', null, false],
			['+ not {host:key.diff()}', null, false],
			['/ not {host:key.diff()}', null, false],
			['* not {host:key.diff()}', null, false],
			['= not {host:key.diff()}', null, false],
			['<> not {host:key.diff()}', null, false],
			['and not {host:key.diff()}', null, false],
			['or not {host:key.diff()}', null, false],

			['{host:key.diff()}=0', null, true],
			['{host:key.count(1,)}=0', null, true],
			['{host:key.count( 1,)}=0', null, true],
			['{host:key.count(  1,)}=0', null, true],
			['{host:key.count(1, )}=0', null, true],
			['{host:key.count(1,  )}=0', null, true],

			['{host:key.str(")}=0', null, false],
			['{host:key.str("")}=0', null, true],
			['{host:key.str(""")}=0', null, false],
			['{host:key.str("""")}=0', null, false],

			['{host:key.str( ")}=0', null, false],
			['{host:key.str( "")}=0', null, true],
			['{host:key.str( """)}=0', null, false],
			['{host:key.str( """")}=0', null, false],

			['{host:key.str(  ")}=0', null, false],
			['{host:key.str(  "")}=0', null, true],
			['{host:key.str(  """)}=0', null, false],
			['{host:key.str(  """")}=0', null, false],

			['{host:key.count(1,")}=0', null, false],
			['{host:key.count(1,"")}=0', null, true],
			['{host:key.count(1,""")}=0', null, false],
			['{host:key.count(1,"""")}=0', null, false],

			['{host:key.count(1, ")}=0', null, false],
			['{host:key.count(1, "")}=0', null, true],
			['{host:key.count(1, """)}=0', null, false],
			['{host:key.count(1, """")}=0', null, false],

			['{host:key.count(1,  ")}=0', null, false],
			['{host:key.count(1,  "")}=0', null, true],
			['{host:key.count(1,  """)}=0', null, false],
			['{host:key.count(1,  """")}=0', null, false],

			['{host:key.count(1,"",")}=0', null, false],
			['{host:key.count(1,"","")}=0', null, true],
			['{host:key.count(1,"",""")}=0', null, false],
			['{host:key.count(1,"","""")}=0', null, false],

			['{host:key.count(1,"", ")}=0', null, false],
			['{host:key.count(1,"", "")}=0', null, true],
			['{host:key.count(1,"", """)}=0', null, false],
			['{host:key.count(1,"", """")}=0', null, false],

			['{host:key.count(1,"",  ")}=0', null, false],
			['{host:key.count(1,"",  "")}=0', null, true],
			['{host:key.count(1,"",  """)}=0', null, false],
			['{host:key.count(1,"",  """")}=0', null, false],

			['{host:key.str("\")}=0', null, false],
			['{host:key.str("\"")}=0', null, true],
			['{host:key.str("\\\\"")}=0', null, true],
			['{host:key.str("\""")}=0', null, false],
			['{host:key.str("\"""")}=0', null, false],

			['{host:key.str(\")}=0', null, true],
			['{host:key.str(param\")}=0', null, true],
			['{host:key.str(param")}=0', null, true],

			['{host:key.str( \")}=0', null, true],
			['{host:key.str( param\")}=0', null, true],
			['{host:key.str( param")}=0', null, true],

			['{host:key.str(  \")}=0', null, true],
			['{host:key.str(  param\")}=0', null, true],
			['{host:key.str(  param")}=0', null, true],

			['{host:key.str(()}=0', null, true],
			['{host:key.str(param()}=0', null, true],

			['{host:key.str( ()}=0', null, true],
			['{host:key.str( param()}=0', null, true],

			['{host:key.str(  ()}=0', null, true],
			['{host:key.str(  param()}=0', null, true],

			['{host:key.str())}=0', null, false],
			['{host:key.str(param))}=0', null, false],

			['{host:key.str( ))}=0', null, false],
			['{host:key.str( param))}=0', null, false],

			['{host:key.str(  ))}=0', null, false],
			['{host:key.str(  param))}=0', null, false],

			['{host:key.str("(")}=0', null, true],
			['{host:key.str("param(")}=0', null, true],

			['{host:key.str(")")}=0', null, true],
			['{host:key.str("param)")}=0', null, true],

			['{host:key.str()}=0', null, true],
			['{host:key.str( )}=0', null, true],
			['{host:key.str(" ")}=0', null, true],
			['{host:key.str(abc)}=0', null, true],
			['{host:key.str(\'abc\')}=0', null, true],
			['{host:key.str("")}=0', null, true],
			['{host:key.last(0)}=0', null, true],
			['{host:key.str(aaa()}=0', null, true],

			['{host:key.last(0)}=0', null, true],
			['{host:key.str(aaa()}=0', null, true],
			['({hostA:keyA.str("abc")}=0) or ({hostB:keyB.last(123)}=0)', null, true],
			['{host:key[asd[].str(aaa()}=0', null, true],
			['{host:key["param].diff()"].diff()}=0', null, true],
			['{host:key[param].diff()].diff()}', null, false],
			['{host:key[asd[,asd[,[]].str(aaa()}=0', null, true],
			['{host:key[[],[],[]].str()}=0', null, true],
			['{host:key[].count(1,[],[])}=0', null, true],
			['({hostA:keyA.str("abc")}) / ({hostB:keyB.last(123)})=(0)', null, true],
			['({hostA:keyA.str("abc")}=0) or ({hostB:keyB.last(123)}=0)', null, true],
			['({hostA:keyA.str("abc")}=0) and ({hostB:keyB.last(123)}=0)', null, true],

			['{hostkey.last(0)}=0', null, false],
			['{host:keylast(0)}=0', null, false],
			['{host:key.last(0}=0', null, false],
			['{host:key.last(0)}=', null, false],
			['{host:key.str()=0', null, false],
			['{host:key.last()}<>0}', null, false],
			['{host:key.str()}<>0', null, true],
			['({host:key.str(aaa()}=0', null, false],
			['(({host:key.str(aaa()}=0)', null, false],
			['{host:key.str(aaa()}=0)', null, false],
			['({hostA:keyA.str("abc")}=0) oror ({hostB:keyB.last(123)}=0)', null, false],
			['{host:key.last()}<>0', null, true],
			['{hostA:keyA.str("abc")} / ({hostB:keyB.last(123)}=0', null, false],
			['({hostA:keyA.str("abc")} / ({hostB:keyB.last(123)})=0', null, false],

			['{constant}', null, false],
			['{cons tant}', null, false],
			['{expression}', null, false],
			['{expre ssion}', null, false],
			['host:key.str()', null, false],
			['{host:key.str()', null, false],
			['host:key.str()}', null, false],
			[' {host:key.str()}', null, true],
			['{host:key.str()} ', null, true],
			['{ host:key.str()}', null, false],
			['{host :key.str()}', null, false],
			['{host:key.str(-5)}', null, true],
			['{host:key.str(+5)}', null, true],
			['{host:key.str([-5)}', null, true],
			['{host:key.str(-5])}', null, true],
			['{host:key.str((-5)}', null, true],
			['{host:key.str(-5))}', null, false],
			['{host:key.str(-5)*1}', null, false],
			['0={host:key["a"b].str()}', null, false],
			['0={host:key[].str(a"b"c)}', null, true],
			['0={host:key[].str("a\"b\"c")}', null, true],
			['0={host:key[].str("a\\\\"b\\\\"c")}', null, true],
			['0={host:key[].str("a"b)}', null, false],
			['0={host:key[].str(,"a"b,)}', null, false],
			['0={host:key[].str("","a"b,"")}', null, false],
			['0={host:key.str)', null, false],
			['1z={host:key.str()', null, false],
			['1z={host:key.str()}', null, false],
			['0={host:key[].str(")}', null, false],
			['0={host:key[].str( ")}', null, false],
			['0={host:key[].str(")}")}', null, true],
			['0={host:key[].str( ")}")}', null, true],
			[
				'({host1:key1.last(0)}/{host2:key2.last(5)})/10+2*{TRIGGER.VALUE} and {$USERMACRO1}+(-{$USERMACRO2})+'.
					'-{$USERMACRO3}*-12K+12.5m',
				[
					'error' => '',
					'expressions' => [
						0 => [
							'expression' => '{host1:key1.last(0)}',
							'pos'=> 1,
							'host' => 'host1',
							'item' => 'key1',
							'function' => 'last(0)',
							'functionName' => 'last',
							'functionParam' => '0',
							'functionParamList' => ['0']
						],
						1 => [
							'expression' => '{host2:key2.last(5)}',
							'pos'=> 22,
							'host' => 'host2',
							'item' => 'key2',
							'function' => 'last(5)',
							'functionName' => 'last',
							'functionParam' => '5',
							'functionParamList' => ['5']
						]
					]
				],
				true
			],
			[
				'-12+{TRIGGER.VALUE}/(({host1:key1[].last(5)}+(5*(1-(-3*5 and ((7 or 9))<>1)*{host2:key2[""].last(123)})/10'.
					'/10/10)+{$USERMACRO1})-{$USERMACRO2})',
				null,
				true
			],
			['{host:key["{$USERMACRO1}",{$USERMACRO2}].str()}', null, true],
			['{host:key[].str("{$USERMACRO1}",{$USERMACRO2})}', null, true],
			['{host:key["{HOSTNAME1}",{HOSTNAME2}].str()}', null, true],
			[
				'{host:key[].str("{HOSTNAME1}",{HOSTNAME2})}',
				[
					'error' => '',
					'expressions' => [
						0 => [
							'expression' => '{host:key[].str("{HOSTNAME1}",{HOSTNAME2})}',
							'pos' => 0,
							'host' => 'host',
							'item' => 'key[]',
							'function' => 'str("{HOSTNAME1}",{HOSTNAME2})',
							'functionName' => 'str',
							'functionParam' => '"{HOSTNAME1}",{HOSTNAME2}',
							'functionParamList' => ['{HOSTNAME1}', '{HOSTNAME2}']
						]
					]
				],
				true
			],
			['{host:key[].count(1,"{HOSTNAME1}",{HOSTNAME2})}', null, true],
			['{host:key[].str()}=-1=--2=---3=----4=-----5', null, false],
			['{host:key[].str()}=-1=--2=---3=----4=-----5', null, false],
			['{host:key[].str()}=-1', null, true],
			['{host:key[].str()}+-1', null, true],
			['{host:key[].str()}--1', null, true],
			['{host:key[].str()}-(-1)', null, true],
			['{host:key[].str()}-(-(-(-1)))', null, true],
			['{host:key[{$$$},"{$$$}"].str()}', null, true],
			['{host:key[{!!!},"{!!!}"].str()}', null, true],
			['{host:key[{$USERMACRO1,"{$USERMACRO"].str()}', null, true],
			['{host:key[].count(1,{$USERMACRO1,"{$USERMACRO")}', null, true],
			['{host:key[{$USERMACRO1}abc,"{$USERMACRO2}"].str()}', null, true],
			['{host:key[{$USERMACRO1},"{$USERMACRO2}abc"].str()}', null, true],
			['{host:key[].count(1,{$USERMACRO1}abc,"{$USERMACRO2}")}', null, true],
			['{host:key[].count(1,{$USERMACRO1},"{$USERMACRO2}abc")}', null, true],
			['{host:key[abc{HOSTNAME1},"{HOSTNAME2}"].str()}', null, true],
			['{host:key[{HOSTNAME1},"abc{HOSTNAME2}"].str()}', null, true],
			['{host:key[].count(1,abc{HOSTNAME1},"{HOSTNAME2}")}', null, true],
			['{host:key[].count(1,{HOSTNAME1},"abc{HOSTNAME2}")}', null, true],
			['{host:key[{host:key.last(0)},"{host:key.last(0)}"].str()}', null, true],
			['{host:key[].count(1,{host:key.last(0)},"{host:key.last(0)}")}', null, false],
			['{host:key[].count(1,{host:key.last(0},"{host:key.last(0)}")}', null, true],
			['{host:key[].count(1,{host:key.last(0)}+{host:key.last(0)}', null, true],
			['{host:key[].count(1,{host:key.last(0)}', null, true],
			['{host:key[{host:key[].last(0)},"{host:key[].last(0)}"].str()}', null, false],
			['{host:key["{host:key[].last(0)}",{host:key[].last(0)}].str()}', null, false],
			['{host:key["{host:key[].last(0)}",{host:key[].last(0)}', null, true],
			['{host:key[{host:key[].last(0)}+{host:key[].last(0)}', null, true],
			['{host:key[{host:key[].last(0)}', null, true],
			['{host:key.last({$UPPERCASE})}', null, true],
			['{host:key.last(0)}+{$UPPERCASE}', null, true],
			['{host:key.last({$lowercase})}', null, true],
			['{host:key.last(0)}+{$lowercase}', null, false],
			['{host:key.last(1.23)}', null, true],
			['{host:key.last(1.23s)}', null, true],
			['{host:key.last(<>1.23)}', null, true],
			['{host:key.abschange()}', null, true],
			['{host:key.abschange(0)}', null, true],
			['{host:key.abschange(0,)}', null, true],
			['{host:key.avg()}', null, true],
			['{host:key.avg(123)}', null, true],
			['{host:key.avg(123s)}', null, true],
			['{host:key.avg(#123)}', null, true],
			['{host:key.avg(123,456)}', null, true],
			['{host:key.avg(123,456s)}', null, true],
			['{host:key.avg(123,456s,)}', null, true],
			['{host:key.change()}', null, true],
			['{host:key.change(0)}', null, true],
			['{host:key.change(0,)}', null, true],
			['{host:key.count()}', null, true],
			['{host:key.count(123)}', null, true],
			['{host:key.count(123,text)}', null, true],
			['{host:key.count(123s,text)}', null, true],
			['{host:key.count(#123,text)}', null, true],
			['{host:key.count(123,text,eq)}', null, true],
			['{host:key.count(123,text,ne)}', null, true],
			['{host:key.count(123,text,gt)}', null, true],
			['{host:key.count(123,text,ge)}', null, true],
			['{host:key.count(123,text,lt)}', null, true],
			['{host:key.count(123,text,le)}', null, true],
			['{host:key.count(123,text,like)}', null, true],
			['{host:key.count(123,text,like,456)}', null, true],
			['{host:key.count(123,text,like,456s)}', null, true],
			['{host:key.count(123,text,nonexistent,456s)}', null, true],
			['{host:key.count(123,text,nonexistent,456s,)}', null, true],
			['{host:key.date()}', null, true],
			['{host:key.date(0)}', null, true],
			['{host:key.date(0,)}', null, true],
			['{host:key.dayofweek()}', null, true],
			['{host:key.dayofweek(0)}', null, true],
			['{host:key.dayofweek(0,)}', null, true],
			['{host:key.delta()}', null, true],
			['{host:key.delta(123)}', null, true],
			['{host:key.delta(123s)}', null, true],
			['{host:key.delta(#123)}', null, true],
			['{host:key.delta(123,456)}', null, true],
			['{host:key.delta(123,456s)}', null, true],
			['{host:key.delta(123,456s,)}', null, true],
			['{host:key.diff()}', null, true],
			['{host:key.diff(0)}', null, true],
			['{host:key.diff(0,)}', null, true],
			['{host:key.fuzzytime()}', null, true],
			['{host:key.fuzzytime(123)}', null, true],
			['{host:key.fuzzytime(#123)}', null, true],
			['{host:key.fuzzytime(123,)}', null, true],
			['{host:key.iregexp()}', null, true],
			['{host:key.iregexp(text)}', null, true],
			['{host:key.iregexp(text,123)}', null, true],
			['{host:key.iregexp(text,123s)}', null, true],
			['{host:key.iregexp(text,#123)}', null, true],
			['{host:key.iregexp(text,#123,)}', null, true],
			['{host:key.last()}', null, true],
			['{host:key.last(0)}', null, true],
			['{host:key.last(#123)}', null, true],
			['{host:key.last(#123,456)}', null, true],
			['{host:key.last(#123,456s)}', null, true],
			['{host:key.last(#123,456s,)}', null, true],
			['{host:key.logseverity()}', null, true],
			['{host:key.logseverity(0)}', null, true],
			['{host:key.logseverity(0,)}', null, true],
			['{host:key.logsource()}', null, true],
			['{host:key.logsource(text)}', null, true],
			['{host:key.logsource(text,)}', null, true],
			['{host:key.max()}', null, true],
			[
				'{host:key.max(123)}',
				[
					'error' => '',
					'expressions' => [
						0 => [
							'expression' => '{host:key.max(123)}',
							'pos' => 0,
							'host' => 'host',
							'item' => 'key',
							'function' => 'max(123)',
							'functionName' => 'max',
							'functionParam' => '123',
							'functionParamList' => ['123']
						]
					]
				],
				true
			],
			['{host:key.max(123s)}', null, true],
			['{host:key.max(#123)}', null, true],
			['{host:key.max(123,456)}', null, true],
			['{host:key.max(123,456s)}', null, true],
			['{host:key.max(123,456s,)}', null, true],
			['{host:key.min()}', null, true],
			['{host:key.min(123)}', null, true],
			['{host:key.min(123s)}', null, true],
			['{host:key.min(#123)}', null, true],
			['{host:key.min(123,456)}', null, true],
			['{host:key.min(123,456s)}', null, true],
			['{host:key.min(123,456s,)}', null, true],
			['{host:key.nodata()}', null, true],
			['{host:key.nodata(123)}', null, true],
			['{host:key.nodata(123s)}', null, true],
			['{host:key.nodata(#123)}', null, true],
			['{host:key.nodata(123s,)}', null, true],
			['{host:key.now()}', null, true],
			['{host:key.now(0)}', null, true],
			['{host:key.now(0,)}', null, true],
			['{host:key.percentile(123,,12)}', null, true],
			['{host:key.percentile(123,123,12)}', null, true],
			['{host:key.percentile(#123,123,12)}', null, true],
			['{host:key.prev()}', null, true],
			['{host:key.prev(0)}', null, true],
			['{host:key.prev(0,)}', null, true],
			['{host:key.regexp()}', null, true],
			['{host:key.regexp(text)}', null, true],
			['{host:key.regexp(text,123)}', null, true],
			['{host:key.regexp(text,123s)}', null, true],
			['{host:key.regexp(text,#123)}', null, true],
			['{host:key.regexp(text,#123,)}', null, true],
			['{host:key.str()}', null, true],
			['{host:key.str(text)}', null, true],
			['{host:key.str(text,123)}', null, true],
			['{host:key.str(text,123s)}', null, true],
			['{host:key.str(text,#123)}', null, true],
			['{host:key.str(text,#123,)}', null, true],
			['{host:key.strlen()}', null, true],
			['{host:key.strlen(0)}', null, true],
			['{host:key.strlen(#123)}', null, true],
			['{host:key.strlen(#123,456)}', null, true],
			['{host:key.strlen(#123,456s)}', null, true],
			['{host:key.strlen(#123,456s,)}', null, true],
			['{host:key.sum()}', null, true],
			['{host:key.sum(123)}', null, true],
			['{host:key.sum(123s)}', null, true],
			['{host:key.sum(#123)}', null, true],
			['{host:key.sum(123,456)}', null, true],
			['{host:key.sum(123,456s)}', null, true],
			['{host:key.sum(123,456s,)}', null, true],
			['{host:key.time()}', null, true],
			['{host:key.time(0)}', null, true],
			['{host:key.time(0,)}', null, true],
			['{host:key.nonexistent()}', null, true],
			['{host:key.last(0)}+{TRIGGER.VALUE}', null, true],
			['{host:key.last(0)}+{trigger.value}', null, false],
			['(({host:key.last(0)}+1.)) + + (({host:key.last(0)}+1.))', null, false],
			['(({host:key.last(0)}+1.)) + - (({host:key.last(0)}+1.))', null, true],
			['(({host:key.last(0)}+1.)) + (({host:key.last(0)}+1.))', null, true],
			['(({host:key.last(0)}+1.))', null, true],
			['(--({host:key.last(0)}+1.))', null, false],
			['(+({host:key.last(0)}+1.))', null, false],
			['(/({host:key.last(0)}+1.))', null, false],
			['({host:key.last(0)}+1.)/({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.)-({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.)+({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.) or ({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.) and ({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.) and ({host:key.last(0)}+1.)/({host:key.last(0)}+1.)', null, true],
			['({host:key.last(0)}+1.)+({host:key.last(0)}+1.)/({host:key.last(0)}+1.)', null, true],
			[
				'1 - (1/({host:key.last(0)}+1.))+(({host:key.last(0)}+1.))/({host:key.last(0)}+1.)/({host:key.last('.
					'0)}+1.)',
				null,
				true
			],
			['(({host:key.last(0)}+.1)) + + (({host:key.last(0)}+.1))', null, false],
			['(({host:key.last(0)}+.1)) + - (({host:key.last(0)}+.1))', null, true],
			['(({host:key.last(0)}+.1)) + (({host:key.last(0)}+.1))', null, true],
			['(({host:key.last(0)}+.1))', null, true],
			['(--({host:key.last(0)}+.1))', null, false],
			['(+({host:key.last(0)}+.1))', null, false],
			['(/({host:key.last(0)}+.1))', null, false],
			['({host:key.last(0)}+.1)/({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1)-({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1)+({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1) or ({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1) and ({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1) and ({host:key.last(0)}+.1)/({host:key.last(0)}+.1)', null, true],
			['({host:key.last(0)}+.1)+({host:key.last(0)}+.1)/({host:key.last(0)}+.1)', null, true],
			[
				'1 - (1/({host:key.last(0)}+.1))+(({host:key.last(0)}+.1))/({host:key.last(0)}+.1)/({host:key.last('.
					'0)}+.1)',
				null,
				true
			],
			['(({host:key.last(0)}+0 .1)) + + (({host:key.last(0)}+0 .1))', null, false],
			['(({host:key.last(0)}+0 .1)) + - (({host:key.last(0)}+0 .1))', null, false],
			['(({host:key.last(0)}+0 .1)) + (({host:key.last(0)}+0 .1))', null, false],
			['(({host:key.last(0)}+0 .1))', null, false],
			['(--({host:key.last(0)}+0 .1))', null, false],
			['(+({host:key.last(0)}+0 .1))', null, false],
			['(/({host:key.last(0)}+0 .1))', null, false],
			['({host:key.last(0)}+0 .1)/({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1)-({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1)+({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1) or ({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1) and ({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1) and ({host:key.last(0)}+0 .1)/({host:key.last(0)}+0 .1)', null, false],
			['({host:key.last(0)}+0 .1)+({host:key.last(0)}+0 .1)/({host:key.last(0)}+0 .1)', null, false],
			[
				'1 - (1/({host:key.last(0)}+0 .1))+(({host:key.last(0)}+0 .1))/({host:key.last(0)}+0 .1)/({host:key'.
					'.last(0)}+0 .1)',
				null,
				false
			],
			['(({host:key.last(0)}+1 K)) + + (({host:key.last(0)}+1 K))', null, false],
			['(({host:key.last(0)}+1 K)) + - (({host:key.last(0)}+1 K))', null, false],
			['(({host:key.last(0)}+1 K)) + (({host:key.last(0)}+1 K))', null, false],
			['(({host:key.last(0)}+1 K))', null, false],
			['(--({host:key.last(0)}+1 K))', null, false],
			['(+({host:key.last(0)}+1 K))', null, false],
			['(/({host:key.last(0)}+1 K))', null, false],
			['({host:key.last(0)}+1 K)/({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K)-({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K)+({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K) or ({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K) and ({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K) and ({host:key.last(0)}+1 K)/({host:key.last(0)}+1 K)', null, false],
			['({host:key.last(0)}+1 K)+({host:key.last(0)}+1 K)/({host:key.last(0)}+1 K)', null, false],
			[
				'1 - (1/({host:key.last(0)}+1 K))+(({host:key.last(0)}+1 K))/({host:key.last(0)}+1 K)/({host:key.la'.
					'st(0)}+1 K)',
				null,
				false
			],
			['(({host:key.last(0)}+.)) + + (({host:key.last(0)}+.))', null, false],
			['(({host:key.last(0)}+.)) + - (({host:key.last(0)}+.))', null, false],
			['(({host:key.last(0)}+.)) + (({host:key.last(0)}+.))', null, false],
			['(({host:key.last(0)}+.))', null, false],
			['(--({host:key.last(0)}+.))', null, false],
			['(+({host:key.last(0)}+.))', null, false],
			['(/({host:key.last(0)}+.))', null, false],
			['({host:key.last(0)}+.)/({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.)-({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.)+({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.) or ({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.) and ({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.) and ({host:key.last(0)}+.)/({host:key.last(0)}+.)', null, false],
			['({host:key.last(0)}+.)+({host:key.last(0)}+.)/({host:key.last(0)}+.)', null, false],
			[
				'1 - (1/({host:key.last(0)}+.))+(({host:key.last(0)}+.))/({host:key.last(0)}+.)/({host:key.last(0)}'.
					'+.)',
				null,
				false
			],
			['(({host:key.last(0)}+.K)) + + (({host:key.last(0)}+.K))', null, false],
			['(({host:key.last(0)}+.K)) + - (({host:key.last(0)}+.K))', null, false],
			['(({host:key.last(0)}+.K)) + (({host:key.last(0)}+.K))', null, false],
			['(({host:key.last(0)}+.K))', null, false],
			['(--({host:key.last(0)}+.K))', null, false],
			['(+({host:key.last(0)}+.K))', null, false],
			['(/({host:key.last(0)}+.K))', null, false],
			['({host:key.last(0)}+.K)/({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K)-({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K)+({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K) or ({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K) and ({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K) and ({host:key.last(0)}+.K)/({host:key.last(0)}+.K)', null, false],
			['({host:key.last(0)}+.K)+({host:key.last(0)}+.K)/({host:key.last(0)}+.K)', null, false],
			[
				'1 - (1/({host:key.last(0)}+.K))+(({host:key.last(0)}+.K))/({host:key.last(0)}+.K)/({host:key.last('.
					'0)}+.K)',
				null,
				false
			],
			['(({host:key.last(0)}+K)) + + (({host:key.last(0)}+K))', null, false],
			['(({host:key.last(0)}+K)) + - (({host:key.last(0)}+K))', null, false],
			['(({host:key.last(0)}+K)) + (({host:key.last(0)}+K))', null, false],
			['(({host:key.last(0)}+K))', null, false],
			['(--({host:key.last(0)}+K))', null, false],
			['(+({host:key.last(0)}+K))', null, false],
			['(/({host:key.last(0)}+K))', null, false],
			['({host:key.last(0)}+K)/({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K)-({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K)+({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K) or ({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K) and ({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K) and ({host:key.last(0)}+K)/({host:key.last(0)}+K)', null, false],
			['({host:key.last(0)}+K)+({host:key.last(0)}+K)/({host:key.last(0)}+K)', null, false],
			[
				'1 - (1/({host:key.last(0)}+K))+(({host:key.last(0)}+K))/({host:key.last(0)}+K)/({host:key.last(0)}'.
					'+K)',
				null,
				false
			],
			[
				'({host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}) + + ({host:key.last(1)}+(1/2+2*2-3 or 4)'.
					' or 23-34>{host:key.last(#1)})',
				null,
				false
			],
			[
				'({host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}) + - ({host:key.last(1)}+(1/2+2*2-3 or '.
					'4) or 23-34>{host:key.last(#1)})',
				[
					'error' => '',
					'expressions' => [
						0 => [
							'expression' => '{host:key.last(1)}',
							'pos'=> 1,
							'host' => 'host',
							'item' => 'key',
							'function' => 'last(1)',
							'functionName' => 'last',
							'functionParam' => '1',
							'functionParamList' => ['1']
						],
						1 => [
							'expression' => '{host:key.last(#1)}',
							'pos'=> 46,
							'host' => 'host',
							'item' => 'key',
							'function' => 'last(#1)',
							'functionName' => 'last',
							'functionParam' => '#1',
							'functionParamList' => ['#1']
						],
						2 => [
							'expression' => '{host:key.last(1)}',
							'pos'=> 72,
							'host' => 'host',
							'item' => 'key',
							'function' => 'last(1)',
							'functionName' => 'last',
							'functionParam' => '1',
							'functionParamList' => ['1']
						],
						3 => [
							'expression' => '{host:key.last(#1)}',
							'pos'=> 117,
							'host' => 'host',
							'item' => 'key',
							'function' => 'last(#1)',
							'functionName' => 'last',
							'functionParam' => '#1',
							'functionParamList' => ['#1']
						]
					]
				],
				true
			],
			[
				'({host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}) + ({host:key.last(1)}+(1/2+2*2-3 or 4) or '.
					'23-34>{host:key.last(#1)})',
				null,
				true
			],
			['({host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)})', null, true],
			['(--{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)})', null, false],
			['(+{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)})', null, false],
			['(/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)})', null, false],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}-{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}+{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)} or {host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)} and {host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)} and {host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}',
				null,
				true
			],
			[
				'{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}+{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34'.
					'>{host:key.last(#1)}/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}',
				null,
				true
			],
			[
				'1 - (1/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)})+({host:key.last(1)}+(1/2+2*2-3 or '.
					'4) or 23-34>{host:key.last(#1)})/{host:key.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}/{host:ke'.
					'y.last(1)}+(1/2+2*2-3 or 4) or 23-34>{host:key.last(#1)}',
				null,
				true
			],
			['({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34) + + ({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, false],
			['({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34) + - ({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, true],
			['({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34) + ({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, true],
			['({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, true],
			['(--{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, false],
			['(+{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, false],
			['(/{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)', null, false],
			['{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34/{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34', null, true],
			['{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34-{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34', null, true],
			['{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34+{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34', null, true],
			['{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34 or {host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34', null, true],
			['{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34 and {host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34', null, true],
			[
				'{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34 and {host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34/{host:key.last(0)}+('.
					'1/2+2*2-3 or 4) or 23-34',
				null,
				true
			],
			[
				'{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34+{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34/{host:key.last(0)}+(1'.
					'/2+2*2-3 or 4) or 23-34',
				null,
				true
			],
			[
				'1 - (1/{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)+({host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34)/{host:key.'.
					'last(0)}+(1/2+2*2-3 or 4) or 23-34/{host:key.last(0)}+(1/2+2*2-3 or 4) or 23-34',
				null,
				true
			],
			['({host:key.{a}}) + + ({host:key.{a}})', null, false],
			['({host:key.{a}}) + - ({host:key.{a}})', null, false],
			['({host:key.{a}}) + ({host:key.{a}})', null, false],
			['({host:key.{a}})', null, false],
			['(--{host:key.{a}})', null, false],
			['(+{host:key.{a}})', null, false],
			['(/{host:key.{a}})', null, false],
			['{host:key.{a}}/{host:key.{a}}', null, false],
			['{host:key.{a}}-{host:key.{a}}', null, false],
			['{host:key.{a}}+{host:key.{a}}', null, false],
			['{host:key.{a}} or {host:key.{a}}', null, false],
			['{host:key.{a}} and {host:key.{a}}', null, false],
			['{host:key.{a}} and {host:key.{a}}/{host:key.{a}}', null, false],
			['{host:key.{a}}+{host:key.{a}}/{host:key.{a}}', null, false],
			['1 - (1/{host:key.{a}})+({host:key.{a}})/{host:key.{a}}/{host:key.{a}}', null, false],
			['({host::key.last{a}}) + + ({host::key.last{a}})', null, false],
			['({host::key.last{a}}) + - ({host::key.last{a}})', null, false],
			['({host::key.last{a}}) + ({host::key.last{a}})', null, false],
			['({host::key.last{a}})', null, false],
			['(--{host::key.last{a}})', null, false],
			['(+{host::key.last{a}})', null, false],
			['(/{host::key.last{a}})', null, false],
			['{host::key.last{a}}/{host::key.last{a}}', null, false],
			['{host::key.last{a}}-{host::key.last{a}}', null, false],
			['{host::key.last{a}}+{host::key.last{a}}', null, false],
			['{host::key.last{a}} or {host::key.last{a}}', null, false],
			['{host::key.last{a}} and {host::key.last{a}}', null, false],
			['{host::key.last{a}} and {host::key.last{a}}/{host::key.last{a}}', null, false],
			['{host::key.last{a}}+{host::key.last{a}}/{host::key.last{a}}', null, false],
			[
				'1 - (1/{host::key.last{a}})+({host::key.last{a}})/{host::key.last{a}}/{host::key.last{a}}',
				null,
				false
			],
			['({host::key.(0)}) + + ({host::key.(0)})', null, false],
			['({host::key.(0)}) + - ({host::key.(0)})', null, false],
			['({host::key.(0)}) + ({host::key.(0)})', null, false],
			['({host::key.(0)})', null, false],
			['(--{host::key.(0)})', null, false],
			['(+{host::key.(0)})', null, false],
			['(/{host::key.(0)})', null, false],
			['{host::key.(0)}/{host::key.(0)}', null, false],
			['{host::key.(0)}-{host::key.(0)}', null, false],
			['{host::key.(0)}+{host::key.(0)}', null, false],
			['{host::key.(0)} or {host::key.(0)}', null, false],
			['{host::key.(0)} and {host::key.(0)}', null, false],
			['{host::key.(0)} and {host::key.(0)}/{host::key.(0)}', null, false],
			['{host::key.(0)}+{host::key.(0)}/{host::key.(0)}', null, false],
			['1 - (1/{host::key.(0)})+({host::key.(0)})/{host::key.(0)}/{host::key.(0)}', null, false],
			['({:key.(0)}) + + ({:key.(0)})', null, false],
			['({:key.(0)}) + - ({:key.(0)})', null, false],
			['({:key.(0)}) + ({:key.(0)})', null, false],
			['({:key.(0)})', null, false],
			['(--{:key.(0)})', null, false],
			['(+{:key.(0)})', null, false],
			['(/{:key.(0)})', null, false],
			['{:key.(0)}/{:key.(0)}', null, false],
			['{:key.(0)}-{:key.(0)}', null, false],
			['{:key.(0)}+{:key.(0)}', null, false],
			['{:key.(0)} or {:key.(0)}', null, false],
			['{:key.(0)} and {:key.(0)}', null, false],
			['{:key.(0)} and {:key.(0)}/{:key.(0)}', null, false],
			['{:key.(0)}+{:key.(0)}/{:key.(0)}', null, false],
			['1 - (1/{:key.(0)})+({:key.(0)})/{:key.(0)}/{:key.(0)}', null, false],
			['({:key.last(0)}) + + ({:key.last(0)})', null, false],
			['({:key.last(0)}) + - ({:key.last(0)})', null, false],
			['({:key.last(0)}) + ({:key.last(0)})', null, false],
			['({:key.last(0)})', null, false],
			['(--{:key.last(0)})', null, false],
			['(+{:key.last(0)})', null, false],
			['(/{:key.last(0)})', null, false],
			['{:key.last(0)}/{:key.last(0)}', null, false],
			['{:key.last(0)}-{:key.last(0)}', null, false],
			['{:key.last(0)}+{:key.last(0)}', null, false],
			['{:key.last(0)} or {:key.last(0)}', null, false],
			['{:key.last(0)} and {:key.last(0)}', null, false],
			['{:key.last(0)} and {:key.last(0)}/{:key.last(0)}', null, false],
			['{:key.last(0)}+{:key.last(0)}/{:key.last(0)}', null, false],
			['1 - (1/{:key.last(0)})+({:key.last(0)})/{:key.last(0)}/{:key.last(0)}', null, false],
			['(({host:key.last(0)})) + + (({host:key.last(0)}))', null, false],
			['(({host:key.last(0)})) + - (({host:key.last(0)}))', null, true],
			['(({host:key.last(0)})) + (({host:key.last(0)}))', null, true],
			['(({host:key.last(0)}))', null, true],
			['(--({host:key.last(0)}))', null, false],
			['(+({host:key.last(0)}))', null, false],
			['(/({host:key.last(0)}))', null, false],
			['({host:key.last(0)})/({host:key.last(0)})', null, true],
			['({host:key.last(0)})-({host:key.last(0)})', null, true],
			['({host:key.last(0)})+({host:key.last(0)})', null, true],
			['({host:key.last(0)}) or ({host:key.last(0)})', null, true],
			['({host:key.last(0)}) and ({host:key.last(0)})', null, true],
			['({host:key.last(0)}) and ({host:key.last(0)})/({host:key.last(0)})', null, true],
			['({host:key.last(0)})+({host:key.last(0)})/({host:key.last(0)})', null, true],
			[
				'1 - (1/({host:key.last(0)}))+(({host:key.last(0)}))/({host:key.last(0)})/({host:key.last(0)})',
				null,
				true
			],
			['({host:.last(0)}) + + ({host:.last(0)})', null, false],
			['({host:.last(0)}) + - ({host:.last(0)})', null, false],
			['({host:.last(0)}) + ({host:.last(0)})', null, false],
			['({host:.last(0)})', null, false],
			['(--{host:.last(0)})', null, false],
			['(+{host:.last(0)})', null, false],
			['(/{host:.last(0)})', null, false],
			['{host:.last(0)}/{host:.last(0)}', null, false],
			['{host:.last(0)}-{host:.last(0)}', null, false],
			['{host:.last(0)}+{host:.last(0)}', null, false],
			['{host:.last(0)} or {host:.last(0)}', null, false],
			['{host:.last(0)} and {host:.last(0)}', null, false],
			['{host:.last(0)} and {host:.last(0)}/{host:.last(0)}', null, false],
			['{host:.last(0)}+{host:.last(0)}/{host:.last(0)}', null, false],
			['1 - (1/{host:.last(0)})+({host:.last(0)})/{host:.last(0)}/{host:.last(0)}', null, false],
			['({$}+{host:key.last(0)}) + + ({$}+{host:key.last(0)})', null, false],
			['({$}+{host:key.last(0)}) + - ({$}+{host:key.last(0)})', null, false],
			['({$}+{host:key.last(0)}) + ({$}+{host:key.last(0)})', null, false],
			['({$}+{host:key.last(0)})', null, false],
			['(--{$}+{host:key.last(0)})', null, false],
			['(+{$}+{host:key.last(0)})', null, false],
			['(/{$}+{host:key.last(0)})', null, false],
			['{$}+{host:key.last(0)}/{$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)}-{$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)}+{$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)} or {$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)} and {$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)} and {$}+{host:key.last(0)}/{$}+{host:key.last(0)}', null, false],
			['{$}+{host:key.last(0)}+{$}+{host:key.last(0)}/{$}+{host:key.last(0)}', null, false],
			[
				'1 - (1/{$}+{host:key.last(0)})+({$}+{host:key.last(0)})/{$}+{host:key.last(0)}/{$}+{host:key.last('.
					'0)}',
				null,
				false
			],
			['( - {$MACRO}+{host:key.last(0)}) + + ( - {$MACRO}+{host:key.last(0)})', null, false],
			['( - {$MACRO}+{host:key.last(0)}) + - ( - {$MACRO}+{host:key.last(0)})', null, true],
			['( - {$MACRO}+{host:key.last(0)}) + ( - {$MACRO}+{host:key.last(0)})', null, true],
			['( - {$MACRO}+{host:key.last(0)})', null, true],
			['(-- - {$MACRO}+{host:key.last(0)})', null, false],
			['(+ - {$MACRO}+{host:key.last(0)})', null, false],
			['(/ - {$MACRO}+{host:key.last(0)})', null, false],
			[' - {$MACRO}+{host:key.last(0)}/ - {$MACRO}+{host:key.last(0)}', null, true],
			[' - {$MACRO}+{host:key.last(0)}- - {$MACRO}+{host:key.last(0)}', null, true],
			[' - {$MACRO}+{host:key.last(0)}+ - {$MACRO}+{host:key.last(0)}', null, true],
			[' - {$MACRO}+{host:key.last(0)} or - {$MACRO}+{host:key.last(0)}', null, true],
			[' - {$MACRO}+{host:key.last(0)} and - {$MACRO}+{host:key.last(0)}', null, true],
			[
				' - {$MACRO}+{host:key.last(0)} and - {$MACRO}+{host:key.last(0)}/ - {$MACRO}+{host:key.last(0)}',
				null,
				true
			],
			[
				' - {$MACRO}+{host:key.last(0)}+ - {$MACRO}+{host:key.last(0)}/ - {$MACRO}+{host:key.last(0)}',
					null,
					true
				],
			[
				'1 - (1/ - {$MACRO}+{host:key.last(0)})+( - {$MACRO}+{host:key.last(0)})/ - {$MACRO}+{host:key.last'.
					'(0)}/ - {$MACRO}+{host:key.last(0)}',
				null,
				true
			],
			['(100G+{host:key.last(0)}) + + (100G+{host:key.last(0)})', null, false],
			['(100G+{host:key.last(0)}) + - (100G+{host:key.last(0)})', null, true],
			['(100G+{host:key.last(0)}) + (100G+{host:key.last(0)})', null, true],
			['(100G+{host:key.last(0)})', null, true],
			['(--100G+{host:key.last(0)})', null, false],
			['(+100G+{host:key.last(0)})', null, false],
			['(/100G+{host:key.last(0)})', null, false],
			['100G+{host:key.last(0)}/100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)}-100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)}+100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)} or 100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)} and 100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)} and 100G+{host:key.last(0)}/100G+{host:key.last(0)}', null, true],
			['100G+{host:key.last(0)}+100G+{host:key.last(0)}/100G+{host:key.last(0)}', null, true],
			[
				'1 - (1/100G+{host:key.last(0)})+(100G+{host:key.last(0)})/100G+{host:key.last(0)}/100G+{host:key.l'.
					'ast(0)}',
				null,
				true
			],
			['({host:key.last(0)}) + + ({host:key.last(0)})', null, false],
			['({host:key.last(0)}) + - ({host:key.last(0)})', null, true],
			['({host:key.last(0)}) + ({host:key.last(0)})', null, true],
			['({host:key.last(0)})', null, true],
			['(--{host:key.last(0)})', null, false],
			['(+{host:key.last(0)})', null, false],
			['(/{host:key.last(0)})', null, false],
			['{host:key.last(0)}/{host:key.last(0)}', null, true],
			['{host:key.last(0)}-{host:key.last(0)}', null, true],
			['{host:key.last(0)}+{host:key.last(0)}', null, true],
			['{host:key.last(0)} or {host:key.last(0)}', null, true],
			['{host:key.last(0)} and {host:key.last(0)}', null, true],
			['{host:key.last(0)} and {host:key.last(0)}/{host:key.last(0)}', null, true],
			['{host:key.last(0)}+{host:key.last(0)}/{host:key.last(0)}', null, true],
			['1 - (1/{host:key.last(0)})+({host:key.last(0)})/{host:key.last(0)}/{host:key.last(0)}', null, true],
			['({host:key.str(0 - 1 / 2 or-4*-56.34K)}) + + ({host:key.str(0 - 1 / 2 or-4*-56.34K)})', null, false],
			['({host:key.str(0 - 1 / 2 or -4*-56.34K)}) + - ({host:key.str(0 - 1 / 2 or -4*-56.34K)})', null, true],
			['({host:key.str(0 - 1 / 2 or -4*-56.34K)}) + ({host:key.str(0 - 1 / 2 or -4*-56.34K)})', null, true],
			['({host:key.str(0 - 1 / 2 or -4*-56.34K)})', null, true],
			['(--{host:key.str(0 - 1 / 2 or-4*-56.34K)})', null, false],
			['(+{host:key.str(0 - 1 / 2 or-4*-56.34K)})', null, false],
			['(/{host:key.str(0 - 1 / 2 or-4*-56.34K)})', null, false],
			['{host:key.str(0 - 1 / 2 or -4*-56.34K)}/{host:key.str(0 - 1 / 2 or -4*-56.34K)}', null, true],
			['{host:key.str(0 - 1 / 2 or -4*-56.34K)}-{host:key.str(0 - 1 / 2 or -4*-56.34K)}', null, true],
			['{host:key.str(0 - 1 / 2 or -4*-56.34K)}+{host:key.str(0 - 1 / 2 or -4*-56.34K)}', null, true],
			['{host:key.str(0 - 1 / 2 or -4*-56.34K)} or {host:key.str(0 - 1 / 2 or -4*-56.34K)}', null, true],
			['{host:key.str(0 - 1 / 2 or -4*-56.34K)} and {host:key.str(0 - 1 / 2 or -4*-56.34K)}', null, true],
			[
				'{host:key.str(0 - 1 / 2 or -4*-56.34K)} and {host:key.str(0 - 1 / 2 or -4*-56.34K)}/{host:key.str(0 - 1 / '.
					'2 or -4*-56.34K)}',
				null,
				true
			],
			[
				'{host:key.str(0 - 1 / 2 or -4*-56.34K)}+{host:key.str(0 - 1 / 2 or -4*-56.34K)}/{host:key.str(0 - 1 / '.
					'2 or -4*-56.34K)}',
				null,
				true
			],
			[
				'1 - (1/{host:key.str(0 - 1 / 2 or -4*-56.34K)})+({host:key.str(0 - 1 / 2 or -4*-56.34K)})/{host:key.st'.
					'r(0 - 1 / 2 or -4*-56.34K)}/{host:key.str(0 - 1 / 2 or -4*-56.34K)}',
				null,
				true
			],
			['({host:key.str(^$$$^%)}) + + ({host:key.str(^$$$^%)})', null, false],
			['({host:key.str(^$$$^%)}) + - ({host:key.str(^$$$^%)})', null, true],
			['({host:key.str(^$$$^%)}) + ({host:key.str(^$$$^%)})', null, true],
			['({host:key.str(^$$$^%)})', null, true],
			['(--{host:key.str(^$$$^%)})', null, false],
			['(+{host:key.str(^$$$^%)})', null, false],
			['(/{host:key.str(^$$$^%)})', null, false],
			['{host:key.str(^$$$^%)}/{host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)}-{host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)}+{host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)} or {host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)} and {host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)} and {host:key.str(^$$$^%)}/{host:key.str(^$$$^%)}', null, true],
			['{host:key.str(^$$$^%)}+{host:key.str(^$$$^%)}/{host:key.str(^$$$^%)}', null, true],
			[
				'1 - (1/{host:key.str(^$$$^%)})+({host:key.str(^$$$^%)})/{host:key.str(^$$$^%)}/{host:key.str(^$$$'.
					'^%)}',
				null,
				true
			],
			['({host:key.str("(^*&%#$%)*")}) + + ({host:key.str("(^*&%#$%)*")})', null, false],
			['({host:key.str("(^*&%#$%)*")}) + - ({host:key.str("(^*&%#$%)*")})', null, true],
			['({host:key.str("(^*&%#$%)*")}) + ({host:key.str("(^*&%#$%)*")})', null, true],
			['({host:key.str("(^*&%#$%)*")})', null, true],
			['(--{host:key.str("(^*&%#$%)*")})', null, false],
			['(+{host:key.str("(^*&%#$%)*")})', null, false],
			['(/{host:key.str("(^*&%#$%)*")})', null, false],
			['{host:key.str("(^*&%#$%)*")}/{host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*&%#$%)*")}-{host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*&%#$%)*")}+{host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*&%#$%)*")} or {host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*&%#$%)*")} and {host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*&%#$%)*")} and {host:key.str("(^*&%#$%)*")}/{host:key.str("(^*&%#$%)*")}', null, true],
			['{host:key.str("(^*%#$%)*")}+{host:key.str("(^*&%#$%)*")}/{host:key.str("(^*&%#$%)*")}', null, true],
			[
				'1 - (1/{host:key.str("(^*&%#$%)*")})+({host:key.str("(^*&%#$%)*")})/{host:key.str("(^*&%#$%)*")}/{'.
					'host:key.str("(^*&%#$%)*")}',
				null,
				true
			],
			['(((((((({host:key.str("")})))))))) + + (((((((({host:key.str("")}))))))))', null, false],
			['(((((((({host:key.str("")})))))))) + - (((((((({host:key.str("")}))))))))', null, true],
			['(((((((({host:key.str("")})))))))) + (((((((({host:key.str("")}))))))))', null, true],
			['(((((((({host:key.str("")}))))))))', null, true],
			['(--((((((({host:key.str("")}))))))))', null, false],
			['(+((((((({host:key.str("")}))))))))', null, false],
			['(/((((((({host:key.str("")}))))))))', null, false],
			['((((((({host:key.str("")})))))))/((((((({host:key.str("")})))))))', null, true],
			['((((((({host:key.str("")})))))))-((((((({host:key.str("")})))))))', null, true],
			['((((((({host:key.str("")})))))))+((((((({host:key.str("")})))))))', null, true],
			['((((((({host:key.str("")})))))))or((((((({host:key.str("")})))))))', null, true],
			['((((((({host:key.str("")})))))))and((((((({host:key.str("")})))))))', null, true],
			[
				'((((((({host:key.str("")})))))))and((((((({host:key.str("")})))))))/((((((({host:key.str("")})))))))',
				null,
				true
			],
			[
				'((((((({host:key.str("")})))))))+((((((({host:key.str("")})))))))/((((((({host:key.str("")})))))))',
				null,
				true
			],
			[
				'1 - (1/((((((({host:key.str("")}))))))))+(((((((({host:key.str("")}))))))))/((((((({host:key.str("'.
					'")})))))))/((((((({host:key.str("")})))))))',
				null,
				true
			],
			[
				'((1 - 1-2-((((((({host:key.str("0")}))))))))) + + ((1 - 1-2-((((((({host:key.str("0")})))))))))',
				null,
				false
			],
			[
				'((1 - 1-2-((((((({host:key.str("0")}))))))))) + - ((1 - 1-2-((((((({host:key.str("0")})))))))))',
				null,
				true
			],
			[
				'((1 - 1-2-((((((({host:key.str("0")}))))))))) + ((1 - 1-2-((((((({host:key.str("0")})))))))))',
				null,
				true
			],
			['((1 - 1-2-((((((({host:key.str("0")})))))))))', null, true],
			['(--(1 - 1-2-((((((({host:key.str("0")})))))))))', null, false],
			['(+(1 - 1-2-((((((({host:key.str("0")})))))))))', null, false],
			['(/(1 - 1-2-((((((({host:key.str("0")})))))))))', null, false],
			['(1 - 1-2-((((((({host:key.str("0")}))))))))/(1 - 1-2-((((((({host:key.str("0")}))))))))', null, true],
			['(1 - 1-2-((((((({host:key.str("0")}))))))))-(1 - 1-2-((((((({host:key.str("0")}))))))))', null, true],
			['(1 - 1-2-((((((({host:key.str("0")}))))))))+(1 - 1-2-((((((({host:key.str("0")}))))))))', null, true],
			['(1 - 1-2-((((((({host:key.str("0")}))))))))or(1 - 1-2-((((((({host:key.str("0")}))))))))', null, true],
			['(1 - 1-2-((((((({host:key.str("0")}))))))))and(1 - 1-2-((((((({host:key.str("0")}))))))))', null, true],
			[
				'(1 - 1-2-((((((({host:key.str("0")}))))))))and(1 - 1-2-((((((({host:key.str("0")}))))))))/'.
					'(1 - 1-2-((((((({host:key.str("0")}))))))))',
				null,
				true
			],
			[
				'(1 - 1-2-((((((({host:key.str("0")}))))))))+(1 - 1-2-((((((({host:key.str("0")}))))))))/'.
					'(1 - 1-2-((((((({host:key.str("0")}))))))))',
				null,
				true
			],
			[
				'1 - (1/(1 - 1-2-((((((({host:key.str("0")})))))))))+((1 - 1-2-((((((({host:key.str("0")})))))))))'.
					'/(1 - 1-2-((((((({host:key.str("0")}))))))))/(1 - 1-2-((((((({host:key.str("0")}))))))))',
				null,
				true
			],
			['{host:log[/data/logs/test.log,incorrect:FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect^FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect/FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect*FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect+FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect-FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect&FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect|FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,(incorrect|FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect|FAIL)].last(0)}', null, true],
			['{host:log[/data/logs/test.log,(incorrect|FAIL)].last(0)}', null, true],
			['{host:log[/data/logs/test.log,{incorrect|FAIL].last(0)}', null, true],
			['{host:log[/data/logs/test.log,incorrect|FAIL}].last(0)}', null, true],
			['{host:log[/data/logs/test.log,{incorrect|FAIL}].last(0)}', null, true],
			['{host:log[/data/logs/test.log,text1(incorrect|FAILtext2].last(0)}', null, true],
			['{host:log[/data/logs/test.log,text1incorrect|FAIL)text2].last(0)}', null, true],
			['{host:log[/data/logs/test.log,text1(incorrect|FAIL)text2].last(0)}', null, true],
			[
				'{Template_App_CCWS:web.page.regexp[0.0.0.0,/ws-callcontrol-1.1/test,{$CCWS_PORT},"[Ss]moke [Tt]est '.
					'= ([Ss]uccess|[Ww]arning|[Ff]ail).*([[:space:]].*)+"].count(#1,event service = failed)}=1',
				null,
				true
			],
			['{host:key.str({$M})} or {host:key.str({$M})} or {$M} + {TRIGGER.VALUE}', null, true],
			['{$M} or {host:key.str({$M})}', null, true],
			['({$M} + 5) or {host:key.str({$M})}', null, true],
			[
				'{hostA:keyA[1,2,3].str("abc",123)}*{hostB:keyB.last(123,"abc","def")}/{host:key["param","abc"].'.
					'last(1,2,3,4,5)}+{host:key.diff()}+{TRIGGER.VALUE}/{$M}-{$M1234}*{$CUSTOM}-{TRIGGER.VALUE}',
				[
					'error' => '',
					'expressions' => [
						0 => [
							'expression' => '{hostA:keyA[1,2,3].str("abc",123)}',
							'pos'=> 0,
							'host' => 'hostA',
							'item' => 'keyA[1,2,3]',
							'function' => 'str("abc",123)',
							'functionName' => 'str',
							'functionParam' => '"abc",123',
							'functionParamList' => ['abc', 123]
						],
						1 => [
							'expression' => '{hostB:keyB.last(123,"abc","def")}',
							'pos'=> 35,
							'host' => 'hostB',
							'item' => 'keyB',
							'function' => 'last(123,"abc","def")',
							'functionName' => 'last',
							'functionParam' => '123,"abc","def"',
							'functionParamList' => [123, 'abc', 'def']
						],
						2 => [
							'expression' => '{host:key["param","abc"].last(1,2,3,4,5)}',
							'pos'=> 70,
							'host' => 'host',
							'item' => 'key["param","abc"]',
							'function' => 'last(1,2,3,4,5)',
							'functionName' => 'last',
							'functionParam' => '1,2,3,4,5',
							'functionParamList' => [1, 2, 3, 4, 5]
						],
						3 => [
							'expression' => '{host:key.diff()}',
							'pos'=> 112,
							'host' => 'host',
							'item' => 'key',
							'function' => 'diff()',
							'functionName' => 'diff',
							'functionParam' => '',
							'functionParamList' => ['']
						]
					]
				],
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}+{$M}',
				null,
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}+{$M',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}+{$',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3or4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}+{',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}+',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE}',
				null,
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALUE',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VALU',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VAL',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.VA',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.V',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER.',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGER',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGGE',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIGG',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRIG',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TRI',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{TR',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{T',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+{',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)'.
					'+',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1)',
				null,
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-1',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(-',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-(',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+-',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)+',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5)',
				null,
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-5',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4-',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(4',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+(',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)+',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3)',
				null,
				true
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(3',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -(',
				null,
				false
			],
			[
				'{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and -',
				null,
				false
			],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m and', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m an', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m a', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m ', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9m', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=9', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8=', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>8', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<>', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7<', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<7', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6<', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>6', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5>', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and 5', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 and', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 an', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 a', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4 ', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or 4', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 or', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 o', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3 ', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+3', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2+', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-2', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1-', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/1', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0/', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*0', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}*', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )}', null, true],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" )', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"" ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\""', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\"', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4\\', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p4', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "p', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", "', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3", ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3",', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3"', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p3', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"p', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,"', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ,', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2 ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p2', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, p', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1, ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1,', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p1', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(p', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func(', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].func', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].fun', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].fu', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].f', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ].', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ]', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"" ', null, false],
			['{host:key[p1, p2 ,"p3", "p4\""', null, false],
			['{host:key[p1, p2 ,"p3", "p4\"', null, false],
			['{host:key[p1, p2 ,"p3", "p4\\', null, false],
			['{host:key[p1, p2 ,"p3", "p4', null, false],
			['{host:key[p1, p2 ,"p3", "p', null, false],
			['{host:key[p1, p2 ,"p3", "', null, false],
			['{host:key[p1, p2 ,"p3", ', null, false],
			['{host:key[p1, p2 ,"p3",', null, false],
			['{host:key[p1, p2 ,"p3"', null, false],
			['{host:key[p1, p2 ,"p3', null, false],
			['{host:key[p1, p2 ,"p', null, false],
			['{host:key[p1, p2 ,"', null, false],
			['{host:key[p1, p2 ,', null, false],
			['{host:key[p1, p2 ', null, false],
			['{host:key[p1, p2', null, false],
			['{host:key[p1, p', null, false],
			['{host:key[p1, ', null, false],
			['{host:key[p1,', null, false],
			['{host:key[p1', null, false],
			['{host:key[p', null, false],
			['{host:key[', null, false],
			['{host:key', null, false],
			['{host:ke', null, false],
			['{host:k', null, false],
			['{host:', null, false],
			['{host', null, false],
			['{hos', null, false],
			['{ho', null, false],
			['{h', null, false],
			['{', null, false],

			// new lines and tabs
			["\r{host:key.last(1)}+1", null, true],
			["\n{host:key.last(1)}+1", null, true],
			["\r\n{host:key.last(1)}+1", null, true],
			["\t{host:key.last(1)}+1", null, true],

			["{\rhost:key.last(1)}+1", null, false],
			["{\nhost:key.last(1)}+1", null, false],
			["{\r\nhost:key.last(1)}+1", null, false],
			["{\thost:key.last(1)}+1", null, false],

			["{host\r:key.last(1)}+1", null, false],
			["{host\n:key.last(1)}+1", null, false],
			["{host\r\n:key.last(1)}+1", null, false],
			["{host\t:key.last(1)}+1", null, false],

			["{host:\rkey.last(1)}+1", null, false],
			["{host:\nkey.last(1)}+1", null, false],
			["{host:\r\nkey.last(1)}+1", null, false],
			["{host:\tkey.last(1)}+1", null, false],

			["{host:key\r.last(1)}+1", null, false],
			["{host:key\n.last(1)}+1", null, false],
			["{host:key\r\n.last(1)}+1", null, false],
			["{host:key\t.last(1)}+1", null, false],

			["{host:key.\rlast(1)}+1", null, false],
			["{host:key.\nlast(1)}+1", null, false],
			["{host:key.\r\nlast(1)}+1", null, false],
			["{host:key.\tlast(1)}+1", null, false],

			["{host:key.last\r(1)}+1", null, false],
			["{host:key.last\n(1)}+1", null, false],
			["{host:key.last\r\n(1)}+1", null, false],
			["{host:key.last\t(1)}+1", null, false],

			["{host:key.last(1)\r}+1", null, false],
			["{host:key.last(1)\n}+1", null, false],
			["{host:key.last(1)\r\n}+1", null, false],
			["{host:key.last(1)\t}+1", null, false],

			["{host:key.last(1)}\r+1", null, true],
			["{host:key.last(1)}\n+1", null, true],
			["{host:key.last(1)}\r\n+1", null, true],
			["{host:key.last(1)}\t+1", null, true],

			["{host:key.last(1)}+\r1", null, true],
			["{host:key.last(1)}+\n1", null, true],
			["{host:key.last(1)}+\r\n1", null, true],
			["{host:key.last(1)}+\t1", null, true],

			["{host:key.last(1)}+1\r", null, true],
			["{host:key.last(1)}+1\n", null, true],
			["{host:key.last(1)}+1\r\n", null, true],
			["{host:key.last(1)}+1\t", null, true],

			["{host:key.last(1)}\r\r+\r\r1", null, true],
			["{host:key.last(1)}\n\n+\n\n1", null, true],
			["{host:key.last(1)}\r\n\r\n+\r\n\r\n1", null, true],
			["{host:key.last(1)}\t\t+\t\t1", null, true],

			["{host:key.last(1)}\r\t+\r\t1", null, true],
			["{host:key.last(1)}\n\t+\n\t1", null, true],
			["{host:key.last(1)}\r\n\t+\r\n\t1", null, true],
			["{host:key.last(1)}\t\t+\t\t1", null, true],

			["(\r{host:key.last(1)}+1) or 1", null, true],
			["(\n{host:key.last(1)}+1) or 1", null, true],
			["(\r\n{host:key.last(1)}+1) or 1", null, true],
			["(\t{host:key.last(1)}+1) or 1", null, true],

			["({host:key.last(1)}+1\r) or 1", null, true],
			["({host:key.last(1)}+1\n) or 1", null, true],
			["({host:key.last(1)}+1\r\n) or 1", null, true],
			["({host:key.last(1)}+1\t) or 1", null, true],
			["({host:key.last(1)}+1\t) or 1", null, true],

			["{host:key.last(1)}\ror not 1", null, true],
			["{host:key.last(1)}\nor not 1", null, true],
			["{host:key.last(1)}\r\nor not 1", null, true],
			["{host:key.last(1)}\tor not 1", null, true],

			["{host:key.last(1)} or\rnot 1", null, true],
			["{host:key.last(1)} or\nnot 1", null, true],
			["{host:key.last(1)} or\r\nnot 1", null, true],
			["{host:key.last(1)} or\tnot 1", null, true],

			["{host:key.last(1)}\rand not 1", null, true],
			["{host:key.last(1)}\nand not 1", null, true],
			["{host:key.last(1)}\r\nand not 1", null, true],
			["{host:key.last(1)}\tand not 1", null, true],

			["{host:key.last(1)} and\rnot 1", null, true],
			["{host:key.last(1)} and\nnot 1", null, true],
			["{host:key.last(1)} and\r\nnot 1", null, true],
			["{host:key.last(1)} and\tnot 1", null, true],

			["{host:key.last(1)} and not\r1", null, true],
			["{host:key.last(1)} and not\n1", null, true],
			["{host:key.last(1)} and not\r\n1", null, true],
			["{host:key.last(1)} and not\t1", null, true],

			["{host:key.count(1,\"\"\r)}=0", null, false],
			["{host:key.count(1,\"\"\n)}=0", null, false],
			["{host:key.count(1,\"\"\r\n)}=0", null, false],
			["{host:key.count(1,\"\"\t)}=0", null, false],

			["{host:key.count(1,\"\r\")}=0", null, true],
			["{host:key.count(1,\"\n\")}=0", null, true],
			["{host:key.count(1,\"\r\n\")}=0", null, true],
			["{host:key.count(1,\"\t\")}=0", null, true],

			["{host:key.count(1,\r\"\")}=0", null, true],
			["{host:key.count(1,\n\"\")}=0", null, true],
			["{host:key.count(1,\r\n\"\")}=0", null, true],
			["{host:key.count(1,\t\"\")}=0", null, true],

			["{host:key.count(1\r,\"\")}=0", null, true],
			["{host:key.count(1\n,\"\")}=0", null, true],
			["{host:key.count(1\r\n,\"\")}=0", null, true],
			["{host:key.count(1\t,\"\")}=0", null, true],

			["{host:key.count(\r1,\"\")}=0", null, true],
			["{host:key.count(\n1,\"\")}=0", null, true],
			["{host:key.count(\r\n1,\"\")}=0", null, true],
			["{host:key.count(\t1,\"\")}=0", null, true],

			// Simple trigger expressions.
			["{host:last()}=0", null, false, ['allow_func_only' => true]],
			["{item.last()}=0", null, false, ['allow_func_only' => true]],
			["{.last()}=0", null, false, ['allow_func_only' => true]],
			["{str(last())}=0", null, false, ['allow_func_only' => true]],

			["{last()}=0", null, true, ['allow_func_only' => true]],
			['{last()}={$USERMACRO: "host:item.macrocontext"}', null, true, ['allow_func_only' => true]],
			['{last()}={$USERMACRO: "{host:item.last()}"}', null, true, ['allow_func_only' => true]],
			['{str("last()")}=0', null, true, ['allow_func_only' => true]],
			['{str("host:item.last()")}=0', null, true, ['allow_func_only' => true]],
			['{str("{host:item.last()}")}=0', null, true, ['allow_func_only' => true]],

			["{last(#5)}={#LLDMACRO}", null, false],
			["{last(#5)}={#LLDMACRO}", null, false, ['lldmacros' => false]],
			["{last(#5)}={#LLDMACRO}", null, true, ['allow_func_only' => true]],

			// collapsed trigger expressions
			['{host:key.func()}', null, false, ['collapsed_expression' => true]],
			['{func()}', null, false, ['collapsed_expression' => true]],
			['{123}', null, true, ['collapsed_expression' => true]],
			['{123} = {$MACRO}', null, true, ['collapsed_expression' => true]],
			['{123} = {#MACRO}', null, true, ['collapsed_expression' => true]],
			['{123} = {#MACRO}', null, false, ['collapsed_expression' => true, 'lldmacros' => false]],

			// Compare strings.
			['{host:key.last()}=""', null, true],
			['{host:key.last()}=" "', null, true],
			['{host:key.last()}="\"abc\""', null, true],
			['{host:key.last()}="\"a\\bc\""', null, false],
			['{host:key.last()}= "\"abc" ', null, true],
			['{host:key.last()}="\\\"', null, true], // Actually looks like {host:key.last()}="\\"
			['{host:key.last()}="\\ \""', null, false],
			['{host:key.last()}=" "      ', null, true],
			['"abc"="abc"', null, true],
			['    "abc"  =   "abc"   ', null, true],
			['"abc"="abc"="abc"', null, true],
			['"abc"="abc" and "abc"', null, true],
			['{host:key.last()}="\ "', null, false],
			['{host:key.last()}="\\', null, false],
			['{host:key.last()}="\"', null, false],
			['{host:key.last()}=""abc', null, false],
			['{host:key.last()}=" "abc', null, false],
			['{host:key.last()}="abc\"', null, false],
			['{host:key.last()}="\""\"', null, false],
			['{host:key.last()}="\\ \"', null, false],
			['{host:key.last()}="\\\\\ "', null, false], // Actually looks like {host:key.last()}="\\\ "
			['{host:key.last()}=" " "', null, false],
			['{host:key.last()}="\n"', null, false],
			['"abc"="abc"and"abc"', null, false],
			['"abc"="abc" and abc"', null, false],

			['last(agent.ping) = 1', null, true, ['lldmacros' => false, 'calculated' => true]],
			['last(agent.ping) = 1 or {TRIGGER.VALUE}', null, false, ['lldmacros' => false, 'calculated' => true]],
			['last(host1:agent.ping) = 1 or last(host2:agent.ping) = 1', null, true, ['lldmacros' => false, 'calculated' => true]],
			['last(host1:agent.version) = "5.0.0" or last(host2:agent.ping) = 1', null, true, ['lldmacros' => false, 'calculated' => true]]
		];
	}

	/**
	 * @dataProvider provider
	 *
	 * @param string      $expression
	 * @param array|null  $result
	 * @param bool        $rc
	 * @param array       $options
	 * @param bool        $options['lldmacros']
	 * @param bool        $options['allow_func_only']
	 */
	public function testParseExpression(string $expression, ?array $result, bool $rc, array $options = []) {
		$expression_data = new C10TriggerExpression($options);

		if ($expression_data->parse($expression)) {
			$this->assertEquals($rc, true);
			$this->assertEquals($rc, $expression_data->isValid);

			if (isset($result)) {
				$this->assertEquals($result['error'], $expression_data->error);
				$this->assertEquals($result['expressions'], $expression_data->expressions);
			}
		}
		else {
			$this->assertEquals($rc, false, "\nError with expression $expression: ".$expression_data->error);
		}
	}

	public static function tokens_provider() {
		return [
			[
				'((-12 + {host:item.str(ГУГЛ)} or {$USERMACRO} and not {TRIGGER.VALUE} or {#LLD} or {{#LLD}.regsub("^([0-9]+)", "{#LLD}: \1")} or 10m))',
				[
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPEN_BRACE,
						'value' => '(',
						'data' => null,
						'pos' => 0,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPEN_BRACE,
						'value' => '(',
						'data' => null,
						'pos' => 1,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => '-',
						'data' => null,
						'pos' => 2,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_NUMBER,
						'value' => '12',
						'pos' => 3,
						'length' => 2,
						'data' => [
							'suffix' => null
						]
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => '+',
						'data' => null,
						'pos' => 6,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_FUNCTION_MACRO,
						'value' => '{host:item.str(ГУГЛ)}',
						'pos' => 8,
						'length' => 25,
						'data' => [
							'host' => 'host',
							'item' => 'item',
							'function' => 'str(ГУГЛ)',
							'functionName' => 'str',
							'functionParams' => ['ГУГЛ'],
							'functionParamsRaw' => [
								'type' => C10FunctionParser::PARAM_ARRAY,
								'raw' => '(ГУГЛ)',
								'pos' => 3,
								'parameters' => [
									[
										'type' => C10FunctionParser::PARAM_UNQUOTED,
										'raw' => 'ГУГЛ',
										'pos' => 1
									]
								]
							]
						]
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 34,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_USER_MACRO,
						'value' => '{$USERMACRO}',
						'data' => null,
						'pos' => 37,
						'length' => 12
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'and',
						'data' => null,
						'pos' => 50,
						'length' => 3
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'not',
						'data' => null,
						'pos' => 54,
						'length' => 3
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_MACRO,
						'value' => '{TRIGGER.VALUE}',
						'data' => null,
						'pos' => 58,
						'length' => 15
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 74,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_LLD_MACRO,
						'value' => '{#LLD}',
						'data' => null,
						'pos' => 77,
						'length' => 6
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 84,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_LLD_MACRO,
						'value' => '{{#LLD}.regsub("^([0-9]+)", "{#LLD}: \1")}',
						'data' => null,
						'pos' => 87,
						'length' => 42
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 130,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_NUMBER,
						'value' => '10m',
						'pos' => 133,
						'length' => 3,
						'data' => [
							'suffix' => 'm'
						]
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_CLOSE_BRACE,
						'value' => ')',
						'data' => null,
						'pos' => 136,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_CLOSE_BRACE,
						'value' => ')',
						'data' => null,
						'pos' => 137,
						'length' => 1
					]
				]
			],
			[
				'(-1.15w + str(host:item, "param2") or {$USERMACRO}) or {#LLD}',
				[
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPEN_BRACE,
						'value' => '(',
						'data' => null,
						'pos' => 0,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => '-',
						'data' => null,
						'pos' => 1,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_NUMBER,
						'value' => '1.15w',
						'pos' => 2,
						'length' => 5,
						'data' => [
							'suffix' => 'w'
						]
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => '+',
						'data' => null,
						'pos' => 8,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_FUNCTION,
						'value' => 'str(host:item, "param2")',
						'pos' => 10,
						'length' => 24,
						'data' => [
							'functionName' => 'str',
							'functionParams' => ['host:item', 'param2'],
							'functionParamsRaw' => [
								'type' => C10FunctionParser::PARAM_ARRAY,
								'raw' => '(host:item, "param2")',
								'pos' => 3,
								'parameters' => [
									[
										'type' => C10FunctionParser::PARAM_UNQUOTED,
										'raw' => 'host:item',
										'pos' => 1
									],
									[
										'type' => C10FunctionParser::PARAM_QUOTED,
										'raw' => '"param2"',
										'pos' => 12
									]
								]
							]
						]
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 35,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_USER_MACRO,
						'value' => '{$USERMACRO}',
						'data' => null,
						'pos' => 38,
						'length' => 12
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_CLOSE_BRACE,
						'value' => ')',
						'data' => null,
						'pos' => 50,
						'length' => 1
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_OPERATOR,
						'value' => 'or',
						'data' => null,
						'pos' => 52,
						'length' => 2
					],
					[
						'type' => C10TriggerExprParserResult::TOKEN_TYPE_LLD_MACRO,
						'value' => '{#LLD}',
						'data' => null,
						'pos' => 55,
						'length' => 6
					]
				],
				['calculated' => true]
			]
		];
	}

	/**
	 * @dataProvider tokens_provider
	 */
	public function testTokens(string $expression, array $tokens, array $options = []) {
		$parser = new C10TriggerExpression($options);
		$result = $parser->parse($expression);
		$this->assertTrue($result instanceof C10TriggerExprParserResult);
		$this->assertEquals($tokens, $result->getTokens());
	}
}
