// Copyright (c) Aptos
// SPDX-License-Identifier: Apache-2.0

use crate::validator_builder::ValidatorBuilder;
use aptos_config::config::NodeConfig;
use aptos_crypto::ed25519::Ed25519PrivateKey;
use aptos_secure_storage::{CryptoStorage, KVStorage, Storage};
use aptos_temppath::TempPath;
use rand::{rngs::StdRng, SeedableRng};

pub fn test_config() -> (NodeConfig, Ed25519PrivateKey) {
    let path = TempPath::new();
    path.create_as_dir().unwrap();
    let (root_keys, _genesis, _genesis_waypoint, validators) = ValidatorBuilder::new(
        path.path(),
        cached_framework_packages::module_blobs().to_vec(),
    )
    .template(NodeConfig::default_for_validator())
    .build(StdRng::from_seed([0; 32]))
    .unwrap();
    let mut configs = validators.into_iter().map(|v| v.config).collect::<Vec<_>>();
    let key = root_keys.root_key;

    let mut config = configs.swap_remove(0);
    config.set_data_dir(path.path().to_path_buf());
    let backend = &config
        .validator_network
        .as_ref()
        .unwrap()
        .identity_from_storage()
        .backend;
    let storage: Storage = std::convert::TryFrom::try_from(backend).unwrap();
    let mut test = aptos_config::config::TestConfig::new_with_temp_dir(Some(path));
    test.execution_key(
        storage
            .export_private_key(aptos_global_constants::EXECUTION_KEY)
            .unwrap(),
    );
    test.operator_key(
        storage
            .export_private_key(aptos_global_constants::OPERATOR_KEY)
            .unwrap(),
    );
    test.owner_key(
        storage
            .export_private_key(aptos_global_constants::OWNER_KEY)
            .unwrap(),
    );
    config.test = Some(test);

    let owner_account = storage
        .get(aptos_global_constants::OWNER_ACCOUNT)
        .unwrap()
        .value;
    let mut sr_test = aptos_config::config::SafetyRulesTestConfig::new(owner_account);
    sr_test.consensus_key(
        storage
            .export_private_key(aptos_global_constants::CONSENSUS_KEY)
            .unwrap(),
    );
    sr_test.execution_key(
        storage
            .export_private_key(aptos_global_constants::EXECUTION_KEY)
            .unwrap(),
    );
    config.consensus.safety_rules.test = Some(sr_test);

    (config, key)
}
