---
id: transaction-builder-generator
title: Transaction Builder Generator
custom_edit_url: https://github.com/aptos-labs/aptos-core/edit/main/language/transaction-builder-generator/README.md
---

# Transaction Builder Generator

A *transaction builder* is a helper function that converts its arguments into the payload of an Aptos transaction calling a particular Move script.

In Rust, the signature of such a function typically looks like this:
```rust
pub fn encode_peer_to_peer_with_metadata_script(
    token: TypeTag,
    payee: AccountAddress,
    amount: u64,
    metadata: Vec<u8>,
    metadata_signature: Vec<u8>,
) -> Script;
```

This crate provide a binary tool `generate-transaction-builders` to generate and install transaction builders in several programming languages.

The tool will also generate and install type definitions for Aptos types such as `TypeTag`, `AccountAddress`, and `Script`.

In practice, hashing and signing Aptos transactions additionally requires a runtime library for Binary Canonical Serialization ("BCS").
Such a library will be installed together with the Aptos types.


## Supported Languages

The following languages are currently supported:

* Python 3

* C++ 17

* Java 8

* Go >= 1.13

* Rust (NOTE: Code generation of dependency-free Rust is experimental. Consider using the libraries of the Aptos repository instead.)

* TypeScript / JavaScript

* Swift (version 5.3)


## Quick Start

From the root of the Aptos repository, run `cargo build -p transaction-builder-generator`.

You may browse command line options with `target/debug/generate-transaction-builders --help`.

### Python

To install Python3 modules `serde`, `bcs`, `aptos_types`, and `aptos_framework` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language python3 \
    --module-name aptos_framework \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    --with-custom-aptos-code aptos-move/transaction-builder-generator/examples/python3/custom_aptos_code/*.py \
    -- \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [Python demo file](examples/python3/stdlib_demo.py) with:
```bash
cp aptos-move/transaction-builder-generator/examples/python3/stdlib_demo.py "$DEST"
PYTHONPATH="$PYTHONPATH:$DEST" python3 "$DEST/stdlib_demo.py"
```

### C++

To install C++ files `serde.hpp`, `bcs.hpp`, `aptos_types.hpp`, `aptos_framework.hpp`, `aptos_framework.cpp` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language cpp \
    --module-name aptos_framework \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [C++ demo file](examples/cpp/stdlib_demo.cpp) with:
```bash
cp aptos-move/transaction-builder-generator/examples/cpp/stdlib_demo.cpp "$DEST"
clang++ --std=c++17 -I "$DEST" "$DEST/aptos_framework.cpp" "$DEST/stdlib_demo.cpp" -o "$DEST/stdlib_demo"
"$DEST/stdlib_demo"
```

### Java

To install Java source packages `com.novi.serde`, `com.novi.bcs`, `com.aptos.types`, and `com.aptos.stdlib` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language java \
    --module-name com.aptos.stdlib \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    --with-custom-aptos-code aptos-move/transaction-builder-generator/examples/java/custom_aptos_code/*.java \
    -- \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [Java demo file](examples/java/StdlibDemo.java) with:
```bash
cp aptos-move/transaction-builder-generator/examples/java/StdlibDemo.java "$DEST"
(find "$DEST" -name "*.java" | xargs javac -cp "$DEST")
java -enableassertions -cp "$DEST" StdlibDemo
```

### Go

To generate the Go "packages" `testing/aptostypes`, and `testing/aptosstdlib` into a target directory `$DEST`, run:

```bash
target/debug/generate-transaction-builders \
    --language go \
    --module-name aptosstdlib \
    --aptos-package-name testing \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [Go demo file](examples/golang/stdlib_demo.go) as follows:
(Note that `$DEST` must be an absolute path)
```bash
cp aptos-move/transaction-builder-generator/examples/golang/stdlib_demo.go "$DEST"
(cd "$DEST" && go mod init testing && go mod edit -replace testing="$DEST" && go run stdlib_demo.go)
```

### Rust (experimental)

To install dependency-free Rust crates `aptos-types` and `aptos-framework` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language rust \
    --module-name framework \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [Rust demo file](examples/rust/stdlib_demo.rs). (See [unit test](tests/generation.rs) for details.)

### TypeScript/JavaScript

To generate the TypeScript "module" `aptosStdlib` and its submodules into a target directory `$DEST`, run:

```bash
target/debug/generate-transaction-builders \
    --language typescript \
    --module-name aptosStdlib \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```

### C#

To install C# source `Serde`, `Bcs`, `Aptos.Types`, and `Aptos.Stdlib` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language csharp \
    --module-name Aptos.Stdlib \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    --with-custom-aptos-code aptos-move/transaction-builder-generator/examples/csharp/custom_aptos_code/*.cs \
    -- \
    "aptos-move/framework/aptos-framework/releases/artifacts/current"
```
Next, you may copy and execute the [C# demo file](examples/csharp/StdlibDemo.cs) with:
```bash
mkdir "$DEST"/Demo
cp aptos-move/transaction-builder-generator/examples/csharp/StdlibDemo.cs "$DEST/Demo"
cd "$DEST/Aptos/Stdlib"
dotnet new classlib -n Aptos.Stdlib -o .
dotnet add Aptos.Stdlib.csproj reference ../Types/Aptos.Types.csproj
rm Class1.cs
cd "../../Demo"
dotnet new sln
dotnet new console
rm Program.cs
dotnet add Demo.csproj reference ../Aptos/Stdlib/Aptos.Stdlib.csproj
dotnet add Demo.csproj reference ../Aptos/Types/Aptos.Types.csproj
dotnet add Demo.csproj reference ../Serde/Serde.csproj
dotnet add Demo.csproj reference ../Bcs/Bcs.csproj
dotnet run --project Demo.csproj
```

### Swift

To install Swift source `Serde`, `Bcs`, `AptosTypes`, and `AptosStdlib` into a target directory `$DEST`, run:
```bash
target/debug/generate-transaction-builders \
    --language swift \
    --module-name AptosStdlib \
    --with-aptos-types "testsuite/generate-format/tests/staged/aptos.yaml" \
    --target-source-dir "$DEST" \
    "language/framework/aptos-framework/releases/artifacts/current"
```

Next, you may copy the [Swift demo](examples/swift/main.swift), create the
Swift package for the transaction builders, and execute the example with the
following:

```bash
cp language/transaction-builder/generator/examples/swift/main.swift "$DEST/Sources/AptosStdlib"
cd "$DEST"
cat >Package.swift <<EOF
// swift-tools-version:5.3

import PackageDescription

let package = Package(
    name: "AptosStdlib",
    targets: [
        .target(
            name: "Serde",
            dependencies: []
        ),
        .target(
            name: "AptosTypes",
            dependencies: ["Serde"]
        ),
        .target(
            name: "AptosStdlib",
            dependencies: ["Serde", "AptosTypes"]
        ),
    ]
)
EOF
swift run
```

## Adding Support for a New Language

Supporting transaction builders in an additional programming language boils down to providing the following items:

1. Code generation for Aptos types (Rust library and tool),

2. BCS runtime (library in target language),

3. Code generation for transaction builders (Rust tool).


Items (1) and (2) are provided by the Rust library `serde-generate` which is developed in a separate [github repository](https://github.com/novifinancial/serde-reflection).

Item (3) --- this tool --- is currently developed in the Aptos repository.

Items (2) and (3) are mostly independent. Both crucially depend on (1) to be sufficiently stable, therefore our suggestion for adding a new language is first to open a new github issue in `serde-generate` and contact the Aptos maintainers.


The new issue created on `serde-generate` should include:

* requirements for the production environment (e.g. Java >= 8);

* suggestions for a testing environment in CircleCI (if not easily available in Debian 10.4 "buster");

* optionally, suggested definitions for a sample of types in the target language (structs, enums, arrays, tuples, vectors, maps, (un)signed integers of various sizes, etc).
