/*

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1alpha2

import (
	"bytes"
	"fmt"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"kubesphere.io/fluentbit-operator/api/fluentbitoperator/v1alpha2/plugins"
)

// EDIT THIS FILE!  THIS IS SCAFFOLDING FOR YOU TO OWN!
// NOTE: json tags are required.  Any new fields you add must have json tags for the fields to be serialized.

// FluentBitConfigSpec defines the desired state of FluentBitConfig
type FluentBitConfigSpec struct {
	// Service defines the global behaviour of the Fluent Bit engine.
	Service *Service `json:"service,omitempty"`
	// Select input plugins
	InputSelector metav1.LabelSelector `json:"inputSelector,omitempty"`
	// Select filter plugins
	FilterSelector metav1.LabelSelector `json:"filterSelector,omitempty"`
	// Select output plugins
	OutputSelector metav1.LabelSelector `json:"outputSelector,omitempty"`
	// Select parser plugins
	ParserSelector metav1.LabelSelector `json:"parserSelector,omitempty"`
}

type Service struct {
	// If true go to background on start
	Daemon *bool `json:"daemon,omitempty"`
	// Interval to flush output
	FlushSeconds *int64 `json:"flushSeconds,omitempty"`
	// Wait time on exit
	GraceSeconds *int64 `json:"graceSeconds,omitempty"`
	// Address to listen
	// +kubebuilder:validation:Pattern:="^\\d{1,3}.\\d{1,3}.\\d{1,3}.\\d{1,3}$"
	HttpListen string `json:"httpListen,omitempty"`
	// Port to listen
	// +kubebuilder:validation:Minimum:=1
	// +kubebuilder:validation:Maximum:=65535
	HttpPort *int32 `json:"httpPort,omitempty"`
	// If true enable statistics HTTP server
	HttpServer *bool `json:"httpServer,omitempty"`
	// File to log diagnostic output
	LogFile string `json:"logFile,omitempty"`
	// Diagnostic level (error/warning/info/debug/trace)
	// +kubebuilder:validation:Enum:=error;warning;info;debug;trace
	LogLevel string `json:"logLevel,omitempty"`
	// Optional 'parsers' config file (can be multiple)
	ParsersFile string `json:"parsersFile,omitempty"`
}

// +kubebuilder:object:root=true
// +kubebuilder:resource:shortName=fbc
// +genclient

// FluentBitConfig is the Schema for the fluentbitconfigs API
type FluentBitConfig struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	Spec FluentBitConfigSpec `json:"spec,omitempty"`
}

// +kubebuilder:object:root=true

// FluentBitConfigList contains a list of FluentBitConfig
type FluentBitConfigList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata,omitempty"`
	Items           []FluentBitConfig `json:"items"`
}

func init() {
	SchemeBuilder.Register(&FluentBitConfig{}, &FluentBitConfigList{})
}

func (s *Service) Params() *plugins.KVs {
	m := plugins.NewKVs()
	if s.Daemon != nil {
		m.Insert("Daemon", fmt.Sprint(*s.Daemon))
	}
	if s.FlushSeconds != nil {
		m.Insert("Flush", fmt.Sprint(*s.FlushSeconds))
	}
	if s.GraceSeconds != nil {
		m.Insert("Grace", fmt.Sprint(*s.GraceSeconds))
	}
	if s.HttpListen != "" {
		m.Insert("Http_Listen", s.HttpListen)
	}
	if s.HttpPort != nil {
		m.Insert("Http_Port", fmt.Sprint(*s.HttpPort))
	}
	if s.HttpServer != nil {
		m.Insert("Http_Server", fmt.Sprint(*s.HttpServer))
	}
	if s.LogFile != "" {
		m.Insert("Log_File", s.LogFile)
	}
	if s.LogLevel != "" {
		m.Insert("Log_Level", s.LogLevel)
	}
	if s.ParsersFile != "" {
		m.Insert("Parsers_File", s.ParsersFile)
	}
	return m
}

func (cfg FluentBitConfig) RenderMainConfig(sl plugins.SecretLoader, inputs InputList, filters FilterList, outputs OutputList) (string, error) {
	var buf bytes.Buffer

	// The Service defines the global behaviour of the Fluent Bit engine.
	if cfg.Spec.Service != nil {
		buf.WriteString("[Service]\n")
		buf.WriteString(cfg.Spec.Service.Params().String())
	}

	inputSections, err := inputs.Load(sl)
	if err != nil {
		return "", err
	}

	filterSections, err := filters.Load(sl)
	if err != nil {
		return "", err
	}

	outputSections, err := outputs.Load(sl)
	if err != nil {
		return "", err
	}
	if inputSections != "" && outputSections == "" {
		outputSections = `[Output]
    Name    null
    Match   *`
	}

	buf.WriteString(inputSections)
	buf.WriteString(filterSections)
	buf.WriteString(outputSections)

	return buf.String(), nil
}

func (cfg FluentBitConfig) RenderParserConfig(sl plugins.SecretLoader, parsers ParserList) (string, error) {
	var buf bytes.Buffer

	parserSections, err := parsers.Load(sl)
	if err != nil {
		return "", err
	}

	buf.WriteString(parserSections)

	return buf.String(), nil
}

func (cfg FluentBitConfig) RenderLuaScript(cl plugins.ConfigMapLoader, filters FilterList) (map[string]string, error) {

	scripts := make(map[string]string)
	for _, f := range filters.Items {
		for _, p := range f.Spec.FilterItems {
			if p.Lua != nil {
				script, err := cl.LoadConfigMap(p.Lua.Script)
				if err != nil {
					return nil, err
				}
				scripts[p.Lua.Script.Key] = script
			}
		}
	}

	return scripts, nil
}
