// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package repo

import (
	"context"

	"github.com/drone/drone/core"
	"github.com/drone/go-scm/scm"
)

type service struct {
	renew      core.Renewer
	client     *scm.Client
	visibility string
	trusted    bool
}

// New returns a new Repository service, providing access to the
// repository information from the source code management system.
func New(client *scm.Client, renewer core.Renewer, visibility string, trusted bool) core.RepositoryService {
	return &service{
		renew:      renewer,
		client:     client,
		visibility: visibility,
		trusted:    trusted,
	}
}

func (s *service) List(ctx context.Context, user *core.User) ([]*core.Repository, error) {
	err := s.renew.Renew(ctx, user, false)
	if err != nil {
		return nil, err
	}

	ctx = context.WithValue(ctx, scm.TokenKey{}, &scm.Token{
		Token:   user.Token,
		Refresh: user.Refresh,
	})
	repos := []*core.Repository{}
	opts := scm.ListOptions{Size: 100}
	for {
		result, meta, err := s.client.Repositories.List(ctx, opts)
		if err != nil {
			return nil, err
		}
		for _, src := range result {
			repos = append(repos, convertRepository(src, s.visibility, s.trusted))
		}
		opts.Page = meta.Page.Next
		opts.URL = meta.Page.NextURL

		if opts.Page == 0 && opts.URL == "" {
			break
		}
	}
	return repos, nil
}

func (s *service) Find(ctx context.Context, user *core.User, repo string) (*core.Repository, error) {
	err := s.renew.Renew(ctx, user, false)
	if err != nil {
		return nil, err
	}

	ctx = context.WithValue(ctx, scm.TokenKey{}, &scm.Token{
		Token:   user.Token,
		Refresh: user.Refresh,
	})
	result, _, err := s.client.Repositories.Find(ctx, repo)
	if err != nil {
		return nil, err
	}
	return convertRepository(result, s.visibility, s.trusted), nil
}

func (s *service) FindPerm(ctx context.Context, user *core.User, repo string) (*core.Perm, error) {
	err := s.renew.Renew(ctx, user, false)
	if err != nil {
		return nil, err
	}

	ctx = context.WithValue(ctx, scm.TokenKey{}, &scm.Token{
		Token:   user.Token,
		Refresh: user.Refresh,
	})
	result, _, err := s.client.Repositories.FindPerms(ctx, repo)
	if err != nil {
		return nil, err
	}
	return &core.Perm{
		Read:  result.Pull,
		Write: result.Push,
		Admin: result.Admin,
	}, nil
}
