// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package session

import (
	"net/http/httptest"
	"testing"
	"time"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"
	"github.com/golang/mock/gomock"
)

func TestLegacyGet_NotLegacy(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		Login: "octocat",
		Hash:  "ulSxuA0FKjNiOFIchk18NNvC6ygSxdtKjiOAS",
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().FindToken(gomock.Any(), mockUser.Hash).Return(mockUser, nil)

	r := httptest.NewRequest("GET", "/", nil)
	r.Header.Set("Authorization", "Bearer ulSxuA0FKjNiOFIchk18NNvC6ygSxdtKjiOAS")

	session, _ := Legacy(users, Config{Secure: false, Timeout: time.Hour, MappingFile: "testdata/mapping.json"})
	user, _ := session.Get(r)
	if user != mockUser {
		t.Errorf("Want authenticated user")
	}
}

func TestLegacyGet(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		Login: "octocat",
		Hash:  "ulSxuA0FKjNiOFIchk18NNvC6ygSxdtKjiOAS",
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().FindLogin(gomock.Any(), gomock.Any()).Return(mockUser, nil)
	r := httptest.NewRequest("GET", "/?access_token=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwidGV4dCI6Im9jdG9jYXQiLCJpYXQiOjE1MTYyMzkwMjJ9.jf17GpOuKu-KAhuvxtjVvmZfwyeC7mEpKNiM6_cGOvo", nil)

	session, _ := Legacy(users, Config{Secure: false, Timeout: time.Hour, MappingFile: "testdata/mapping.json"})
	user, err := session.Get(r)
	if err != nil {
		t.Error(err)
		return
	}
	if user != mockUser {
		t.Errorf("Want authenticated user")
	}
}

func TestLegacyGet_UserNotFound(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	users := mock.NewMockUserStore(controller)
	r := httptest.NewRequest("GET", "/?access_token=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwidGV4dCI6ImJpbGx5aWRvbCIsImlhdCI6MTUxNjIzOTAyMn0.yxTCucstDM7BaixXBMAJCXup9zBaFr02Kalv_PqCDM4", nil)

	session, _ := Legacy(users, Config{Secure: false, Timeout: time.Hour, MappingFile: "testdata/mapping.json"})
	_, err := session.Get(r)
	if err == nil || err.Error() != "Legacy token: cannot lookup user" {
		t.Errorf("Expect user lookup error, got %v", err)
		return
	}
}

func TestLegacyGet_InvalidSignature(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	users := mock.NewMockUserStore(controller)
	r := httptest.NewRequest("GET", "/?access_token=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwidGV4dCI6InNwYWNlZ2hvc3QiLCJpYXQiOjE1MTYyMzkwMjJ9.jlGcn2WI_oEZyLqYrvNvDXNbG3H3rqMyqQI2Gc6CHIY", nil)

	session, _ := Legacy(users, Config{Secure: false, Timeout: time.Hour, MappingFile: "testdata/mapping.json"})
	_, err := session.Get(r)
	if err == nil || err.Error() != "signature is invalid" {
		t.Errorf("Expect user lookup error, got %v", err)
		return
	}
}
