// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package acl

import (
	"net/http"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/logger"

	"github.com/go-chi/chi"
)

// CheckMembership returns an http.Handler middleware that authorizes only
// authenticated users with the required membership to an organization
// to the requested repository resource.
func CheckMembership(service core.OrganizationService, admin bool) func(http.Handler) http.Handler {
	return func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			namespace := chi.URLParam(r, "namespace")
			log := logger.FromRequest(r)
			ctx := r.Context()

			user, ok := request.UserFrom(ctx)
			if !ok {
				render.Unauthorized(w, errors.ErrUnauthorized)
				log.Debugln("api: authentication required for access")
				return
			}
			log = log.WithField("user.admin", user.Admin)

			// if the user is an administrator they are always
			// granted access to the organization data.
			if user.Admin {
				next.ServeHTTP(w, r)
				return
			}

			isMember, isAdmin, err := service.Membership(ctx, user, namespace)
			if err != nil {
				render.Unauthorized(w, errors.ErrNotFound)
				log.Debugln("api: organization membership not found")
				return
			}

			log = log.
				WithField("organization.member", isMember).
				WithField("organization.admin", isAdmin)

			if isMember == false {
				render.Unauthorized(w, errors.ErrNotFound)
				log.Debugln("api: organization membership is required")
				return
			}

			if isAdmin == false && admin == true {
				render.Unauthorized(w, errors.ErrNotFound)
				log.Debugln("api: organization administrator is required")
				return
			}

			log.Debugln("api: organization membership verified")
			next.ServeHTTP(w, r)
		})
	}
}
