// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package core

import "testing"

func TestSecretValidate(t *testing.T) {
	tests := []struct {
		secret *Secret
		error  error
	}{
		{
			secret: &Secret{Name: "password", Data: "correct-horse-battery-staple"},
			error:  nil,
		},
		{
			secret: &Secret{Name: ".some_random-password", Data: "correct-horse-battery-staple"},
			error:  nil,
		},
		{
			secret: &Secret{Name: "password", Data: ""},
			error:  errSecretDataInvalid,
		},
		{
			secret: &Secret{Name: "", Data: "correct-horse-battery-staple"},
			error:  errSecretNameInvalid,
		},
		{
			secret: &Secret{Name: "docker/password", Data: "correct-horse-battery-staple"},
			error:  errSecretNameInvalid,
		},
	}
	for i, test := range tests {
		got, want := test.secret.Validate(), test.error
		if got != want {
			t.Errorf("Want error %v, got %v at index %d", want, got, i)
		}
	}
}

func TestSecretSafeCopy(t *testing.T) {
	before := Secret{
		ID:              1,
		RepoID:          2,
		Name:            "docker_password",
		Namespace:       "octocat",
		Type:            "",
		Data:            "correct-horse-battery-staple",
		PullRequest:     true,
		PullRequestPush: true,
	}
	after := before.Copy()
	if got, want := after.ID, before.ID; got != want {
		t.Errorf("Want secret ID %d, got %d", want, got)
	}
	if got, want := after.RepoID, before.RepoID; got != want {
		t.Errorf("Want secret RepoID %d, got %d", want, got)
	}
	if got, want := after.Name, before.Name; got != want {
		t.Errorf("Want secret Name %s, got %s", want, got)
	}
	if got, want := after.Namespace, before.Namespace; got != want {
		t.Errorf("Want secret Namespace %s, got %s", want, got)
	}
	if got, want := after.PullRequest, before.PullRequest; got != want {
		t.Errorf("Want secret PullRequest %v, got %v", want, got)
	}
	if got, want := after.PullRequestPush, before.PullRequestPush; got != want {
		t.Errorf("Want secret PullRequest %v, got %v", want, got)
	}
	if after.Data != "" {
		t.Errorf("Expect secret is empty after copy")
	}
}
