<?php
/* For licensing terms, see /license.txt */

require_once api_get_path(LIBRARY_PATH).'banner.lib.php';
require_once api_get_path(SYS_PATH).'vendor/twig/twig/lib/Twig/Autoloader.php';

/**
 * Class Template
 *
 * @author Julio Montoya <gugli100@gmail.com>
 * @todo better organization of the class, methods and variables
 *
 */
class Template
{
    /**
     * The Template folder name see main/template
     * @var string
     */
    public $templateFolder = 'default';

    /**
     * The theme that will be used: chamilo, public_admin, chamilo_red, etc
     * This variable is set from the database
     * @var string
     */
    public $theme = '';

    /**
     * @var string
     */
    public $preview_theme = '';
    public $title = null;
    public $show_header;
    public $show_footer;
    public $help;
    public $menu_navigation = array(); //Used in the userportal.lib.php function: return_navigation_course_links()
    public $show_learnpath = false; // This is a learnpath section or not?
    public $plugin = null;
    public $course_id = null;
    public $user_is_logged_in = false;
    public $twig = null;

    /* Loads chamilo plugins */
    public $load_plugins = false;
    public $params = array();
    public $force_plugin_load = false;

    /**
     * @param string $title
     * @param bool $show_header
     * @param bool $show_footer
     * @param bool $show_learnpath
     * @param bool $hide_global_chat
     * @param bool $load_plugins
     */
    public function __construct(
        $title = '',
        $show_header = true,
        $show_footer = true,
        $show_learnpath = false,
        $hide_global_chat = false,
        $load_plugins = true
    ) {
        // Page title
        $this->title            = $title;
        $this->show_learnpath   = $show_learnpath;
        $this->hide_global_chat = $hide_global_chat;
        $this->load_plugins     = $load_plugins;

        // Twig settings
        Twig_Autoloader::register();

        $template_paths = array(
            api_get_path(SYS_CODE_PATH).'template', //template folder
            api_get_path(SYS_PLUGIN_PATH) //plugin folder
        );

        $cache_folder = api_get_path(SYS_ARCHIVE_PATH).'twig';

        if (!is_dir($cache_folder)) {
            mkdir($cache_folder, api_get_permissions_for_new_directories());
        }

        $loader = new Twig_Loader_Filesystem($template_paths);

        //Setting Twig options depending on the server see http://twig.sensiolabs.org/doc/api.html#environment-options
        if (api_get_setting('server_type') == 'test') {
            $options = array(
                //'cache' => api_get_path(SYS_ARCHIVE_PATH), //path to the cache folder
                'autoescape'       => false,
                'debug'            => true,
                'auto_reload'      => true,
                'optimizations'    => 0, // turn on optimizations with -1
                'strict_variables' => false, //If set to false, Twig will silently ignore invalid variables
            );
        } else {
            $options = array(
                'cache'            => $cache_folder, //path to the cache folder
                'autoescape'       => false,
                'debug'            => false,
                'auto_reload'      => false,
                'optimizations'    => -1, // turn on optimizations with -1
                'strict_variables' => false //If set to false, Twig will silently ignore invalid variables
            );
        }

        $this->twig = new Twig_Environment($loader, $options);

        $this->twig->addFilter('get_plugin_lang', new Twig_Filter_Function('get_plugin_lang'));
        $this->twig->addFilter('get_lang', new Twig_Filter_Function('get_lang'));
        $this->twig->addFilter('get_path', new Twig_Filter_Function('api_get_path'));
        $this->twig->addFilter('get_setting', new Twig_Filter_Function('api_get_setting'));
        $this->twig->addFilter('var_dump', new Twig_Filter_Function('var_dump'));
        $this->twig->addFilter('return_message', new Twig_Filter_Function('Display::return_message_and_translate'));
        $this->twig->addFilter('display_page_header', new Twig_Filter_Function('Display::page_header_and_translate'));
        $this->twig->addFilter(
            'display_page_subheader',
            new Twig_Filter_Function('Display::page_subheader_and_translate')
        );
        $this->twig->addFilter('icon', new Twig_Filter_Function('Template::get_icon_path'));
        $this->twig->addFilter('format_date', new Twig_Filter_Function('Template::format_date'));
        $this->twig->addFilter('api_get_local_time', new Twig_Filter_Function('api_get_local_time'));

        /*
          $lexer = new Twig_Lexer($this->twig, array(
          //'tag_comment'  => array('{*', '*}'),
          //'tag_comment'  => array('{#', '#}'),
          //'tag_block'    => array('{', '}'),
          //'tag_variable' => array('{$', '}'),
          ));
          $this->twig->setLexer($lexer); */

        //Setting system variables
        $this->set_system_parameters();

        //Setting user variables
        $this->set_user_parameters();

        //Setting course variables
        $this->set_course_parameters();

        //header and footer are showed by default
        $this->set_footer($show_footer);
        $this->set_header($show_header);

        $this->set_header_parameters();
        $this->set_footer_parameters();

        $defaultStyle = api_get_configuration_value('default_template');
        if (!empty($defaultStyle)) {
            $this->templateFolder = $defaultStyle;
        }

        $this->assign('template', $this->templateFolder);
        $this->assign('css_styles', $this->theme);
        $this->assign('login_class', null);

        // Chamilo plugins
        if ($this->show_header) {
            if ($this->load_plugins) {

                $this->plugin = new AppPlugin();

                //1. Showing installed plugins in regions
                $plugin_regions = $this->plugin->get_plugin_regions();
                foreach ($plugin_regions as $region) {
                    $this->set_plugin_region($region);
                }

                //2. Loading the course plugin info
                global $course_plugin;
                if (isset($course_plugin) && !empty($course_plugin) && !empty($this->course_id)) {
                    //Load plugin get_langs
                    $this->plugin->load_plugin_lang_variables($course_plugin);
                }
            }
        }
    }

    public static function get_icon_path($image, $size = ICON_SIZE_SMALL)
    {
        return Display:: return_icon($image, '', array(), $size, false, true);
    }

    public static function format_date($timestamp, $format = null)
    {
        return api_format_date($timestamp, $format);
    }

    /**
     * Return the item's url key:
     *
     *      c_id=xx&id=xx
     *
     * @param object $item
     * @return string
     */
    public static function key($item)
    {
        $id     = isset($item->id) ? $item->id : null;
        $c_id   = isset($item->c_id) ? $item->c_id : null;
        $result = '';
        if ($c_id) {
            $result = "c_id=$c_id";
        }
        if ($id) {
            if ($result) {
                $result .= "&amp;id=$id";
            } else {
                $result .= "&amp;id=$id";
            }
        }
        return $result;
    }

    /**
     * @param string $helpInput
     */
    public function set_help($helpInput = null)
    {
        if (!empty($helpInput)) {
            $help = $helpInput;
        } else {
            $help = $this->help;
        }

        $content = '';
        if (api_get_setting('enable_help_link') == 'true') {
            if (!empty($help)) {
                $help = Security::remove_XSS($help);
                $content = '<li class="help">';
                $content .= Display::url(
                    Display::return_icon('help.large.png', get_lang('Help')),
                    api_get_path(WEB_CODE_PATH).'help/help.php?open='.$help.'&height=400&width=600', array('class' => 'ajax'));
                $content .= '</li>';
            }
        }
        $this->assign('help_content', $content);
    }

    /**
     * Use template system to parse the actions menu
     * @todo finish it!
     **/
    public function set_actions($actions)
    {
        $action_string = '';
        if (!empty($actions)) {
            foreach ($actions as $action) {
                $action_string .= $action;
            }
        }
        $this->assign('actions', $actions);
    }

    /**
     * Shortcut to display a 1 col layout (index.php)
     * */
    public function display_one_col_template()
    {
        $tpl = $this->get_template('layout/layout_1_col.tpl');
        $this->display($tpl);
    }

    /**
     * Shortcut to display a 2 col layout (userportal.php)
     **/
    public function display_two_col_template()
    {
        $tpl = $this->get_template('layout/layout_2_col.tpl');
        $this->display($tpl);
    }

    /**
     * Displays an empty template
     */
    public function display_blank_template()
    {
        $tpl = $this->get_template('layout/blank.tpl');
        $this->display($tpl);
    }

    /**
     * Displays an empty template
     */
    public function display_no_layout_template()
    {
        $tpl = $this->get_template('layout/no_layout.tpl');
        $this->display($tpl);
    }

    /**
     * Sets the footer visibility
     * @param bool true if we show the footer
     */
    public function set_footer($status)
    {
        $this->show_footer = $status;
        $this->assign('show_footer', $status);
    }

    /**
     * return true if toolbar has to be displayed for user
     * @return bool
     */
    public static function isToolBarDisplayedForUser()
    {
        //Toolbar
        $show_admin_toolbar = api_get_setting('show_admin_toolbar');
        $show_toolbar = false;

        switch ($show_admin_toolbar) {
            case 'do_not_show':
                break;
            case 'show_to_admin':
                if (api_is_platform_admin()) {
                    $show_toolbar = true;
                }
                break;
            case 'show_to_admin_and_teachers':
                if (api_is_platform_admin() || api_is_allowed_to_edit()) {
                    $show_toolbar = true;
                }
                break;
            case 'show_to_all':
                $show_toolbar = true;
                break;
        }
        return $show_toolbar;
    }

    /**
     * Sets the header visibility
     * @param bool true if we show the header
     */
    public function set_header($status)
    {
        $this->show_header = $status;
        $this->assign('show_header', $status);

        $show_toolbar = 0;

        if (self::isToolBarDisplayedForUser()) {
            $show_toolbar = 1;
        }

        $this->assign('show_toolbar', $show_toolbar);

        //Only if course is available
        $show_course_shortcut        = null;
        $show_course_navigation_menu = null;

        if (!empty($this->course_id) && $this->user_is_logged_in) {
            if (api_get_setting('show_toolshortcuts') != 'false') {
                //Course toolbar
                $show_course_shortcut = CourseHome::show_navigation_tool_shortcuts();
            }
            if (api_get_setting('show_navigation_menu') != 'false') {
                //Course toolbar
                $show_course_navigation_menu = CourseHome::show_navigation_menu();
            }
        }
        $this->assign('show_course_shortcut', $show_course_shortcut);
        $this->assign('show_course_navigation_menu', $show_course_navigation_menu);
    }

    /**
     * @param string $name
     *
     * @return string
     */
    public function get_template($name)
    {
        return $this->templateFolder.'/'.$name;
    }

    /**
     * Set course parameters
     */
    private function set_course_parameters()
    {
        //Setting course id
        $course = api_get_course_info();
        if (empty($course)) {
            $this->assign('course_is_set', false);
            return;
        }
        $this->assign('course_is_set', true);
        $this->course_id = $course['id'];
        $_c = array(
            'id' => $course['id'],
            'code' => $course['code'],
            'title' => $course['name'],
            'visibility' => $course['visibility'],
            'language' => $course['language'],
            'directory' => $course['directory'],
            'session_id' => api_get_session_id(),
            'user_is_teacher' => api_is_course_admin(),
            'student_view' => (!empty($_GET['isStudentView']) && $_GET['isStudentView'] == 'true'),
        );
        $this->assign('course_code', $course['code']);
        $this->assign('_c', $_c);
    }

    /**
     * Set user parameters
     */
    private function set_user_parameters()
    {
        $user_info               = array();
        $user_info['logged']     = 0;
        $this->user_is_logged_in = false;
        if (api_user_is_login()) {
            $user_info           = api_get_user_info(api_get_user_id());
            $user_info['logged'] = 1;

            $user_info['is_admin'] = 0;
            if (api_is_platform_admin()) {
                $user_info['is_admin'] = 1;
            }

            $user_info['messages_count'] = MessageManager::get_new_messages();
            $this->user_is_logged_in     = true;
        }
        //Setting the $_u array that could be use in any template
        $this->assign('_u', $user_info);
    }

    /**
     * Set system parameters
     */
    private function set_system_parameters()
    {
        global $_configuration;

        //Setting app paths/URLs
        $_p = array(
            'web'        => api_get_path(WEB_PATH),
            'web_course' => api_get_path(WEB_COURSE_PATH),
            'web_main'   => api_get_path(WEB_CODE_PATH),
            'web_css'    => api_get_path(WEB_CSS_PATH),
            'web_ajax'   => api_get_path(WEB_AJAX_PATH),
            'web_img'    => api_get_path(WEB_IMG_PATH),
            'web_plugin' => api_get_path(WEB_PLUGIN_PATH),
            'web_lib'    => api_get_path(WEB_LIBRARY_PATH),
            'web_self' => api_get_self(),
            'web_query_vars' => api_htmlentities($_SERVER['QUERY_STRING']),
            'web_self_query_vars' => api_htmlentities($_SERVER['REQUEST_URI']),
            'web_cid_query' => api_get_cidreq(),
        );
        $this->assign('_p', $_p);

        //Here we can add system parameters that can be use in any template
        $_s = array(
            'software_name'  => $_configuration['software_name'],
            'system_version' => $_configuration['system_version'],
            'site_name'      => api_get_setting('siteName'),
            'institution'    => api_get_setting('Institution')
        );
        $this->assign('_s', $_s);
    }

    /**
     * Set theme, include CSS files
     */
    public function set_css_files()
    {
        global $disable_js_and_css_files;
        $css = array();

        $this->theme = api_get_visual_theme();

        if (!empty($this->preview_theme)) {
            $this->theme = $this->preview_theme;
        }

        //Base CSS
        $css[] = api_get_cdn_path(api_get_path(WEB_CSS_PATH).'base.css');

        //Default CSS responsive design
        $css[] = api_get_cdn_path(api_get_path(WEB_CSS_PATH).'bootstrap-responsive.css');


        //Extra CSS files
        $css[] = api_get_path(WEB_LIBRARY_PATH).'javascript/thickbox.css';
        $css[] = api_get_path(WEB_LIBRARY_PATH).'javascript/chosen/chosen.css';

        if (api_is_global_chat_enabled()) {
            $css[] = api_get_path(WEB_LIBRARY_PATH).'javascript/chat/css/chat.css';
        }

        $css[] = api_get_path(WEB_CSS_PATH).'font_awesome/css/font-awesome.css';
        $css[] = api_get_path(WEB_LIBRARY_PATH).'javascript/mediaelement/mediaelementplayer.css';
        //THEME CSS STYLE
        $css[] = api_get_cdn_path(api_get_path(WEB_CSS_PATH).'responsive.css');
        $css[] = api_get_cdn_path(api_get_path(WEB_CSS_PATH).$this->theme.'/default.css');

        if ($this->show_learnpath) {
            $css[] = api_get_path(WEB_CSS_PATH).$this->theme.'/learnpath.css';

            // if we have a SCORM file in theme, don't use default_scorm.css file
            if (is_file(api_get_path(SYS_CSS_PATH).$this->theme.'/scorm.css')) {
                $css[] = api_get_path(WEB_CSS_PATH).$this->theme.'/scorm.css';
            } else {
                $css[] = api_get_cdn_path(api_get_path(WEB_CSS_PATH).'default_scorm.css');
            }
        }

        $css_file_to_string = null;
        foreach ($css as $file) {
            $css_file_to_string .= api_get_css($file);
        }

        // @todo move this somewhere else. Special fix when using tablets in order to see the text near icons
        if (SHOW_TEXT_NEAR_ICONS == true) {
            //hack in order to fix the actions buttons
            $css_file_to_string .= '<style>
                .td_actions a {
                    float:left;
                    width:100%;
                }
                .forum_message_left a {
                    float:left;
                    width:100%;
                }
                </style>';
        }

        $navigator_info = api_get_navigator();
        if ($navigator_info['name'] == 'Internet Explorer' && $navigator_info['version'] == '6') {
            $css_file_to_string .= 'img, div { behavior: url('.api_get_path(WEB_LIBRARY_PATH).'javascript/iepngfix/iepngfix.htc) } '."\n";
        }

        if (!$disable_js_and_css_files) {
            $this->assign('css_file_to_string', $css_file_to_string);

            $style_print = api_get_css(api_get_cdn_path(api_get_path(WEB_CSS_PATH).$this->theme.'/print.css'), 'print');
            $this->assign('css_style_print', $style_print);
        }

        // Logo
        $logo = return_logo($this->theme);
        $this->assign('logo', $logo);
    }

    /**
     * Declare and define the template variable that will be used to load
     * javascript libraries in the header.
     */
    public function set_js_files()
    {
        global $disable_js_and_css_files, $htmlHeadXtra;

        //JS files
        $js_files = array(
            'modernizr.js',
            'jquery.min.js',
            'chosen/chosen.jquery.min.js',
            'thickbox.js',
            'bootstrap/bootstrap.js',
            'mediaelement/mediaelement-and-player.min.js'
        );

        if (api_is_global_chat_enabled()) {
            //Do not include the global chat in LP
            if ($this->show_learnpath == false && $this->show_footer == true && $this->hide_global_chat == false) {
                $js_files[] = 'chat/js/chat.js';
            }
        }

        if (api_get_setting('accessibility_font_resize') == 'true') {
            $js_files[] = 'fontresize.js';
        }

        if (api_get_setting('include_asciimathml_script') == 'true') {
            $js_files[] = 'asciimath/ASCIIMathML.js';
        }

        $js_file_to_string = null;

        foreach ($js_files as $js_file) {
            $js_file_to_string .= api_get_js($js_file);
        }

        //Loading email_editor js
        if (!api_is_anonymous() && api_get_setting('allow_email_editor') == 'true') {
            $js_file_to_string .= $this->fetch('default/mail_editor/email_link.js.tpl');
        }

        if (!$disable_js_and_css_files) {
            $this->assign('js_file_to_string', $js_file_to_string);

            //Adding jquery ui by default
            $extra_headers = api_get_jquery_ui_js();

            //$extra_headers = '';
            if (isset($htmlHeadXtra) && $htmlHeadXtra) {
                foreach ($htmlHeadXtra as & $this_html_head) {
                    $extra_headers .= $this_html_head."\n";
                }
            }
            $this->assign('extra_headers', $extra_headers);
        }
    }

    /**
     * Special function to declare last-minute JS libraries which depend on
     * other things to be declared first. In particular, it might be useful
     * under IE9 with compatibility mode, which for some reason is getting
     * upset when a variable is used in a function (even if not used yet)
     * when this variable hasn't been defined yet.
     */
    public function set_js_files_post()
    {
        global $disable_js_and_css_files, $htmlHeadXtra;
        $js_files = array();
        if (api_is_global_chat_enabled()) {
            //Do not include the global chat in LP
            if ($this->show_learnpath == false && $this->show_footer == true && $this->hide_global_chat == false) {
                $js_files[] = 'chat/js/chat.js';
            }
        }
        $js_file_to_string = null;

        foreach ($js_files as $js_file) {
            $js_file_to_string .= api_get_js($js_file);
        }
        if (!$disable_js_and_css_files) {
            $this->assign('js_file_to_string_post', $js_file_to_string);
        }
    }

    /**
     * Set header parameters
     */
    private function set_header_parameters()
    {
        global $httpHeadXtra, $_course, $interbreadcrumb, $language_file, $noPHP_SELF, $_configuration, $this_section;
        $help = $this->help;
        $nameTools             = $this->title;
        $navigation            = return_navigation_array();
        $this->menu_navigation = $navigation['menu_navigation'];

        $this->assign('system_charset', api_get_system_encoding());

        if (isset($httpHeadXtra) && $httpHeadXtra) {
            foreach ($httpHeadXtra as & $thisHttpHead) {
                header($thisHttpHead);
            }
        }

        $this->assign('online_button', Display::return_icon('online.png'));
        $this->assign('offline_button',Display::return_icon('offline.png'));

        // Get language iso-code for this page - ignore errors
        $this->assign('document_language', api_get_language_isocode());

        $course_title = isset($_course['name']) ? $_course['name'] : null;

        $title_list = array();

        $title_list[] = api_get_setting('Institution');
        $title_list[] = api_get_setting('siteName');

        if (!empty($course_title)) {
            $title_list[] = $course_title;
        }
        if ($nameTools != '') {
            $title_list[] = $nameTools;
        }

        $title_string = '';
        for ($i = 0; $i < count($title_list); $i++) {
            $title_string .= $title_list[$i];
            if (isset($title_list[$i + 1])) {
                $item = trim($title_list[$i + 1]);
                if (!empty($item)) {
                    $title_string .= ' - ';
                }
            }
        }

        $this->assign('title_string', $title_string);

        //Setting the theme and CSS files
        $this->set_css_files();
        $this->set_js_files();
        //$this->set_js_files_post();

        $browser = api_browser_support('check_browser');
        if ($browser[0] == 'Internet Explorer' && $browser[1] >= '11') {
            $browser_head = '<meta http-equiv="X-UA-Compatible" content="IE=EmulateIE9" />';
            $this->assign('browser_specific_head', $browser_head);
        }

        // Implementation of prefetch.
        // See http://cdn.chamilo.org/main/img/online.png for details
        $prefetch = '';
        if (!empty($_configuration['cdn_enable'])) {
            $prefetch .= '<meta http-equiv="x-dns-prefetch-control" content="on">';
            foreach ($_configuration['cdn'] as $host => $exts) {
                $prefetch .= '<link rel="dns-prefetch" href="'.$host.'">';
            }
        }

        $this->assign('prefetch', $prefetch);
        $this->assign('text_direction', api_get_text_direction());
        $this->assign('section_name', 'section-'.$this_section);

        $favico = '<link rel="shortcut icon" href="'.api_get_path(WEB_PATH).'favicon.ico" type="image/x-icon" />';

        if (isset($_configuration['multiple_access_urls']) && $_configuration['multiple_access_urls']) {
            $access_url_id = api_get_current_access_url_id();
            if ($access_url_id != -1) {
                $url_info  = api_get_access_url($access_url_id);
                $url       = api_remove_trailing_slash(preg_replace('/https?:\/\//i', '', $url_info['url']));
                $clean_url = replace_dangerous_char($url);
                $clean_url = str_replace('/', '-', $clean_url);
                $clean_url .= '/';
                $homep           = api_get_path(REL_PATH).'home/'.$clean_url; //homep for Home Path
                $icon_real_homep = api_get_path(SYS_PATH).'home/'.$clean_url;

                //we create the new dir for the new sites
                if (is_file($icon_real_homep.'favicon.ico')) {
                    $favico = '<link rel="shortcut icon" href="'.$homep.'favicon.ico" type="image/x-icon" />';
                }
            }
        }

        $this->assign('favico', $favico);

        $this->set_help();

        //@todo move this in the template
        $bug_notification_link = '';
        if (api_get_setting('show_link_bug_notification') == 'true' && $this->user_is_logged_in) {
            $bug_notification_link = '<li class="report">
		        						<a href="http://support.chamilo.org/projects/chamilo-18/wiki/How_to_report_bugs" target="_blank">
		        						<img src="'.api_get_path(WEB_IMG_PATH).'bug.large.png" style="vertical-align: middle;" alt="'.get_lang('ReportABug').'" title="'.get_lang(
                    'ReportABug'
                ).'"/></a>
		    						  </li>';
        }

        $this->assign('bug_notification_link', $bug_notification_link);

        $notification = return_notification_menu();
        $this->assign('notification_menu', $notification);

        //Preparing values for the menu

        //Logout link
        if (isset($_configuration['hide_logout_button']) && $_configuration['hide_logout_button'] == 'true') {
            $this->assign('logout_link', null);
        } else {
            $this->assign('logout_link', api_get_path(WEB_PATH).'index.php?logout=logout&uid='.api_get_user_id());
        }

        //Profile link
        if (api_get_setting('allow_social_tool') == 'true') {
            $profile_url  = api_get_path(WEB_CODE_PATH).'social/home.php';
            $profile_link = Display::url(get_lang('Profile'), $profile_url);
        } else {
            $profile_url  = api_get_path(WEB_CODE_PATH).'auth/profile.php';
            $profile_link = Display::url(get_lang('Profile'), $profile_url);
        }
        $this->assign('profile_link', $profile_link);
        $this->assign('profile_url', $profile_url);

        //Message link
        $message_link = null;
        $message_url  = null;
        if (api_get_setting('allow_message_tool') == 'true') {
            $message_url  = api_get_path(WEB_CODE_PATH).'messages/inbox.php';
            $message_link = '<a href="'.api_get_path(WEB_CODE_PATH).'messages/inbox.php">'.get_lang('Inbox').'</a>';
        }
        $this->assign('message_link', $message_link);
        $this->assign('message_url', $message_url);

        $institution = api_get_setting('Institution');
        $portal_name = empty($institution) ? api_get_setting('siteName') : $institution;

        $this->assign('portal_name', $portal_name);

        //Menu
        $menu = return_menu();
        $this->assign('menu', $menu);

        //Setting notifications


        $count_unread_message = 0;
        if (api_get_setting('allow_message_tool') == 'true') {
            // get count unread message and total invitations
            $count_unread_message = MessageManager::get_number_of_messages(true);
        }

        $total_invitations = 0;
        if (api_get_setting('allow_social_tool') == 'true') {
            $number_of_new_messages_of_friend = SocialManager::get_message_number_invitation_by_user_id(
                api_get_user_id()
            );
            $group_pending_invitations        = GroupPortalManager::get_groups_by_user(
                api_get_user_id(),
                GROUP_USER_PERMISSION_PENDING_INVITATION,
                false
            );
            $group_pending_invitations        = 0;
            if (!empty($group_pending_invitations)) {
                $group_pending_invitations = count($group_pending_invitations);
            }
            $total_invitations = intval($number_of_new_messages_of_friend) + $group_pending_invitations + intval(
                    $count_unread_message
                );
        }
        $total_invitations = (!empty($total_invitations) ? Display::badge($total_invitations) : null);

        $this->assign('user_notifications', $total_invitations);


        //Breadcrumb
        $breadcrumb = return_breadcrumb($interbreadcrumb, $language_file, $nameTools);
        $this->assign('breadcrumb', $breadcrumb);

        //Extra content
        $extra_header = null;
        if (!api_is_platform_admin()) {
            $extra_header = trim(api_get_setting('header_extra_content'));
        }
        $this->assign('header_extra_content', $extra_header);

        //if ($this->show_header == 1) {
            header('Content-Type: text/html; charset='.api_get_system_encoding());
            header(
                'X-Powered-By: '.$_configuration['software_name'].' '.substr($_configuration['system_version'], 0, 1)
            );
        //}
    }

    /**
     * Set footer parameters
     */
    private function set_footer_parameters()
    {
        global $_configuration;

        //Show admin data
        //$this->assign('show_administrator_data', api_get_setting('show_administrator_data'));

        if (api_get_setting('show_administrator_data') == 'true') {
            //Administrator name
            $administrator_data = get_lang('Manager').' : '.Display::encrypted_mailto_link(
                    api_get_setting('emailAdministrator'),
                    api_get_person_name(api_get_setting('administratorName'), api_get_setting('administratorSurname'))
                );
            $this->assign('administrator_name', $administrator_data);
        }

        //Loading footer extra content
        if (!api_is_platform_admin()) {
            $extra_footer = trim(api_get_setting('footer_extra_content'));
            if (!empty($extra_footer)) {
                $this->assign('footer_extra_content', $extra_footer);
            }
        }

        //Tutor name
        if (api_get_setting('show_tutor_data') == 'true') {
            // Course manager
            $id_course  = api_get_course_id();
            $id_session = api_get_session_id();
            if (isset($id_course) && $id_course != -1) {
                $tutor_data = '';
                if ($id_session != 0) {
                    $coachs_email = CourseManager::get_email_of_tutor_to_session($id_session, $id_course);
                    $email_link   = array();
                    foreach ($coachs_email as $coach) {
                        $email_link[] = Display::encrypted_mailto_link($coach['email'], $coach['complete_name']);
                    }
                    if (count($coachs_email) > 1) {
                        $tutor_data .= get_lang('Coachs').' : ';
                        $tutor_data .= array_to_string($email_link, CourseManager::USER_SEPARATOR);
                    } elseif (count($coachs_email) == 1) {
                        $tutor_data .= get_lang('Coach').' : ';
                        $tutor_data .= array_to_string($email_link, CourseManager::USER_SEPARATOR);
                    } elseif (count($coachs_email) == 0) {
                        $tutor_data .= '';
                    }
                }
                $this->assign('session_teachers', $tutor_data);
            }
        }

        if (api_get_setting('show_teacher_data') == 'true') {
            // course manager
            $id_course = api_get_course_id();
            if (isset($id_course) && $id_course != -1) {
                $teacher_data = '';
                $mail         = CourseManager::get_emails_of_tutors_to_course($id_course);
                if (!empty($mail)) {
                    $teachers_parsed = array();
                    foreach ($mail as $value) {
                        foreach ($value as $email => $name) {
                            $teachers_parsed[] = Display::encrypted_mailto_link($email, $name);
                        }
                    }
                    $label = get_lang('Teacher');
                    if (count($mail) > 1) {
                        $label = get_lang('Teachers');
                    }
                    $teacher_data .= $label.' : '.array_to_string($teachers_parsed, CourseManager::USER_SEPARATOR);
                }
                $this->assign('teachers', $teacher_data);
            }
        }
        /* $stats = '';
          $this->assign('execution_stats', $stats); */
    }

    /**
     * Show header template.
     */
    public function show_header_template()
    {
        $tpl = $this->get_template('layout/show_header.tpl');

        $this->display($tpl);
    }

    /**
     * Show footer template.
     */
    public function show_footer_template()
    {
        $tpl = $this->get_template('layout/show_footer.tpl');
        $this->display($tpl);
    }

    /**
     * Sets the plugin content in a template variable
     * @param string $plugin_region
     * @return null
     */
    public function set_plugin_region($plugin_region)
    {
        if (!empty($plugin_region)) {
            $region_content = $this->plugin->load_region($plugin_region, $this, $this->force_plugin_load);
            if (!empty($region_content)) {
                $this->assign('plugin_'.$plugin_region, $region_content);
            } else {
                $this->assign('plugin_'.$plugin_region, null);
            }
        }
        return null;
    }

    /**
     * @param string $template
     * @return string
     */
    public function fetch($template = null)
    {
        $template = $this->twig->loadTemplate($template);
        return $template->render($this->params);
    }

    /**
     * @param $tpl_var
     * @param null $value
     */
    public function assign($tpl_var, $value = null)
    {
        $this->params[$tpl_var] = $value;
    }

    /**
     * @param string $template
     */
    public function display($template)
    {
        $this->assign('flash_messages', Display::getFlashToString());
        Display::cleanFlashMessages();
        echo $this->twig->render($template, $this->params);
    }

    /**
     * Adds a body class for login pages
     */
    public function setLoginBodyClass()
    {
        $this->assign('login_class', 'section-login');
    }

    /**
     * The theme that will be used if the database is not working.
     * @return string
     */
    public static function getThemeFallback()
    {
        $theme = api_get_configuration_value('theme_fallback');
        if (empty($theme)) {
            $theme = 'chamilo';
        }
        return $theme;
    }
}
