/*
 * Copyright 2019 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jf.ptgen.penrose;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.PrecisionModel;

import java.util.HashSet;
import java.util.Random;
import java.util.Set;

/**
 * Represents a penrose tiling generated by de Bruijn's method.
 *
 * (1) https://www.mathpages.com/home/kmath621/kmath621.htm
 * (2) http://www.ams.org/publicoutreach/feature-column/fcarc-ribbons
 */
public class PenroseTiling {
    public final StripFamily[] stripFamilies;

    public static PrecisionModel PRECISION_MODEL = new PrecisionModel(1E10);
    public static GeometryFactory GEOMETRY_FACTORY = new GeometryFactory(PRECISION_MODEL);

    /**
     * Create a new tiling using the given offsets.
     *
     * The given offsets are the offset for each line family. i.e. the Sj offset
     * mentioned in (1)
     *
     * @param offsets An array of 5 offsets, 1 for each line family. These offset must
     *                follow the constraints outlined in (1). They must be distinct from
     *                each other, the sum of any two must not be an integer, and the sum
     *                of all 5 must be an integer.
     */
    public PenroseTiling(double[] offsets) {
        stripFamilies = new StripFamily[] {
                new StripFamily(this, offsets[0], 0),
                new StripFamily(this, offsets[1], 1),
                new StripFamily(this, offsets[2], 2),
                new StripFamily(this, offsets[3], 3),
                new StripFamily(this, offsets[4], 4),
        };
    }

    /**
     * Create a new, random tiling.
     *
     * @param random the Random instance to use.
     */
    public PenroseTiling(Random random) {
        this(randomOffsets(random));
    }

    private static double[] randomOffsets(Random random) {
        double[] offsets = new double[5];
        double sum = 0;
        // It's technically possible for 2 random offsets to break one of the constraints
        // mentioned in (1), but in practice, it shouldn't be an issue.
        for (int i = 0; i < 4; i++) {
            offsets[i] = random.nextDouble();
            sum += offsets[i];
        }
        offsets[4] = Math.ceil(sum) - sum;
        return offsets;
    }

    /**
     * Get the given StripFamily.
     *
     * @param familyIndex The index of the strip family to get. Must be in [0,4].
     * @return The given StripFamily.
     */
    public StripFamily getStripFamily(int familyIndex) {
        return stripFamilies[familyIndex];
    }

    /**
     * Visit all rhombii for the given bounding box.
     *
     * A rhombus is visited if the majority of it is contained in the bounding box. For rhombii
     * that are equally split, the bounding box with the lower x and then lower y "wins".
     *
     * The guarantees that, if you have multiple adjacent bounding boxes in a grid, a given rhombus
     * will be assigned to exactly one of the bounding boxes. This is useful to split up a tiling
     * into multiple rectangular-ish segments that can then be re-joined with no gaps or overlaps.
     */
    public void visitRhombii(BoundingBox boundingBox, RhombusVisitor visitor) {
        Set<Rhombus> processedRhombii = new HashSet<>();
        Set<Rhombus> pendingRhombii = new HashSet<>();
        Set<Strip> existingStrips = new HashSet<>();

        StripFamily initialFamily = getStripFamily(0);

        // every unit in pentagrid space is ~2.5 units in the penrose space
        Coordinate midpoint = boundingBox.polygon.getCentroid().getCoordinate();
        int approximateMultiple = (int)(midpoint.x / 2.5);
        int approximateTarget = (int)(midpoint.y / 2.5);

        Strip strip = initialFamily.getStrip(approximateMultiple);

        Rhombus initialRhombus = strip.getRhombus(approximateTarget);
        if (!keepRhombus(boundingBox, initialRhombus)) {
            throw new RuntimeException("Initial rhombus is outside the bounding box. Maybe the "
                    + "bounding boxes are too small?");
        }

        visitRhombus(processedRhombii, pendingRhombii, visitor, initialRhombus);
        processStrip(boundingBox, existingStrips, pendingRhombii, processedRhombii, visitor,
                initialRhombus.strip1, initialRhombus.strip2);
        processStrip(boundingBox, existingStrips, pendingRhombii, processedRhombii, visitor,
                initialRhombus.strip2, initialRhombus.strip1);

        while (!pendingRhombii.isEmpty()) {
            Rhombus rhombus = pendingRhombii.iterator().next();
            pendingRhombii.remove(rhombus);
            processStrip(boundingBox, existingStrips, pendingRhombii, processedRhombii, visitor,
                    rhombus.strip2, rhombus.strip1);
        }
    }

    private static void visitRhombus(Set<Rhombus> processedRhombii, Set<Rhombus> pendingRhombii,
                                     RhombusVisitor visitor, Rhombus rhombus) {
        if (!processedRhombii.contains(rhombus)) {
            processedRhombii.add(rhombus);
            visitor.visitRhombus(rhombus);
        }

        pendingRhombii.add(rhombus);
    }

    private static boolean keepRhombus(BoundingBox boundingBox, Rhombus rhombus) {
        return rhombus.getContainingBoundingBox(boundingBox.gridOrigin, boundingBox.gridSize)
                .equals(boundingBox);
    }

    private static void processStrip(BoundingBox boundingBox, Set<Strip> existingStrips,
                                     Set<Rhombus> pendingRhombii, Set<Rhombus> processedRhombii,
                                     RhombusVisitor visitor, Strip strip1, Strip strip2) {
        if (existingStrips.contains(strip1)) {
            return;
        }
        existingStrips.add(strip1);

        boolean first = true;
        for (Rhombus next: strip1.getRhombii(strip2, true)) {
            if (first) {
                first = false;
                continue;
            }
            if (keepRhombus(boundingBox, next)) {
                visitRhombus(processedRhombii, pendingRhombii, visitor, next);
            } else {
                processedRhombii.add(next);
            }
            if (!continueProcessingStrip(next, boundingBox)) {
                break;
            }
        }

        first = true;
        for (Rhombus next: strip1.getRhombii(strip2, false)) {
            if (first) {
                first = false;
                continue;
            }

            if (keepRhombus(boundingBox, next)) {
                visitRhombus(processedRhombii, pendingRhombii, visitor, next);
            } else {
                processedRhombii.add(next);
            }
            if (!continueProcessingStrip(next, boundingBox)) {
                break;
            }
        }
    }

    private static boolean continueProcessingStrip(Rhombus rhombus, BoundingBox boundingBox) {
        // +/- 5, in order to catch the case of a strip parallel with an edge that goes in and out
        // of the bounding box 5 should enough for approximately 2 strips width
        for (Coordinate vertex : rhombus.getVertices()) {
            if (vertex.x < (boundingBox.origin.x - 5) || vertex.x > (boundingBox.extent.x + 5)) {
                return false;
            }
            if (vertex.y < (boundingBox.origin.y - 5) || vertex.y > (boundingBox.extent.y + 5)) {
                return false;
            }
        }
        return true;
    }

    /**
     * A callback for visiting a rhombus when generating a tiling.
     */
    public interface RhombusVisitor {
        /**
         * When generating a tiling, this method will be called for every rhombus in the tiling.
         * @param rhombus The rhombus to visit.
         */
        void visitRhombus(Rhombus rhombus);
    }
}
