package pike

import (
	"encoding/json"
	"errors"
	"fmt"
)

const terraform string = "terraform"

// GetAWSPermissions for AWS resources.
func GetAWSPermissions(result ResourceV2) ([]string, error) {
	var (
		err         error
		Permissions []string
	)

	switch result.TypeName {
	case "resource", terraform:
		{
			Permissions, err = GetAWSResourcePermissions(result)

			if err != nil {
				return Permissions, err
			}
		}
	case "data":
		{
			Permissions, err = GetAWSDataPermissions(result)
			if err != nil {
				return Permissions, err
			}
		}
	case "module":
		{
			// do nothing this is a module not a base resource type, and
			// we shouldn't really be able to get here unless well bad naming
		}
	default:
		{
			return nil, fmt.Errorf("unknown permission resource type %s", result.Name)
		}
	}

	return Permissions, nil
}

// GetAWSResourcePermissions looks up permissions required for resources
//
//goland:noinspection GoLinter
func GetAWSResourcePermissions(result ResourceV2) ([]string, error) {
	var (
		Permissions []string
		err         error
	)

	if temp := AwsLookup(result.Name); temp != nil {

		Permissions, err = GetPermissionMap(temp.([]byte), result.Attributes, result.Name)
	} else {
		return nil, fmt.Errorf("%s not implemented", result.Name)
	}

	return Permissions, err
}

func AwsLookup(name string) interface{} {
	TFLookup := map[string]interface{}{
		"aws_accessanalyzer_analyzer":                  awsAccessAnalyzer,
		"aws_account_alternate_contact":                awsAccountAlternativeContact,
		"aws_account_primary_contact":                  awsAccountPrimaryContact,
		"aws_acm_certificate":                          AWSAcmCertificate,
		"aws_acm_certificate_validation":               placeholder,
		"aws_acmpca_certificate":                       awsAcmpcaCertificate,
		"aws_acmpca_certificate_authority":             awsAcmpcaCertificateAuthority,
		"aws_acmpca_certificate_authority_certificate": awsAcmpcaCertificateAuthorityCertificate,
		"aws_alb":                                                  awsLb,
		"aws_alb_listener":                                         awsLbListener,
		"aws_alb_target_group":                                     awsLbTargetGroup,
		"aws_alb_target_group_attachment":                          awsLbTargetGroupAttachment,
		"aws_ami":                                                  awsAmi,
		"aws_ami_copy":                                             awsAmiCopy,
		"aws_ami_from_instance":                                    awsAmiFromInstance,
		"aws_ami_launch_permission":                                awsAmiLauchPermission,
		"aws_api_gateway_account":                                  awsAPIGatewayAccount,
		"aws_api_gateway_api_key":                                  awsApigatewayv2Api,
		"aws_api_gateway_authorizer":                               awsApigatewayv2Api,
		"aws_api_gateway_base_path_mapping":                        awsApigatewayv2Api,
		"aws_api_gateway_client_certificate":                       awsAPIGatewayRestAPI,
		"aws_api_gateway_deployment":                               awsApigatewayv2Api,
		"aws_api_gateway_documentation_part":                       awsApigatewayv2Api,
		"aws_api_gateway_documentation_version":                    awsApigatewayv2Api,
		"aws_api_gateway_domain_name":                              awsApigatewayv2Api,
		"aws_api_gateway_gateway_response":                         awsApigatewayv2Api,
		"aws_api_gateway_integration":                              awsApigatewayv2Api,
		"aws_api_gateway_integration_response":                     awsApigatewayv2Api,
		"aws_api_gateway_method":                                   awsApigatewayv2Api,
		"aws_api_gateway_method_response":                          awsApigatewayv2Api,
		"aws_api_gateway_method_settings":                          awsApigatewayv2Api,
		"aws_api_gateway_model":                                    awsApigatewayv2Api,
		"aws_api_gateway_request_validator":                        awsAPIGatewayRestAPI,
		"aws_api_gateway_resource":                                 awsApigatewayv2Api,
		"aws_api_gateway_rest_api":                                 awsAPIGatewayRestAPI,
		"aws_api_gateway_rest_api_policy":                          awsApigatewayv2Api,
		"aws_api_gateway_stage":                                    awsApigatewayv2Api,
		"aws_api_gateway_usage_plan":                               awsApigatewayv2Api,
		"aws_api_gateway_usage_plan_key":                           awsApigatewayv2Api,
		"aws_api_gateway_vpc_link":                                 awsApigatewayv2Api,
		"aws_apigatewayv2_api":                                     awsApigatewayv2Api,
		"aws_apigatewayv2_api_mapping":                             awsApigatewayv2Api,
		"aws_apigatewayv2_authorizer":                              awsApigatewayv2Api,
		"aws_apigatewayv2_deployment":                              awsApigatewayv2Api,
		"aws_apigatewayv2_domain_name":                             awsApigatewayv2Api,
		"aws_apigatewayv2_integration":                             awsApigatewayv2Api,
		"aws_apigatewayv2_integration_response":                    awsApigatewayv2Api,
		"aws_apigatewayv2_model":                                   awsApigatewayv2Api,
		"aws_apigatewayv2_route":                                   awsApigatewayv2Api,
		"aws_apigatewayv2_route_response":                          awsApigatewayv2Api,
		"aws_apigatewayv2_stage":                                   awsApigatewayv2Api,
		"aws_apigatewayv2_vpc_link":                                awsApigatewayv2Api,
		"aws_app_cookie_stickiness_policy":                         awsAppCookieStickinessPolicy,
		"aws_appautoscaling_policy":                                awsAppautoscalingPolicy,
		"aws_appautoscaling_scheduled_action":                      awsAppautoscalingScheduledAction,
		"aws_appautoscaling_target":                                awsAppautoscalingTarget,
		"aws_appconfig_application":                                awsAppconfigApplication,
		"aws_appconfig_configuration_profile":                      awsAppconfigConfigurationProfile,
		"aws_applicationinsights_application":                      awsApplicationinsightsApplication,
		"aws_athena_data_catalog":                                  awsAthenaDataCatalog,
		"aws_athena_database":                                      awsAthenaDatabase,
		"aws_athena_named_query":                                   awsAthenaNamedQuery,
		"aws_athena_workgroup":                                     awsAthenaWorkgroup,
		"aws_autoscaling_attachment":                               awsAutoscalingAttachment,
		"aws_autoscaling_group":                                    awsAutoscalingGroup,
		"aws_autoscaling_lifecycle_hook":                           awsAutoscalingLifecycleHook,
		"aws_autoscaling_notification":                             awsAutoscalingNotification,
		"aws_autoscaling_policy":                                   awsAutoscalingPolicy,
		"aws_backup_framework":                                     awsBackupFramework,
		"aws_backup_global_settings":                               awsBackupGlobalSettings,
		"aws_backup_plan":                                          awsBackupPlan,
		"aws_backup_region_settings":                               awsBackupRegionSettings,
		"aws_backup_report_plan":                                   awsBackupReportPlan,
		"aws_backup_selection":                                     awsBackupSelection,
		"aws_backup_vault":                                         awsBackupVault,
		"aws_backup_vault_lock_configuration":                      awsBackupVaultLockConfiguration,
		"aws_backup_vault_notifications":                           awsBackupVaultNotification,
		"aws_backup_vault_policy":                                  awsBackupVaultPolicy,
		"aws_batch_compute_environment":                            awsBatchComputeEnvironment,
		"aws_batch_job_definition":                                 awsBatchJobDefinition,
		"aws_batch_job_queue":                                      awsBatchJobQueue,
		"aws_batch_scheduling_policy":                              awsBatchSchedulingPolicy,
		"aws_budgets_budget":                                       awsBudgetsBudget,
		"aws_budgets_budget_action":                                awsBudgetsBudgetAction,
		"aws_ce_anomaly_monitor":                                   awsCeAnomalyMonitor,
		"aws_ce_anomaly_subscription":                              awsCeAnomalySubscription,
		"aws_ce_cost_allocation_tag":                               awsCeCostAllocationTag,
		"aws_ce_cost_category":                                     awsCeCostCategory,
		"aws_cloud9_environment_ec2":                               awsCloud9EnvironmentEc2,
		"aws_cloudformation_stack":                                 awsCloudformationStack,
		"aws_cloudformation_stack_set":                             awsCloudFormationStackSet,
		"aws_cloudformation_stack_set_instance":                    awsCloudFormationStackSetInstance,
		"aws_cloudformation_type":                                  awsCloudformationType,
		"aws_cloudfront_cache_policy":                              awsCloudfrontCachePolicy,
		"aws_cloudfront_continuous_deployment_policy":              awsCloudfrontContinuousDeploymentPolicy,
		"aws_cloudfront_distribution":                              awsCloudfrontDistribution,
		"aws_cloudfront_field_level_encryption_config":             awsCloudfrontFieldLevelEncryptionConfig,
		"aws_cloudfront_field_level_encryption_profile":            awsCloudfrontFieldLevelEncryptionProfile,
		"aws_cloudfront_function":                                  awsCloudfrontFunction,
		"aws_cloudfront_key_group":                                 awsCloudfrontKeyGroup,
		"aws_cloudfront_monitoring_subscription":                   awsCloudfrontMonitoringSubscription,
		"aws_cloudfront_origin_access_control":                     awsCloudfrontOriginAccessControl,
		"aws_cloudfront_origin_access_identity":                    awsCloudfrontOriginAccessIdentity,
		"aws_cloudfront_origin_request_policy":                     awsCloudfrontOriginRequestPolicy,
		"aws_cloudfront_public_key":                                awsCloudfrontPublicKey,
		"aws_cloudfront_response_headers_policy":                   awsCloudfrontResponseHeadersPolicy,
		"aws_cloudsearch_domain":                                   awsCloudsearchDomain,
		"aws_cloudsearch_domain_service_access_policy":             awsCloudsearchDomainServiceAccessPolicy,
		"aws_cloudtrail":                                           awsCloudtrail,
		"aws_cloudtrail_event_data_store":                          awsCloudtrailEventDataStore,
		"aws_cloudwatch_composite_alarm":                           awsCloudwatchCompositeAlarm,
		"aws_cloudwatch_dashboard":                                 awsCloudwatchDashboard,
		"aws_cloudwatch_event_api_destination":                     awsCloudwatchEventAPIDestination,
		"aws_cloudwatch_event_archive":                             awsCloudwatchEventArchive,
		"aws_cloudwatch_event_bus":                                 awsCloudwatchEventBus,
		"aws_cloudwatch_event_bus_policy":                          awsCloudwatchEventBusPolicy,
		"aws_cloudwatch_event_connection":                          awsCloudwatchEventConnection,
		"aws_cloudwatch_event_endpoint":                            awsCloudwatchEventEndpoint,
		"aws_cloudwatch_event_permission":                          awsCloudwatchEventPermission,
		"aws_cloudwatch_event_rule":                                awsCloudwatchEventRule,
		"aws_cloudwatch_event_target":                              awsCloudwatchEventTarget,
		"aws_cloudwatch_log_data_protection_policy":                awsCloudwatchLogDataProtectionPolicy,
		"aws_cloudwatch_log_destination":                           awsCloudwatchLogDestination,
		"aws_cloudwatch_log_destination_policy":                    awsCloudwatchLogDestinationPolicy,
		"aws_cloudwatch_log_group":                                 awsCloudwatchLogGroup,
		"aws_cloudwatch_log_metric_filter":                         awsCloudwatchLogMetricFilter,
		"aws_cloudwatch_log_resource_policy":                       awsCloudwatchLogResourcePolicy,
		"aws_cloudwatch_log_stream":                                awsCloudwatchLogStream,
		"aws_cloudwatch_log_subscription_filter":                   awsCloudwatchLogSubscriptionFilter,
		"aws_cloudwatch_metric_alarm":                              awsCloudwatchMetricAlarm,
		"aws_cloudwatch_metric_stream":                             awsCloudwatchMetricStream,
		"aws_cloudwatch_query_definition":                          awsCloudwatchQueryDestination,
		"aws_codeartifact_domain":                                  awsCodeartifactDomain,
		"aws_codeartifact_domain_permissions_policy":               awsCodeartifactDomainPermissionsPolicy,
		"aws_codeartifact_repository":                              awsCodeartifactRepository,
		"aws_codeartifact_repository_permissions_policy":           awsCodeartifactRepositoryPermissionsPolicy,
		"aws_codebuild_project":                                    awsCodebuildProject,
		"aws_codebuild_report_group":                               awsCodebuildReportGroup,
		"aws_codebuild_resource_policy":                            awsCodebuildResourcePolicy,
		"aws_codebuild_source_credential":                          awsCodebuildSourceCredential,
		"aws_codebuild_webhook":                                    awsCodebuildWebhook,
		"aws_codecommit_approval_rule_template":                    awsCodecommitApprovalRuleTemplate,
		"aws_codecommit_approval_rule_template_association":        awsCodecommitApprovalRuleTemplateAssociation,
		"aws_codecommit_repository":                                awsCodecommitRepository,
		"aws_codecommit_trigger":                                   awsCodecommitTrigger,
		"aws_codedeploy_app":                                       awsCodedeployApp,
		"aws_codedeploy_deployment_config":                         awsCodedeployDeploymentConfig,
		"aws_codedeploy_deployment_group":                          awsCodedeployDeploymentGroup,
		"aws_codepipeline":                                         awsCodepipeline,
		"aws_cognito_identity_pool":                                awsCognitoIdentityPool,
		"aws_cognito_identity_provider":                            awsCognitoIdentityProvider,
		"aws_cognito_resource_server":                              awsCognitoResourceServer,
		"aws_cognito_risk_configuration":                           awsCognitoRiskConfiguration,
		"aws_cognito_user":                                         awsCognitoUser,
		"aws_cognito_user_group":                                   awsCognitoUserGroup,
		"aws_cognito_user_in_group":                                awsCognitoUserInGroup,
		"aws_cognito_user_pool":                                    awsCognitoUserPool,
		"aws_cognito_user_pool_client":                             awsCognitoUserPoolClient,
		"aws_cognito_user_pool_domain":                             awsCognitoUserPoolDomain,
		"aws_cognito_user_pool_ui_customization":                   awsCognitoUserPoolUICustomization,
		"aws_config_config_rule":                                   awsConfigConfigRule,
		"aws_config_configuration_aggregator":                      awsConfigConfigurationAggregator,
		"aws_config_configuration_recorder":                        awsConfigConfigurationRecorder,
		"aws_config_configuration_recorder_status":                 awsConfigConfigurationRecorderStatus,
		"aws_config_delivery_channel":                              awsConfigDeliveryChannel,
		"aws_customer_gateway":                                     awsCustomerGateway,
		"aws_datapipeline_pipeline":                                awsDatapipelinePipeline,
		"aws_datapipeline_pipeline_definition":                     awsDatapipelinePipelineDefinition,
		"aws_dax_cluster":                                          awsDaxCluster,
		"aws_dax_parameter_group":                                  awsDaxParameterGroup,
		"aws_dax_subnet_group":                                     awsDaxSubnetGroup,
		"aws_db_cluster_snapshot":                                  awsDBClusterSnapshot,
		"aws_db_event_subscription":                                awsDBEventSubscription,
		"aws_db_instance":                                          awsDBInstance,
		"aws_db_option_group":                                      awsDBOptionGroup,
		"aws_db_parameter_group":                                   awsDBParameterGroup,
		"aws_db_subnet_group":                                      awsDBSubnetGroup,
		"aws_default_network_acl":                                  awsDefaultNetworkACL,
		"aws_default_route_table":                                  awsDefaultRouteTable,
		"aws_default_security_group":                               awsDefaultSecurityGroup,
		"aws_default_subnet":                                       awsDefaultSubnet,
		"aws_default_vpc":                                          awsDefaultVpc,
		"aws_default_vpc_dhcp_options":                             awsDefaultVpcDhcpOptions,
		"aws_directory_service_directory":                          awsDirectoryServiceDirectory,
		"aws_directory_service_log_subscription":                   awsDirectoryServiceLogSubscription,
		"aws_dlm_lifecycle_policy":                                 awsDlmLifecyclePolicy,
		"aws_docdb_cluster":                                        awsRdsCluster,
		"aws_docdb_cluster_instance":                               awsNeptuneClusterInstance,
		"aws_docdb_cluster_parameter_group":                        awsRdsClusterParameterGroup,
		"aws_docdb_cluster_snapshot":                               awsDBClusterSnapshot,
		"aws_docdb_event_subscription":                             awsNeptuneEventSubscription,
		"aws_docdb_global_cluster":                                 awsRdsGlobalCluster,
		"aws_docdb_subnet_group":                                   awsDBSubnetGroup,
		"aws_dx_gateway":                                           awsDxGateway,
		"aws_dx_gateway_association":                               awsDxGatewayAssociation,
		"aws_dx_hosted_transit_virtual_interface_accepter":         awsDxHostedTransitVirtualInterfaceAccepter,
		"aws_dynamodb_contributor_insights":                        awsDynamodbContributorInsights,
		"aws_dynamodb_global_table":                                awsDynamodbGlobalTable,
		"aws_dynamodb_table":                                       awsDynamodbTable,
		"aws_dynamodb_table_item":                                  awsDynamodbTableItem,
		"aws_dynamodb_tag":                                         awsDynamodbTag,
		"aws_ebs_default_kms_key":                                  awsEbsDefaultKmsKey,
		"aws_ebs_encryption_by_default":                            awsEbsEncryptionByDefault,
		"aws_ebs_snapshot":                                         awsEbsSnapshot,
		"aws_ebs_snapshot_copy":                                    awsEbsSnapshotCopy,
		"aws_ebs_volume":                                           awsEbsVolume,
		"aws_ec2_availability_zone_group":                          awsEc2AvailabilityGroup,
		"aws_ec2_capacity_reservation":                             awsEc2CapacityReservation,
		"aws_ec2_network_insights_analysis":                        awsEc2NetworkInsightsAnalysis,
		"aws_ec2_network_insights_path":                            awsEc2NetworkInsightsPath,
		"aws_ec2_tag":                                              awsEc2Tag,
		"aws_ec2_transit_gateway":                                  awsEc2TransitGateway,
		"aws_ec2_transit_gateway_route":                            awsEc2TransitGatewayRoute,
		"aws_ec2_transit_gateway_route_table":                      awsEc2TransitGatewayRouteTable,
		"aws_ec2_transit_gateway_route_table_association":          awsEc2TransitGatewayRouteTableAssociation,
		"aws_ec2_transit_gateway_route_table_propagation":          awsEc2TransitGatewayRouteTablePropagation,
		"aws_ec2_transit_gateway_vpc_attachment":                   awsEc2TransitGatewayVpcAttachment,
		"aws_ecr_lifecycle_policy":                                 awsEcrLifecyclePolicy,
		"aws_ecr_pull_through_cache_rule":                          awsEcrPullThroughCacheRule,
		"aws_ecr_registry_policy":                                  awsEcrRegistryPolicy,
		"aws_ecr_registry_scanning_configuration":                  awsEcrRegistryScanningConfiguration,
		"aws_ecr_replication_configuration":                        awsEcrReplicationConfiguration,
		"aws_ecr_repository":                                       awsEcrRepository,
		"aws_ecr_repository_policy":                                awsEcrRepositoryPolicy,
		"aws_ecrpublic_repository":                                 awsEcrPublicRepository,
		"aws_ecrpublic_repository_policy":                          awsEcrpublicRepositoryPolicy,
		"aws_ecs_cluster":                                          awsEcsCluster,
		"aws_ecs_service":                                          awsEcsService,
		"aws_ecs_task_definition":                                  awsEcsTaskDefinition,
		"aws_efs_access_point":                                     awsEfsAccessPoint,
		"aws_efs_backup_policy":                                    awsEfsBackupPolicy,
		"aws_efs_file_system":                                      awsEfsFileSystem,
		"aws_efs_file_system_policy":                               awsEfsFileSystemPolicy,
		"aws_efs_mount_target":                                     awsEfsMountTarget,
		"aws_efs_replication_configuration":                        awsEfsReplicationConfiguration,
		"aws_egress_only_internet_gateway":                         awsEgressOnlyInternetGateway,
		"aws_eip":                                                  awsEip,
		"aws_eip_association":                                      awsEipAssociation,
		"aws_eks_addon":                                            awsEksAddon,
		"aws_eks_cluster":                                          awsEksCluster,
		"aws_eks_node_group":                                       awsEksNodeGroup,
		"aws_elastic_beanstalk_application":                        awsElasticBeanstalkApplication,
		"aws_elastic_beanstalk_application_version":                awsElasticBeanstalkApplicationVersion,
		"aws_elastic_beanstalk_configuration_template":             awsElasticBeanstalkConfigurationTemplate,
		"aws_elastic_beanstalk_environment":                        awsElastiBeanstalkEnvironment,
		"aws_elasticache_cluster":                                  awsElasticacheCluster,
		"aws_elasticache_parameter_group":                          awsElasticacheParameterGroup,
		"aws_elasticache_replication_group":                        awsElasticacheReplicationGroup,
		"aws_elasticache_subnet_group":                             awsElasticacheSubnetGroup,
		"aws_elasticache_user":                                     awsElasticacheUser,
		"aws_elasticache_user_group":                               awsElasticacheUserGroup,
		"aws_elasticsearch_domain":                                 awsElasticsearchDomain,
		"aws_elasticsearch_domain_policy":                          awsElasticsearchDomainPolicy,
		"aws_elb":                                                  awsElb,
		"aws_emr_cluster":                                          awsEmrCluster,
		"aws_emr_managed_scaling_policy":                           awsEmrManagedScalingPolicy,
		"aws_emr_security_configuration":                           awsEmrSecurityConfiguration,
		"aws_flow_log":                                             awsFlowLog,
		"aws_fsx_openzfs_file_system":                              awsFsxOpenzfsFileSystem,
		"aws_fsx_openzfs_snapshot":                                 awsFsxOpenzfsSnaphot,
		"aws_fsx_openzfs_volume":                                   awsFsxOpenzfsVolume,
		"aws_glacier_vault":                                        awsGlacierVault,
		"aws_glacier_vault_lock":                                   awsGlacierVaultLock,
		"aws_glue_catalog_database":                                awsGlueCatalogDatabase,
		"aws_glue_catalog_table":                                   awsGlueCatalogTable,
		"aws_glue_classifier":                                      awsGlueClassifier,
		"aws_glue_connection":                                      awsGlueConnection,
		"aws_glue_crawler":                                         awsGlueCrawler,
		"aws_glue_data_catalog_encryption_settings":                awsGlueDataCatalogEncryptionSettings,
		"aws_glue_job":                                             awsGlueJob,
		"aws_glue_ml_transform":                                    awsGlueMlTransform,
		"aws_glue_registry":                                        awsGlueRegistry,
		"aws_glue_resource_policy":                                 awsGlueResourcePolicy,
		"aws_glue_schema":                                          awsGlueSchema,
		"aws_glue_security_configuration":                          awsGlueSecurityConfiguration,
		"aws_glue_trigger":                                         awsGlueTrigger,
		"aws_glue_user_defined_function":                           awsGlueUserDefinedFunction,
		"aws_glue_workflow":                                        awsGlueWorkflow,
		"aws_grafana_workspace_api_key":                            awsGrafanaWorkspaceAPIKey,
		"aws_iam_access_key":                                       awsIamAccessKey,
		"aws_iam_account_alias":                                    awsIamAccountAlias,
		"aws_iam_account_password_policy":                          awsIamAccountPasswordPolicy,
		"aws_iam_group":                                            awsIamGroup,
		"aws_iam_group_membership":                                 awsIamGroupMembership,
		"aws_iam_group_policy":                                     awsIamGroupPolicy,
		"aws_iam_group_policy_attachment":                          awsIamGroupPolicyAttachment,
		"aws_iam_instance_profile":                                 awsIamInstanceProfile,
		"aws_iam_openid_connect_provider":                          awsIamOpenidConnectProvider,
		"aws_iam_policy":                                           awsIamPolicy,
		"aws_iam_policy_attachment":                                awsIamPolicyAttachment,
		"aws_iam_role":                                             awsIamRole,
		"aws_iam_role_policy":                                      awsIamRolePolicy,
		"aws_iam_role_policy_attachment":                           awsIamRolePolicyAttachment,
		"aws_iam_saml_provider":                                    awsIamSamlProvider,
		"aws_iam_security_token_service_preferences":               awsIamSecurityTokenServicePreferences,
		"aws_iam_server_certificate":                               awsIamServerCertificate,
		"aws_iam_service_linked_role":                              awsIamServiceLinkedRole,
		"aws_iam_service_specific_credential":                      awsIamServiceSpecificCredential,
		"aws_iam_signing_certificate":                              awsIamSigningCertificate,
		"aws_iam_user":                                             awsIamUser,
		"aws_iam_user_group_membership":                            awsIamUserGroupMembership,
		"aws_iam_user_login_profile":                               awsIamUserLoginProfile,
		"aws_iam_user_policy":                                      awsIamUserPolicy,
		"aws_iam_user_policy_attachment":                           awsIamUserPolicyAttachment,
		"aws_iam_user_ssh_key":                                     awsIamUserSSHKey,
		"aws_iam_virtual_mfa_device":                               awsIamVirtualMfaDevice,
		"aws_imagebuilder_component":                               awsImagebuilderComponent,
		"aws_imagebuilder_container_recipe":                        awsImagebuilderContainerRecipe,
		"aws_imagebuilder_distribution_configuration":              awsImagebuilderDistributionConfiguration,
		"aws_imagebuilder_image":                                   awsImagebuilderImage,
		"aws_imagebuilder_image_pipeline":                          awsImagebuilderImagePipeline,
		"aws_imagebuilder_image_recipe":                            awsImagebuilderImageRecipe,
		"aws_imagebuilder_infrastructure_configuration":            awsImagebuilderInstrastructureConfiguration,
		"aws_inspector2_delegated_admin_account":                   awsInspector2DelegatedAdminAccount,
		"aws_inspector2_enabler":                                   awsInspector2Enabler,
		"aws_inspector2_member_association":                        awsInspector2MemberAssociation,
		"aws_inspector2_organization_configuration":                awsInspector2OrganizationConfiguration,
		"aws_inspector_assessment_target":                          awsInspectorAssessmentTarget,
		"aws_inspector_assessment_template":                        awsInspectorAssessmentTemplate,
		"aws_inspector_resource_group":                             awsInspectorResouceGroup,
		"aws_instance":                                             awsInstance,
		"aws_internet_gateway":                                     awsInternetGateway,
		"aws_internetmonitor_monitor":                              awsInternetmonitorMonitor,
		"aws_key_pair":                                             awsKeyPair,
		"aws_kinesis_firehose_delivery_stream":                     awsKinesisFirehoseDeliveryStream,
		"aws_kinesis_stream":                                       awsKinesisStream,
		"aws_kinesis_stream_consumer":                              awsKinesisStreamConsumer,
		"aws_kinesis_video_stream":                                 awsKinesisVideoStream,
		"aws_kms_alias":                                            awsKmsAlias,
		"aws_kms_ciphertext":                                       awsKmsCiphertext,
		"aws_kms_custom_key_store":                                 awsKmsCustomKeyStore,
		"aws_kms_external_key":                                     awsKmsExternalKey,
		"aws_kms_grant":                                            awsKmsGrant,
		"aws_kms_key":                                              awsKmsKey,
		"aws_kms_key_policy":                                       awsKmsKeyPolicy,
		"aws_kms_replica_external_key":                             awsKmsReplicaExternalKey,
		"aws_kms_replica_key":                                      awsKmsReplicaKey,
		"aws_lambda_alias":                                         awsLambdaAlias,
		"aws_lambda_code_signing_config":                           awsLambdaCodeSigningConfig,
		"aws_lambda_event_source_mapping":                          awsLambdaEventSourceMapping,
		"aws_lambda_function":                                      awsLambdaFunction,
		"aws_lambda_function_event_invoke_config":                  awsLambdaFunctionEventInvokeConfig,
		"aws_lambda_function_url":                                  awsLambdaFunctionURL,
		"aws_lambda_invocation":                                    awsLambdaInvocation,
		"aws_lambda_layer_version":                                 awsLambdaLayerVersion,
		"aws_lambda_layer_version_permission":                      awsLambdaLayerVersionPermission,
		"aws_lambda_permission":                                    awsLambdaPermission,
		"aws_lambda_provisioned_concurrency_config":                awsLambdaProvisionedConcurrencyConfig,
		"aws_launch_configuration":                                 awsLaunchConfiguration,
		"aws_launch_template":                                      awsLaunchTemplate,
		"aws_lb":                                                   awsLb,
		"aws_lb_cookie_stickiness_policy":                          awsLbCookieStickinessPolicy,
		"aws_lb_listener":                                          awsLbListener,
		"aws_lb_listener_rule":                                     awsLbListenerRule,
		"aws_lb_target_group":                                      awsLbTargetGroup,
		"aws_lb_target_group_attachment":                           awsLbTargetGroupAttachment,
		"aws_lightsail_instance":                                   awsLightsailInstance,
		"aws_lightsail_instance_public_ports":                      awsLightsailInstancePublicPorts,
		"aws_lightsail_key_pair":                                   awsLightsailKeyPair,
		"aws_lightsail_static_ip":                                  awsLightsailStaticIP,
		"aws_lightsail_static_ip_attachment":                       awsLightsailStaticIPAttachment,
		"aws_load_balancer_backend_server_policy":                  awsLoadBalancerBackendServerPolicy,
		"aws_load_balancer_listener_policy":                        awsLoadBalancerListenerPolicy,
		"aws_load_balancer_policy":                                 awsLoadBalancerPolicy,
		"aws_location_geofence_collection":                         awsLocationGeofenceCollection,
		"aws_location_map":                                         awsLocationMap,
		"aws_location_place_index":                                 awsLocationPlaceIndex,
		"aws_location_route_calculator":                            awsLocationRouteCalculator,
		"aws_location_tracker":                                     awsLocationTracker,
		"aws_location_tracker_association":                         awsLocationTrackerAssociation,
		"aws_media_convert_queue":                                  awsMediaConvertQueue,
		"aws_medialive_input":                                      awsMedialiveInput,
		"aws_medialive_input_security_group":                       awsMedialiveInputSecurityGroup,
		"aws_memorydb_cluster":                                     awsMemorydbCluster,
		"aws_memorydb_snapshot":                                    awsMemorydbSnapshot,
		"aws_memorydb_subnet_group":                                awsMemorydbSubnetGroup,
		"aws_mq_broker":                                            awsMqBroker,
		"aws_mq_configuration":                                     awsMqConfiguration,
		"aws_msk_cluster":                                          awsMskCluster,
		"aws_msk_configuration":                                    awsMskConfiguration,
		"aws_msk_scram_secret_association":                         awsMskScramSecretAssociation,
		"aws_msk_serverless_cluster":                               awsMskServerlessCluster,
		"aws_nat_gateway":                                          awsNatGateway,
		"aws_neptune_cluster":                                      awsNeptuneCluster,
		"aws_neptune_cluster_endpoint":                             awsNeptuneClusterEndpoint,
		"aws_neptune_cluster_instance":                             awsNeptuneClusterInstance,
		"aws_neptune_cluster_parameter_group":                      awsRdsClusterParameterGroup,
		"aws_neptune_cluster_snapshot":                             awsNeptuneClusterSnapshot,
		"aws_neptune_event_subscription":                           awsNeptuneEventSubscription,
		"aws_neptune_parameter_group":                              awsDBParameterGroup,
		"aws_neptune_subnet_group":                                 awsDBSubnetGroup,
		"aws_network_acl":                                          awsNetworkACL,
		"aws_network_acl_rule":                                     awsNetworkACLRule,
		"aws_network_interface":                                    awsNetworkInterface,
		"aws_networkfirewall_firewall":                             awsNetworkfirewallFirewall,
		"aws_networkfirewall_firewall_policy":                      awsNetworkfirewallFirewallPolicy,
		"aws_networkfirewall_logging_configuration":                awsNetworkfirewallLoggingConfiguration,
		"aws_networkfirewall_resource_policy":                      awsNetworkfirewallResourcePolicy,
		"aws_networkfirewall_rule_group":                           awsNetworkfirewallRuleGroup,
		"aws_opensearch_domain":                                    awsElasticsearchDomain,
		"aws_opensearch_domain_policy":                             awsElasticsearchDomainPolicy,
		"aws_opensearchserverless_access_policy":                   awsOpenseachserverlessAccessPolicy,
		"aws_opensearchserverless_collection":                      awsOpenseachserverlessCollection,
		"aws_opensearchserverless_lifecycle_policy":                awsOpenseachserverlessLifecyclePolicy,
		"aws_opensearchserverless_security_config":                 awsOpenseachserverlessSecurityConfig,
		"aws_opensearchserverless_security_policy":                 awsOpenseachserverlessSecurityPolicy,
		"aws_opensearchserverless_vpc_endpoint":                    awsOpenseachserverlessVpcEndpoint,
		"aws_organizations_policy":                                 awsOrganizationsPolicy,
		"aws_organizations_policy_attachment":                      awsOrganizationsPolicyAttachment,
		"aws_placement_group":                                      awsPlacementGroup,
		"aws_proxy_protocol_policy":                                awsProxyProtocolPolicy,
		"aws_ram_principal_association":                            awsRAMPrincipleAssociation,
		"aws_ram_resource_association":                             awsRAMResourceAssociation,
		"aws_ram_resource_share":                                   awsRAMResourceShare,
		"aws_rds_cluster":                                          awsRdsCluster,
		"aws_rds_cluster_activity_stream":                          awsRdsClusterActivityStream,
		"aws_rds_cluster_endpoint":                                 awsRdsClusterEndpoint,
		"aws_rds_cluster_instance":                                 awsDBInstance,
		"aws_rds_cluster_parameter_group":                          awsRdsClusterParameterGroup,
		"aws_rds_cluster_role_association":                         awsRdsClusterRoleAssociation,
		"aws_rds_global_cluster":                                   awsRdsGlobalCluster,
		"aws_redshift_authentication_profile":                      awsRedshiftAuthenticationProfile,
		"aws_redshift_cluster":                                     awsRedshiftCluster,
		"aws_redshift_cluster_iam_roles":                           awsRedshiftClusterIamRoles,
		"aws_redshift_event_subscription":                          awsRedshiftEventSubscription,
		"aws_redshift_hsm_client_certificate":                      awsRedshiftHsmClientCertififcate,
		"aws_redshift_hsm_configuration":                           awsRedshiftHsmConfiguration,
		"aws_redshift_parameter_group":                             awsRedshiftParameterGroup,
		"aws_redshift_scheduled_action":                            awsRedshiftScheduledAction,
		"aws_redshift_snapshot_copy_grant":                         awsRedshiftSnapshotCopyGrant,
		"aws_redshift_snapshot_schedule":                           awsRedshiftSnapshotSchedule,
		"aws_redshift_snapshot_schedule_association":               awsRedshiftSnapshotScheduleAssociation,
		"aws_redshift_subnet_group":                                awsRedshiftSubnetGroup,
		"aws_redshift_usage_limit":                                 awsRedshiftUsageLimit,
		"aws_resourcegroups_group":                                 awsResourcegroupsGroup,
		"aws_route":                                                awsRoute,
		"aws_route53_cidr_collection":                              awsRoute53CidrCollection,
		"aws_route53_cidr_location":                                awsRoute53CidrLocation,
		"aws_route53_delegation_set":                               awsRoute53DelegationSet,
		"aws_route53_health_check":                                 awsRoute53HealthCheck,
		"aws_route53_hosted_zone_dnssec":                           awsRoute53HostedZoneDnssec,
		"aws_route53_key_signing_key":                              awsRoute53KeySiginingKey,
		"aws_route53_query_log":                                    awsRoute53QueryLog,
		"aws_route53_record":                                       awsRoute53Record,
		"aws_route53_resolver_config":                              awsRoute53ResolverConfig,
		"aws_route53_resolver_dnssec_config":                       awsRoute53ResolverDnssecConfig,
		"aws_route53_resolver_endpoint":                            awsRoute53ResolverEndpoint,
		"aws_route53_resolver_firewall_config":                     awsRoute53ResolverFirewallConfig,
		"aws_route53_resolver_firewall_domain_list":                awsRoute53ResolverFirewallDomainList,
		"aws_route53_resolver_firewall_rule":                       awsRoute53ResolverFirewallRule,
		"aws_route53_resolver_firewall_rule_group":                 awsRoute53ResolverFirewallRuleGroup,
		"aws_route53_resolver_firewall_rule_group_association":     awsRoute53ResolverFirewallRuleGroupAssociation,
		"aws_route53_resolver_query_log_config":                    awsRoute53ResolverQueryLogConfig,
		"aws_route53_resolver_query_log_config_association":        awsRoute53ResolverQueryLogConfigAssociation,
		"aws_route53_resolver_rule":                                awsRoute53ResolverRule,
		"aws_route53_resolver_rule_association":                    awsRoute53ResolverRuleAssociation,
		"aws_route53_traffic_policy":                               awsRoute53TrafficPolicy,
		"aws_route53_traffic_policy_instance":                      awsRoute53TrafficPolicyInstance,
		"aws_route53_vpc_association_authorization":                awsRoute53VpcAssociationAuthorization,
		"aws_route53_zone":                                         awsRoute53Zone,
		"aws_route53_zone_association":                             awsRoute53ZoneAssociation,
		"aws_route_table":                                          awsRouteTable,
		"aws_route_table_association":                              awsRouteTableAssociation,
		"aws_s3_bucket":                                            awsS3Bucket,
		"aws_s3_bucket_accelerate_configuration":                   awsS3BucketAccelerateConfiguration,
		"aws_s3_bucket_acl":                                        awsS3BucketACL,
		"aws_s3_bucket_cors_configuration":                         awsS3BucketCorsConfiguration,
		"aws_s3_bucket_intelligent_tiering_configuration":          awsS3BucketIntelligentTieringConfiguration,
		"aws_s3_bucket_inventory":                                  awsS3BucketInventory,
		"aws_s3_bucket_lifecycle_configuration":                    awsS3BucketLifecycleConfiguration,
		"aws_s3_bucket_logging":                                    awsS3BucketLogging,
		"aws_s3_bucket_metric":                                     awsS3BucketMetric,
		"aws_s3_bucket_notification":                               awsS3BucketNotification,
		"aws_s3_bucket_object":                                     awsS3Object,
		"aws_s3_bucket_object_lock_configuration":                  awsS3BucketObjectLockCOnfiguration,
		"aws_s3_bucket_ownership_controls":                         awsS3BucketOwnershipControls,
		"aws_s3_bucket_policy":                                     awsS3BucketPolicy,
		"aws_s3_bucket_public_access_block":                        awsS3BucketPublicAccessBlock,
		"aws_s3_bucket_replication_configuration":                  awsS3BucketReplicationConfiguration,
		"aws_s3_bucket_request_payment_configuration":              awsS3BucketRequestPaymentConfiguration,
		"aws_s3_bucket_server_side_encryption_configuration":       awsS3BucketServerSideEncryptionConfiguration,
		"aws_s3_bucket_versioning":                                 awsS3BucketVersioning,
		"aws_s3_bucket_website_configuration":                      awsS3BucketWebsiteConfiguration,
		"aws_s3_object":                                            awsS3Object,
		"aws_sagemaker_endpoint_configuration":                     awsSagemakerEndpointConfiguration,
		"aws_sagemaker_model":                                      awsSagemakerModel,
		"aws_secretsmanager_secret":                                awsSecretsmanagerSecret,
		"aws_secretsmanager_secret_policy":                         awsSecretsmanagerSecretPolicy,
		"aws_secretsmanager_secret_rotation":                       awsSecretsmanagerSecretRotation,
		"aws_secretsmanager_secret_version":                        awsSecretsmanagerSecretVersion,
		"aws_security_group":                                       awsSecurityGroup,
		"aws_security_group_rule":                                  awsSecurityGroupRule,
		"aws_servicecatalog_portfolio":                             awsServicecatalogPortfolio,
		"aws_servicequotas_service_quota":                          awsServicequotasServiceQuota,
		"aws_ses_domain_dkim":                                      awsSesDomainDkim,
		"aws_ses_domain_identity":                                  awsSesDomainIdentity,
		"aws_ses_domain_identity_verification":                     awsSesDomainIdentityVerification,
		"aws_ses_domain_mail_from":                                 awsSesDomainMailFrom,
		"aws_ses_identity_notification_topic":                      awsSesIdentityNotificationTopic,
		"aws_ses_identity_policy":                                  awsSesIdentityPolicy,
		"aws_ses_receipt_rule":                                     awsSesReceiptRule,
		"aws_ses_receipt_rule_set":                                 awsSesReceiptRuleSet,
		"aws_sfn_activity":                                         awsSfnActivity,
		"aws_sfn_alias":                                            awsSfnAlias,
		"aws_sfn_state_machine":                                    awsSfnStateMachine,
		"aws_signer_signing_job":                                   awsSignerSigningJob,
		"aws_signer_signing_profile":                               awsSignerSigningProfile,
		"aws_signer_signing_profile_permission":                    awsSignerSigningProfilePermission,
		"aws_sns_sms_preferences":                                  awsSnsSmsPreferences,
		"aws_sns_topic":                                            awsSnsTopic,
		"aws_sns_topic_data_protection_policy":                     awsSnsTopicDataProtection,
		"aws_sns_topic_policy":                                     awsSnsTopicPolicy,
		"aws_sns_topic_subscription":                               awsSnsTopicSubscription,
		"aws_spot_instance_request":                                awsSpotInstanceRequest,
		"aws_sqs_queue":                                            awsSqsQueue,
		"aws_sqs_queue_policy":                                     awsSqsQueuePolicy,
		"aws_sqs_queue_redrive_allow_policy":                       awsSqsQueueRedriveAllowPolicy,
		"aws_sqs_queue_redrive_policy":                             awsSqsQueueRedrivePolicy,
		"aws_ssm_document":                                         awsSsmDocument,
		"aws_ssm_maintenance_window":                               awsSsmMaintenanceWindow,
		"aws_ssm_maintenance_window_target":                        awsSsmMaintenanceWindowTarget,
		"aws_ssm_maintenance_window_task":                          awsSsmMaintenanceWindowTask,
		"aws_ssm_parameter":                                        awsSsmParameter,
		"aws_ssm_patch_baseline":                                   awsSsmPatchBaseline,
		"aws_ssm_patch_group":                                      awsSsmPatchGroup,
		"aws_ssmcontacts_contact":                                  awsSsmcontactsContact,
		"aws_ssmcontacts_contact_channel":                          awsSsmcontactContactChannel,
		"aws_ssmcontacts_plan":                                     awsSsmcontactsPlan,
		"aws_ssmincidents_replication_set":                         awsSsmincidentsReplicationSet,
		"aws_subnet":                                               awsSubnet,
		"aws_swf_domain":                                           awsSwfDomain,
		"aws_timestreamwrite_database":                             awsTimestreamwriteDatabase,
		"aws_timestreamwrite_table":                                awsTimestreamwriteTable,
		"aws_volume_attachment":                                    awsVolumeAttachment,
		"aws_vpc":                                                  awsVpc,
		"aws_vpc_dhcp_options":                                     awsVpcDhcpOptions,
		"aws_vpc_dhcp_options_association":                         awsVpcDhcpOptionsAssociation,
		"aws_vpc_endpoint":                                         awsVpcEndpoint,
		"aws_vpc_endpoint_route_table_association":                 awsVpcEndpointRouteTableAssociation,
		"aws_vpc_endpoint_service":                                 awsVpcEndpointService,
		"aws_vpc_endpoint_subnet_association":                      awsVpcEndpointSubnetAssociation,
		"aws_vpc_ipam":                                             awsVpcIpam,
		"aws_vpc_ipam_pool":                                        awsVpcIpamPool,
		"aws_vpc_ipam_pool_cidr":                                   awsVpcIpamPoolCidr,
		"aws_vpc_ipam_pool_cidr_allocation":                        awsVpcIpamPoolCidrAllocation,
		"aws_vpc_ipam_preview_next_cidr":                           awsVpcIpamPreviewNextCidr,
		"aws_vpc_ipam_resource_discovery":                          awsVpcIpamResourceDiscovery,
		"aws_vpc_ipam_resource_discovery_association":              awsVpcIpamResourceDiscoveryAssociation,
		"aws_vpc_ipam_scope":                                       awsVpcIpamScope,
		"aws_vpc_ipv4_cidr_block_association":                      awsVpcIpv4CidrBlockAssociation,
		"aws_vpc_peering_connection":                               awsVpcPeeringConnection,
		"aws_vpc_peering_connection_accepter":                      awsVpcPeeringConnectionAccepter,
		"aws_vpc_peering_connection_options":                       awsVpcPeeringConnectionOptions,
		"aws_vpc_security_group_egress_rule":                       awsVpcSecurityGroupEgressRule,
		"aws_vpc_security_group_ingress_rule":                      awsVpcSecurityGroupIngressRule,
		"aws_vpn_connection":                                       awsVpnConnection,
		"aws_vpn_gateway":                                          awsVpnGateway,
		"aws_vpn_gateway_attachment":                               awsVpnGatewayAttachment,
		"aws_vpn_gateway_route_propagation":                        awsVpnGatewayRoutePropagation,
		"aws_waf_byte_match_set":                                   awsWafByteMatchSet,
		"aws_waf_geo_match_set":                                    awsWafGeoMatchSet,
		"aws_waf_ipset":                                            awsWafIpset,
		"aws_waf_rate_based_rule":                                  awsWafRateBasedRule,
		"aws_waf_regex_match_set":                                  awsWafRegexMatchSet,
		"aws_waf_regex_pattern_set":                                awsWafRegexPatternSet,
		"aws_waf_rule":                                             awsWafRule,
		"aws_waf_rule_group":                                       awsWafRuleGroup,
		"aws_waf_size_constraint_set":                              awsWafSizeConstraintSet,
		"aws_waf_sql_injection_match_set":                          awsWafSqlInjectionMatchSet,
		"aws_waf_web_acl":                                          awsWafWebAcl,
		"aws_waf_xss_match_set":                                    awsWafXssNatchSet,
		"aws_wafregional_byte_match_set":                           awsWafregionalByteMatchSet,
		"aws_wafregional_geo_match_set":                            awsWafregionalGeoMatchSet,
		"aws_wafregional_ipset":                                    awsWafregionalIpset,
		"aws_wafregional_rate_based_rule":                          awsWafregionalRateBasedRule,
		"aws_wafregional_regex_match_set":                          awsWafregionalRegexMatchSet,
		"aws_wafregional_regex_pattern_set":                        awsWafregionalRegexPatternSet,
		"aws_wafregional_rule":                                     awsWafregionalRule,
		"aws_wafregional_rule_group":                               awsWafregionalRuleGroup,
		"aws_wafregional_size_constraint_set":                      awsWafregionalSizeConstraintSet,
		"aws_wafregional_sql_injection_match_set":                  awsWafregionalSqlInjectionMatchSet,
		"aws_wafregional_web_acl":                                  awsWafregionalWebAcl,
		"aws_wafregional_xss_match_set":                            awsWafregionalXssNatchSet,
		"aws_wafv2_ip_set":                                         awsWafv2IpSet,
		"aws_wafv2_regex_pattern_set":                              awsWafv2RegexPatternSet,
		"aws_wafv2_rule_group":                                     awsWafv2RuleGroup,
		"aws_wafv2_web_acl":                                        awsWafv2WebACL,
		"aws_wafv2_web_acl_association":                            awsWafv2WebACLAssociation,
		"aws_wafv2_web_acl_logging_configuration":                  awsWafv2WebACLLoggingConfiguration,
		"aws_workspaces_directory":                                 awsWorkspacesDirectory,
		"aws_workspaces_workspace":                                 awsWorkspacesWorkspace,
		"aws_xray_encryption_config":                               awsXrayEncryptionConfig,
		"aws_xray_group":                                           awsXrayGroup,
		"aws_xray_sampling_rule":                                   awsXraySamplingRule,
		"backend":                                                  s3backend,
		"aws_vpclattice_access_log_subscription":                   awsVpclatticeAccesLogSubscription,
		"aws_vpclattice_auth_policy":                               awsVpclatticeAuthPolicy,
		"aws_vpclattice_listener":                                  awsVpclatticeListener,
		"aws_vpclattice_listener_rule":                             awsVpclatticeListernerRule,
		"aws_vpclattice_resource_policy":                           awsVpclatticeResourcePolicy,
		"aws_vpclattice_service":                                   awsVpclatticeService,
		"aws_vpclattice_service_network":                           awsVpclatticeServiceNetwork,
		"aws_vpclattice_service_network_service_association":       awsVpclatticeServiceNetworkServiceAssocation,
		"aws_vpclattice_service_network_vpc_association":           awsVpclatticeServiceNetworkVpcAssociation,
		"aws_vpclattice_target_group":                              awsVpclatticeTargetGroup,
		"aws_vpclattice_target_group_attachment":                   awsVpclatticeTargetGroupAssociation,
		"aws_autoscaling_group_tag":                                awsAutoscalingGroupTag,
		"aws_autoscaling_schedule":                                 awsAutoscalingSchedule,
		"aws_autoscaling_traffic_source_attachment":                awsAutoscalingTrafficSourceAttachment,
		"aws_autoscalingplans_scaling_plan":                        awsAutoscalingplansScalingPlan,
		"aws_elb_attachment":                                       awsElbAttachment,
		"aws_connect_bot_association":                              awsConnectBotAssociation,
		"aws_connect_contact_flow":                                 awsConnectContactFlow,
		"aws_connect_contact_flow_module":                          awsConnectContactFlowModule,
		"aws_connect_hours_of_operation":                           awsConnectHoursOfOperation,
		"aws_connect_instance":                                     awsConnectInstance,
		"aws_connect_instance_storage_config":                      awsConnectInstanceStorageConfig,
		"aws_connect_phone_number":                                 awsConnectPhoneNumber,
		"aws_connect_queue":                                        awsConnectQueue,
		"aws_connect_quick_connect":                                awsConnectQuickConnect,
		"aws_connect_routing_profile":                              awsConnectRoutingProfile,
		"aws_connect_security_profile":                             awsConnectSecurityProfile,
		"aws_connect_user":                                         awsConnectUser,
		"aws_connect_user_hierarchy_structure":                     awsConnectUserHierarchyStructure,
		"aws_connect_user_hierarchy_group":                         awsConnectUserHierarchyGroup,
		"aws_connect_vocabulary":                                   awsConnectVocabulary,
		"aws_verifiedaccess_endpoint":                              awsVerifiedaccessEndpoint,
		"aws_verifiedaccess_group":                                 awsVerifiedaccessGroup,
		"aws_verifiedaccess_instance":                              awsVerifiedaccessInstance,
		"aws_verifiedaccess_instance_logging_configuration":        awsVerifiedaccessInstanceLoggingConfiguration,
		"aws_verifiedaccess_trust_provider":                        awsVerifiedaccessTrustProvider,
		"aws_servicecatalog_budget_resource_association":           awsServicecatalogBudgetResourceAssociation,
		"aws_servicecatalog_constraint":                            awsServicecatalogConstraint,
		"aws_servicecatalog_organizations_access":                  awsServicecatalogOrganizationAccess,
		"aws_servicecatalog_portfolio_share":                       awsServicecatalogPortfolioShare,
		"aws_servicecatalog_principal_portfolio_association":       awsServicecatalogPrincipalPortfolioAssociation,
		"aws_servicecatalog_product":                               awsServicecatalogProduct,
		"aws_servicecatalog_product_portfolio_association":         awsServicecatalogProductPortfolioAssociation,
		"aws_servicecatalog_provisioned_product":                   awsServicecatalogProvisionedProduct,
		"aws_servicecatalog_service_action":                        awsServicecatalogServiceAction,
		"aws_servicecatalog_tag_option":                            awsServicecatalogTagOption,
		"aws_servicecatalog_tag_option_resource_association":       awsServicecatalogTagOptionResourceAssociation,
		"aws_servicequotas_template":                               awsServiceQuotasTemplate,
		"aws_servicequotas_template_association":                   awsServiceQuotasTemplateAssociation,
		"aws_codecatalyst_source_repository":                       placeholder,
		"aws_codeguruprofiler_profiling_group":                     awsCodeguruprofilerProfilingGroup,
		"aws_codegurureviewer_repository_association":              awsCodegurureviewerRepositoryAssociation,
		"aws_codepipeline_custom_action_type":                      awsCodepipelineCustomActionType,
		"aws_codepipeline_webhook":                                 awsCodepipelineWebhook,
		"aws_codestarconnections_connection":                       awsCodestarconnectionsConnection,
		"aws_codestarconnections_host":                             awsCodestarconnectionsHost,
		"aws_codestarnotifications_notification_rule":              awsCodestarconnectionsNotificationsRule,
		"aws_codecatalyst_dev_environment":                         placeholder,
		"aws_codecatalyst_project":                                 placeholder,
		"aws_auditmanager_account_registration":                    awsAuditmanagerAccountRegistration,
		"aws_auditmanager_assessment":                              awsAuditmanagerAssessment,
		"aws_auditmanager_assessment_delegation":                   awsAuditmanagerAssessmentDelegation,
		"aws_auditmanager_assessment_report":                       awsAuditmanagerAssessmentReport,
		"aws_auditmanager_control":                                 awsAuditmanagerControl,
		"aws_auditmanager_framework":                               awsAuditmanagerFramework,
		"aws_auditmanager_framework_share":                         awsAuditmanagerFrameworkShare,
		"aws_auditmanager_organization_admin_account_registration": awsAuditmanagerOrganizationAdminAccountRegistration,
		"aws_bedrock_custom_model":                                 awsBedrockCustomModel,
		"aws_bedrock_model_invocation_logging_configuration":       awsBedrockModelInvocationLoggingConfiguration,
		"aws_bedrock_provisioned_model_throughput":                 awsBedrockProvisionedModelThroughput,
		"aws_bedrockagent_agent":                                   awsBedrockagentAgent,
		"aws_bedrockagent_agent_action_group":                      awsBedrockagentActionGroup,
		"aws_bedrockagent_agent_alias":                             awsBedrockagentAgentAlias,
		"aws_datasync_agent":                                       awsDatasyncAgent,
		"aws_datasync_location_azure_blob":                         awsDatasyncLocationAzureBlob,
		"aws_datasync_location_efs":                                awsDatasyncLocationEfs,
		"aws_datasync_location_fsx_lustre_file_system":             awsDatasyncLocationFsxLustreFileSystem,
		"aws_datasync_location_fsx_ontap_file_system":              awsDatasyncLocationFsxOntapFileSystem,
		"aws_datasync_location_fsx_openzfs_file_system":            awsDatasyncLocationFsxOpenzfsFileSystem,
		"aws_datasync_location_fsx_windows_file_system":            awsDatasyncLocationFsxWindowsFileSystem,
		"aws_datasync_location_hdfs":                               awsDatasyncLocationHdfs,
		"aws_datasync_location_nfs":                                awsDatasyncLocationNfs,
		"aws_datasync_location_object_storage":                     awsDatasyncLocationObjectStorage,
		"aws_datasync_location_s3":                                 awsDatasyncLocationS3,
		"aws_datasync_location_smb":                                awsDatasyncLocationSmb,
		"aws_datasync_task":                                        awsDatasyncTask,
		"aws_fsx_lustre_file_system":                               awsFsxLustreFileSystem,
		"aws_fsx_windows_file_system":                              awsFsxWindowsFileSystem,
		"aws_fsx_backup":                                           awsFsxBackup,
		"aws_fsx_data_repository_association":                      awsFsxDataRepositoryAssociation,
		"aws_fsx_file_cache":                                       awsFsxFileCache,
		"aws_fsx_ontap_file_system":                                awsFsxOntapFileSystem,
		"aws_fsx_ontap_storage_virtual_machine":                    awsFsxOntapStorageVirtualMachine,
		"aws_fsx_ontap_volume":                                     awsFsxOntapVolume,
		"aws_datazone_domain":                                      awsDatazoneDomain,
		"aws_datazone_environment_blueprint_configuration":         awsDatazoneEnvironmentBlueprintConfiguration,
		"aws_storagegateway_cache":                                 awsStoragegatewayCache,
		"aws_storagegateway_cached_iscsi_volume":                   awsStoragegatewayCachedIscsiVolume,
		"aws_storagegateway_file_system_association":               awsStoragegatewayFileSystemAssociation,
		"aws_storagegateway_gateway":                               awsStoragegatewayGateway,
		"aws_storagegateway_nfs_file_share":                        awsStoragegatewayNfsFileShare,
		"aws_storagegateway_smb_file_share":                        awsStoragegatewaySmbFileShare,
		"aws_storagegateway_stored_iscsi_volume":                   awsStoragegatewayStoredIscsiVolume,
		"aws_storagegateway_tape_pool":                             awsStoragegatewayTapePool,
		"aws_storagegateway_upload_buffer":                         awsStoragegatewayUploadBuffer,
		"aws_storagegateway_working_storage":                       awsStoragegatewayWorkingStorage,
	}

	return TFLookup[name]
}

// Contains looks if slice contains string.
func Contains(s []string, e string) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}

	return false
}

// GetPermissionMap Anonymous parsing.
func GetPermissionMap(raw []byte, attributes []string, resource string) ([]string, error) {
	var mappings []interface{}
	err := json.Unmarshal(raw, &mappings)

	if err != nil {
		return nil, fmt.Errorf("failed to unmarshal json %w for %s", err, resource)
	}

	if mappings == nil {
		return nil, errors.New("mappings are empty")
	}

	temp, ok := mappings[0].(map[string]interface{})

	if !ok {
		return nil, fmt.Errorf("assertion to map[string]interface{} failed")
	}

	myAttributes, ok := temp["attributes"].(map[string]interface{})

	if !ok {
		fmt.Errorf("assertion failed")
	}

	var found []string

	for _, attribute := range attributes {
		if myAttributes[attribute] != nil {
			for _, entry := range myAttributes[attribute].([]interface{}) {

				found = append(
					found,
					entry.(string),
				)
			}
		}
	}

	for _, action := range []string{"apply", "plan", "modify", "destroy"} {
		if temp[action] != nil {
			for _, entry := range temp[action].([]interface{}) {
				found = append(found, entry.(string))
			}
		}
	}

	return found, nil
}
