// Copyright (C) 2018-2020 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <gtest/gtest.h>
#include <parsers.h>
#include <cpp/ie_cnn_net_reader.h>
#include <ie_cnn_net_reader_impl.h>
#include <mock_icnn_network.hpp>
#include <gmock/gmock-more-actions.h>
#include "cnn_network_impl.hpp"
#include "mock_iformat_parser.hpp"
#include <single_layer_common.hpp>
#include <thread>

using namespace testing;
using namespace InferenceEngine;
using namespace InferenceEngine::details;
using namespace std;

class CNNNetReaderImplTest : public ::testing::Test {
public:
    StatusCode sts = OK;
    ResponseDesc resp;
};

struct MockFormatParserCreator : public FormatParserCreator {
    MockFormatParserCreator() {
        _parser = make_shared<MockIFormatParser>();
    }

    std::shared_ptr<IFormatParser> create(size_t version) override {
        return _parser;
    }

    MockIFormatParser* getParser() {
        return _parser.get();
    }

private:
    std::shared_ptr<MockIFormatParser> _parser;
};

TEST_F(CNNNetReaderImplTest, validateIsCalled) {
    std::string model =
            "<net name=\"PVANET\" version=\"2\" batch=\"1\">"
            "    <layers>"
            "        <layer name=\"data\" type=\"Input\" precision=\"FP32\" id=\"0\">"
            "            <output>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>544</dim>"
            "                    <dim>992</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_conv\" type=\"Convolution\" precision=\"FP32\" id=\"2\">"
            "            <convolution_data stride-x=\"2\" stride-y=\"2\" pad-x=\"3\" pad-y=\"3\" kernel-x=\"7\" kernel-y=\"7\" output=\"16\" group=\"1\"/>"
            "            <input>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>544</dim>"
            "                    <dim>992</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "            <weights offset=\"0\" size=\"9408\"/>"
            "            <biases offset=\"9408\" size=\"64\"/>"
            "        </layer>"
            "        <layer name=\"conv1_1_neg\" type=\"Power\" precision=\"FP32\" id=\"3\">"
            "            <power_data power=\"1\" scale=\"-1\" shift=\"0\"/>"
            "            <input>"
            "                <port id=\"4\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"5\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_concat\" type=\"Concat\" precision=\"FP32\" id=\"4\">"
            "            <concat_data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"6\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "                <port id=\"7\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"8\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_scale\" type=\"ScaleShift\" precision=\"FP32\" id=\"5\">"
            "            <input>"
            "                <port id=\"9\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"10\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "            <weights offset=\"9472\" size=\"128\"/>"
            "            <biases offset=\"9600\" size=\"128\"/>"
            "        </layer>"
            "        <layer name=\"conv1_1_relu\" type=\"ReLU\" precision=\"FP32\" id=\"6\">"
            "            <data negative_slope=\"0\" engine=\"caffe.ReLUParameter.DEFAULT\"/>"
            "            <input>"
            "                <port id=\"11\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"12\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"pool1\" type=\"Pooling\" precision=\"FP32\" id=\"7\">"
            "            <pooling_data kernel-x=\"3\" kernel-y=\"3\" pad-x=\"0\" pad-y=\"0\" stride-x=\"2\" stride-y=\"2\" rounding-type=\"ceil\" pool-method=\"max\"/>"
            "            <input>"
            "                <port id=\"13\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"14\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>136</dim>"
            "                    <dim>248</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "    </layers>"
            "    <edges>"
            "        <edge from-layer=\"0\" from-port=\"0\" to-layer=\"2\" to-port=\"2\"/>"
            "        <edge from-layer=\"2\" from-port=\"3\" to-layer=\"3\" to-port=\"4\"/>"
            "        <edge from-layer=\"2\" from-port=\"3\" to-layer=\"4\" to-port=\"6\"/>"
            "        <edge from-layer=\"3\" from-port=\"5\" to-layer=\"4\" to-port=\"7\"/>"
            "        <edge from-layer=\"4\" from-port=\"8\" to-layer=\"5\" to-port=\"9\"/>"
            "        <edge from-layer=\"5\" from-port=\"10\" to-layer=\"6\" to-port=\"11\"/>"
            "        <edge from-layer=\"6\" from-port=\"12\" to-layer=\"7\" to-port=\"13\"/>"
            "    </edges>"
            "</net>";
    auto parserCreator = make_shared<MockFormatParserCreator>();
    CNNNetReaderImpl reader(parserCreator);
    auto network = make_shared<MockCNNNetworkImpl>();
    auto name = std::string{"AlexNet"};

    EXPECT_CALL(*parserCreator->getParser(), Parse(_)).Times(1).WillOnce(Return(network));
    EXPECT_CALL(*network.get(), validate(_)).Times(1);
    EXPECT_CALL(*network.get(), getName()).Times(1).WillOnce(ReturnRef(name));

    ASSERT_NO_THROW(sts = reader.ReadNetwork(model.data(), model.length(), &resp));
    ASSERT_EQ(OK, sts);
}

TEST_F(CNNNetReaderImplTest, cycleIsDetectedInReader) {
    std::string model =
            "<net batch=\"1\" name=\"model\" version=\"2\">"
            "    <layers>"
            "        <layer id=\"0\" name=\"data\" precision=\"FP32\" type=\"Input\">"
            "            <output>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>227</dim>"
            "                    <dim>227</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"1\" name=\"conv1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"64\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,2,2\" stride-x=\"2\" stride-y=\"2\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>227</dim>"
            "                    <dim>227</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>113</dim>"
            "                    <dim>113</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"0\" size=\"6912\"/>"
            "                <biases offset=\"6912\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"2\" name=\"relu_conv1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>113</dim>"
            "                    <dim>113</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>113</dim>"
            "                    <dim>113</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"3\" name=\"pool1\" precision=\"FP32\" type=\"Pooling\">"
            "            <data exclude-pad=\"false\" kernel-x=\"3\" kernel-y=\"3\" pad-x=\"0\" pad-y=\"0\" pool-method=\"max\" rounding_type=\"ceil\" stride=\"1,1,2,2\" stride-x=\"2\" stride-y=\"2\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>113</dim>"
            "                    <dim>113</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"4\" name=\"fire2/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"16\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"7168\" size=\"4096\"/>"
            "                <biases offset=\"11264\" size=\"64\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"5\" name=\"fire2/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"6\" name=\"fire2/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"64\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"11328\" size=\"4096\"/>"
            "                <biases offset=\"15424\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"7\" name=\"fire2/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"8\" name=\"fire2/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"64\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"15680\" size=\"36864\"/>"
            "                <biases offset=\"52544\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"9\" name=\"fire2/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"10\" name=\"fire2/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"11\" name=\"fire3/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"16\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "                <port id=\"4\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"52800\" size=\"8192\"/>"
            "                <biases offset=\"60992\" size=\"64\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"12\" name=\"fire3/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"13\" name=\"fire3/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"64\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"61056\" size=\"4096\"/>"
            "                <biases offset=\"65152\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"14\" name=\"fire3/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"15\" name=\"fire3/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"64\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"65408\" size=\"36864\"/>"
            "                <biases offset=\"102272\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"16\" name=\"fire3/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"17\" name=\"fire3/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"18\" name=\"pool3\" precision=\"FP32\" type=\"Pooling\">"
            "            <data exclude-pad=\"false\" kernel-x=\"3\" kernel-y=\"3\" pad-x=\"0\" pad-y=\"0\" pool-method=\"max\" rounding_type=\"ceil\" stride=\"1,1,2,2\" stride-x=\"2\" stride-y=\"2\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"19\" name=\"fire4/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"32\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"102528\" size=\"16384\"/>"
            "                <biases offset=\"118912\" size=\"128\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"20\" name=\"fire4/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"21\" name=\"fire4/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"128\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"119040\" size=\"16384\"/>"
            "                <biases offset=\"135424\" size=\"512\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"22\" name=\"fire4/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"23\" name=\"fire4/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"128\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"135936\" size=\"147456\"/>"
            "                <biases offset=\"283392\" size=\"512\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"24\" name=\"fire4/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"25\" name=\"fire4/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"26\" name=\"fire5/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"32\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"283904\" size=\"32768\"/>"
            "                <biases offset=\"316672\" size=\"128\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"27\" name=\"fire5/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"28\" name=\"fire5/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"128\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"316800\" size=\"16384\"/>"
            "                <biases offset=\"333184\" size=\"512\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"29\" name=\"fire5/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"30\" name=\"fire5/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"128\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"333696\" size=\"147456\"/>"
            "                <biases offset=\"481152\" size=\"512\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"31\" name=\"fire5/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"32\" name=\"fire5/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>128</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"33\" name=\"pool5\" precision=\"FP32\" type=\"Pooling\">"
            "            <data exclude-pad=\"false\" kernel-x=\"3\" kernel-y=\"3\" pad-x=\"0\" pad-y=\"0\" pool-method=\"max\" rounding_type=\"ceil\" stride=\"1,1,2,2\" stride-x=\"2\" stride-y=\"2\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"34\" name=\"fire6/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"48\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"481664\" size=\"49152\"/>"
            "                <biases offset=\"530816\" size=\"192\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"35\" name=\"fire6/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"36\" name=\"fire6/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"192\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"531008\" size=\"36864\"/>"
            "                <biases offset=\"567872\" size=\"768\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"37\" name=\"fire6/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"38\" name=\"fire6/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"192\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"568640\" size=\"331776\"/>"
            "                <biases offset=\"900416\" size=\"768\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"39\" name=\"fire6/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"40\" name=\"fire6/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>384</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"41\" name=\"fire7/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"48\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>384</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"901184\" size=\"73728\"/>"
            "                <biases offset=\"974912\" size=\"192\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"42\" name=\"fire7/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"43\" name=\"fire7/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"192\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"975104\" size=\"36864\"/>"
            "                <biases offset=\"1011968\" size=\"768\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"44\" name=\"fire7/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"45\" name=\"fire7/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"192\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>48</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"1012736\" size=\"331776\"/>"
            "                <biases offset=\"1344512\" size=\"768\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"46\" name=\"fire7/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"47\" name=\"fire7/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>192</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>384</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"48\" name=\"fire8/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"64\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>384</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"1345280\" size=\"98304\"/>"
            "                <biases offset=\"1443584\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"49\" name=\"fire8/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"50\" name=\"fire8/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"256\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"1443840\" size=\"65536\"/>"
            "                <biases offset=\"1509376\" size=\"1024\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"51\" name=\"fire8/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"52\" name=\"fire8/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"256\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"1510400\" size=\"589824\"/>"
            "                <biases offset=\"2100224\" size=\"1024\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"53\" name=\"fire8/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"54\" name=\"fire8/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>512</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"55\" name=\"fire9/squeeze1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"64\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>512</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"2101248\" size=\"131072\"/>"
            "                <biases offset=\"2232320\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"56\" name=\"fire9/relu_squeeze1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"57\" name=\"fire9/expand1x1\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"256\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"2232576\" size=\"65536\"/>"
            "                <biases offset=\"2298112\" size=\"1024\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"58\" name=\"fire9/relu_expand1x1\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"59\" name=\"fire9/expand3x3\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"3\" kernel-y=\"3\" output=\"256\" pad-x=\"1\" pad-y=\"1\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"2299136\" size=\"589824\"/>"
            "                <biases offset=\"2888960\" size=\"1024\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"60\" name=\"fire9/relu_expand3x3\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"61\" name=\"fire9/concat\" precision=\"FP32\" type=\"Concat\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>256</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>512</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"62\" name=\"conv10\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilation-x=\"1\" dilation-y=\"1\" group=\"1\" kernel-x=\"1\" kernel-y=\"1\" output=\"1000\" pad-x=\"0\" pad-y=\"0\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>512</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"2889984\" size=\"2048000\"/>"
            "                <biases offset=\"4937984\" size=\"4000\"/>"
            "            </blobs>"
            "        </layer>"
            "        <layer id=\"63\" name=\"relu_conv10\" precision=\"FP32\" type=\"ReLU\">"
            "            <data engine=\"caffe.ReLUParameter.DEFAULT\" negative_slope=\"0.0\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"64\" name=\"pool10\" precision=\"FP32\" type=\"Pooling\">"
            "            <data exclude-pad=\"false\" kernel-x=\"14\" kernel-y=\"14\" pad-x=\"0\" pad-y=\"0\" pool-method=\"avg\" rounding_type=\"ceil\" stride=\"1,1,1,1\" stride-x=\"1\" stride-y=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>14</dim>"
            "                    <dim>14</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>1</dim>"
            "                    <dim>1</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"65\" name=\"prob\" precision=\"FP32\" type=\"SoftMax\">"
            "            <data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>1</dim>"
            "                    <dim>1</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>1000</dim>"
            "                    <dim>1</dim>"
            "                    <dim>1</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "    </layers>"
            "    <edges>"
            "        <edge from-layer=\"0\" from-port=\"0\" to-layer=\"1\" to-port=\"0\"/>"
            "        <edge from-layer=\"1\" from-port=\"3\" to-layer=\"2\" to-port=\"0\"/>"
            "        <edge from-layer=\"2\" from-port=\"1\" to-layer=\"3\" to-port=\"0\"/>"
            "        <edge from-layer=\"3\" from-port=\"1\" to-layer=\"4\" to-port=\"0\"/>"
            "        <edge from-layer=\"4\" from-port=\"3\" to-layer=\"5\" to-port=\"0\"/>"
            "        <edge from-layer=\"5\" from-port=\"1\" to-layer=\"6\" to-port=\"0\"/>"
            "        <edge from-layer=\"6\" from-port=\"3\" to-layer=\"7\" to-port=\"0\"/>"
            "        <edge from-layer=\"5\" from-port=\"1\" to-layer=\"8\" to-port=\"0\"/>"
            "        <edge from-layer=\"8\" from-port=\"3\" to-layer=\"9\" to-port=\"0\"/>"
            "        <edge from-layer=\"7\" from-port=\"1\" to-layer=\"10\" to-port=\"0\"/>"
            "        <edge from-layer=\"9\" from-port=\"1\" to-layer=\"10\" to-port=\"1\"/>"
            "        <edge from-layer=\"10\" from-port=\"2\" to-layer=\"11\" to-port=\"0\"/>"
            "        <edge from-layer=\"11\" from-port=\"3\" to-layer=\"12\" to-port=\"0\"/>"
            "        <edge from-layer=\"11\" from-port=\"4\" to-layer=\"8\" to-port=\"1\"/>"
            "        <edge from-layer=\"12\" from-port=\"1\" to-layer=\"13\" to-port=\"0\"/>"
            "        <edge from-layer=\"13\" from-port=\"3\" to-layer=\"14\" to-port=\"0\"/>"
            "        <edge from-layer=\"12\" from-port=\"1\" to-layer=\"15\" to-port=\"0\"/>"
            "        <edge from-layer=\"15\" from-port=\"3\" to-layer=\"16\" to-port=\"0\"/>"
            "        <edge from-layer=\"14\" from-port=\"1\" to-layer=\"17\" to-port=\"0\"/>"
            "        <edge from-layer=\"16\" from-port=\"1\" to-layer=\"17\" to-port=\"1\"/>"
            "        <edge from-layer=\"17\" from-port=\"2\" to-layer=\"18\" to-port=\"0\"/>"
            "        <edge from-layer=\"18\" from-port=\"1\" to-layer=\"19\" to-port=\"0\"/>"
            "        <edge from-layer=\"19\" from-port=\"3\" to-layer=\"20\" to-port=\"0\"/>"
            "        <edge from-layer=\"20\" from-port=\"1\" to-layer=\"21\" to-port=\"0\"/>"
            "        <edge from-layer=\"21\" from-port=\"3\" to-layer=\"22\" to-port=\"0\"/>"
            "        <edge from-layer=\"20\" from-port=\"1\" to-layer=\"23\" to-port=\"0\"/>"
            "        <edge from-layer=\"23\" from-port=\"3\" to-layer=\"24\" to-port=\"0\"/>"
            "        <edge from-layer=\"22\" from-port=\"1\" to-layer=\"25\" to-port=\"0\"/>"
            "        <edge from-layer=\"24\" from-port=\"1\" to-layer=\"25\" to-port=\"1\"/>"
            "        <edge from-layer=\"25\" from-port=\"2\" to-layer=\"26\" to-port=\"0\"/>"
            "        <edge from-layer=\"26\" from-port=\"3\" to-layer=\"27\" to-port=\"0\"/>"
            "        <edge from-layer=\"27\" from-port=\"1\" to-layer=\"28\" to-port=\"0\"/>"
            "        <edge from-layer=\"28\" from-port=\"3\" to-layer=\"29\" to-port=\"0\"/>"
            "        <edge from-layer=\"27\" from-port=\"1\" to-layer=\"30\" to-port=\"0\"/>"
            "        <edge from-layer=\"30\" from-port=\"3\" to-layer=\"31\" to-port=\"0\"/>"
            "        <edge from-layer=\"29\" from-port=\"1\" to-layer=\"32\" to-port=\"0\"/>"
            "        <edge from-layer=\"31\" from-port=\"1\" to-layer=\"32\" to-port=\"1\"/>"
            "        <edge from-layer=\"32\" from-port=\"2\" to-layer=\"33\" to-port=\"0\"/>"
            "        <edge from-layer=\"33\" from-port=\"1\" to-layer=\"34\" to-port=\"0\"/>"
            "        <edge from-layer=\"34\" from-port=\"3\" to-layer=\"35\" to-port=\"0\"/>"
            "        <edge from-layer=\"35\" from-port=\"1\" to-layer=\"36\" to-port=\"0\"/>"
            "        <edge from-layer=\"36\" from-port=\"3\" to-layer=\"37\" to-port=\"0\"/>"
            "        <edge from-layer=\"35\" from-port=\"1\" to-layer=\"38\" to-port=\"0\"/>"
            "        <edge from-layer=\"38\" from-port=\"3\" to-layer=\"39\" to-port=\"0\"/>"
            "        <edge from-layer=\"37\" from-port=\"1\" to-layer=\"40\" to-port=\"0\"/>"
            "        <edge from-layer=\"39\" from-port=\"1\" to-layer=\"40\" to-port=\"1\"/>"
            "        <edge from-layer=\"40\" from-port=\"2\" to-layer=\"41\" to-port=\"0\"/>"
            "        <edge from-layer=\"41\" from-port=\"3\" to-layer=\"42\" to-port=\"0\"/>"
            "        <edge from-layer=\"42\" from-port=\"1\" to-layer=\"43\" to-port=\"0\"/>"
            "        <edge from-layer=\"43\" from-port=\"3\" to-layer=\"44\" to-port=\"0\"/>"
            "        <edge from-layer=\"42\" from-port=\"1\" to-layer=\"45\" to-port=\"0\"/>"
            "        <edge from-layer=\"45\" from-port=\"3\" to-layer=\"46\" to-port=\"0\"/>"
            "        <edge from-layer=\"44\" from-port=\"1\" to-layer=\"47\" to-port=\"0\"/>"
            "        <edge from-layer=\"46\" from-port=\"1\" to-layer=\"47\" to-port=\"1\"/>"
            "        <edge from-layer=\"47\" from-port=\"2\" to-layer=\"48\" to-port=\"0\"/>"
            "        <edge from-layer=\"48\" from-port=\"3\" to-layer=\"49\" to-port=\"0\"/>"
            "        <edge from-layer=\"49\" from-port=\"1\" to-layer=\"50\" to-port=\"0\"/>"
            "        <edge from-layer=\"50\" from-port=\"3\" to-layer=\"51\" to-port=\"0\"/>"
            "        <edge from-layer=\"49\" from-port=\"1\" to-layer=\"52\" to-port=\"0\"/>"
            "        <edge from-layer=\"52\" from-port=\"3\" to-layer=\"53\" to-port=\"0\"/>"
            "        <edge from-layer=\"51\" from-port=\"1\" to-layer=\"54\" to-port=\"0\"/>"
            "        <edge from-layer=\"53\" from-port=\"1\" to-layer=\"54\" to-port=\"1\"/>"
            "        <edge from-layer=\"54\" from-port=\"2\" to-layer=\"55\" to-port=\"0\"/>"
            "        <edge from-layer=\"55\" from-port=\"3\" to-layer=\"56\" to-port=\"0\"/>"
            "        <edge from-layer=\"56\" from-port=\"1\" to-layer=\"57\" to-port=\"0\"/>"
            "        <edge from-layer=\"57\" from-port=\"3\" to-layer=\"58\" to-port=\"0\"/>"
            "        <edge from-layer=\"56\" from-port=\"1\" to-layer=\"59\" to-port=\"0\"/>"
            "        <edge from-layer=\"59\" from-port=\"3\" to-layer=\"60\" to-port=\"0\"/>"
            "        <edge from-layer=\"58\" from-port=\"1\" to-layer=\"61\" to-port=\"0\"/>"
            "        <edge from-layer=\"60\" from-port=\"1\" to-layer=\"61\" to-port=\"1\"/>"
            "        <edge from-layer=\"61\" from-port=\"2\" to-layer=\"62\" to-port=\"0\"/>"
            "        <edge from-layer=\"62\" from-port=\"3\" to-layer=\"63\" to-port=\"0\"/>"
            "        <edge from-layer=\"63\" from-port=\"1\" to-layer=\"64\" to-port=\"0\"/>"
            "        <edge from-layer=\"64\" from-port=\"1\" to-layer=\"65\" to-port=\"0\"/>"
            "    </edges>"
            "</net>";
    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());

    ASSERT_EQ(GENERAL_ERROR, reader.ReadNetwork(model.data(), model.length(), &resp));
}

TEST_F(CNNNetReaderImplTest, canRead3DConvolution) {
    std::string model =
            "<net batch=\"1\" name=\"Convolution_only\" version=\"3\">"
            "    <layers>"
            "        <layer id=\"0\" name=\"1\" precision=\"FP32\" type=\"Input\">"
            "            <output>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>16</dim>"
            "                    <dim>112</dim>"
            "                    <dim>112</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"1\" name=\"3D_conv\" precision=\"FP32\" type=\"Convolution\">"
            "            <data dilations=\"1,3,5\" group=\"1\" kernel=\"1,3,5\" output=\"64\" pads_begin=\"1,3,5\" pads_end=\"1,3,5\" strides=\"1,3,5\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>16</dim>"
            "                    <dim>112</dim>"
            "                    <dim>112</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>16</dim>"
            "                    <dim>56</dim>"
            "                    <dim>56</dim>"
            "                </port>"
            "            </output>"
            "            <blobs>"
            "                <weights offset=\"0\" size=\"263424\"/>"
            "                <biases offset=\"263424\" size=\"256\"/>"
            "            </blobs>"
            "        </layer>"
            "    </layers>"
            "    <edges>"
            "        <edge from-layer=\"0\" from-port=\"0\" to-layer=\"1\" to-port=\"0\"/>"
            "    </edges>"
            "</net>";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    ASSERT_EQ(OK, reader.ReadNetwork(model.data(), model.length(), &resp));
    ResponseDesc resp;
    auto network = reader.getNetwork(&resp);

    CNNLayerPtr layer;
    ASSERT_EQ(OK, network->getLayerByName("3D_conv", layer, nullptr));
    auto* conv = dynamic_cast<ConvolutionLayer*>(layer.get());
    ASSERT_NE(nullptr, conv);
    ASSERT_EQ(conv->_kernel[X_AXIS], 5);
    ASSERT_EQ(conv->_kernel[Y_AXIS], 3);
    ASSERT_EQ(conv->_kernel[Z_AXIS], 1);
    ASSERT_EQ(conv->_dilation[X_AXIS], 5);
    ASSERT_EQ(conv->_dilation[Y_AXIS], 3);
    ASSERT_EQ(conv->_dilation[Z_AXIS], 1);
    ASSERT_EQ(conv->_stride[X_AXIS], 5);
    ASSERT_EQ(conv->_stride[Y_AXIS], 3);
    ASSERT_EQ(conv->_stride[Z_AXIS], 1);
    ASSERT_EQ(conv->_padding[X_AXIS], 5);
    ASSERT_EQ(conv->_padding[Y_AXIS], 3);
    ASSERT_EQ(conv->_padding[Z_AXIS], 1);
    ASSERT_EQ(conv->_pads_end[X_AXIS], 5);
    ASSERT_EQ(conv->_pads_end[Y_AXIS], 3);
    ASSERT_EQ(conv->_pads_end[Z_AXIS], 1);
}

TEST_F(CNNNetReaderImplTest, canRead3DPooling) {
    std::string model =
            "<net batch=\"1\" name=\"Pooling_only\" version=\"3\">"
            "    <layers>"
            "        <layer id=\"0\" name=\"1\" precision=\"FP32\" type=\"Input\">"
            "            <output>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>16</dim>"
            "                    <dim>112</dim>"
            "                    <dim>112</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer id=\"1\" name=\"3D_pooling\" precision=\"FP32\" type=\"Pooling\">"
            "            <data exclude-pad=\"true\" kernel=\"1,3,5\" pads_begin=\"1,3,5\" pads_end=\"1,3,5\" pool-method=\"max\" rounding_type=\"ceil\" strides=\"1,3,5\"/>"
            "            <input>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>16</dim>"
            "                    <dim>112</dim>"
            "                    <dim>112</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"1\">"
            "                    <dim>1</dim>"
            "                    <dim>64</dim>"
            "                    <dim>8</dim>"
            "                    <dim>28</dim>"
            "                    <dim>28</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "    </layers>"
            "    <edges>"
            "        <edge from-layer=\"0\" from-port=\"0\" to-layer=\"1\" to-port=\"0\"/>"
            "    </edges>"
            "</net>";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    ASSERT_EQ(OK, reader.ReadNetwork(model.data(), model.length(), &resp));
    ResponseDesc resp;
    auto network = reader.getNetwork(&resp);

    CNNLayerPtr layer;

    ASSERT_EQ(OK, network->getLayerByName("3D_pooling", layer, nullptr));
    auto* pool = dynamic_cast<PoolingLayer*>(layer.get());
    ASSERT_NE(nullptr, pool);
    ASSERT_EQ(pool->_kernel[X_AXIS], 5);
    ASSERT_EQ(pool->_kernel[Y_AXIS], 3);
    ASSERT_EQ(pool->_kernel[Z_AXIS], 1);
    ASSERT_EQ(pool->_stride[X_AXIS], 5);
    ASSERT_EQ(pool->_stride[Y_AXIS], 3);
    ASSERT_EQ(pool->_stride[Z_AXIS], 1);
    ASSERT_EQ(pool->_padding[X_AXIS], 5);
    ASSERT_EQ(pool->_padding[Y_AXIS], 3);
    ASSERT_EQ(pool->_padding[Z_AXIS], 1);
    ASSERT_EQ(pool->_pads_end[X_AXIS], 5);
    ASSERT_EQ(pool->_pads_end[Y_AXIS], 3);
    ASSERT_EQ(pool->_pads_end[Z_AXIS], 1);
}

TEST_F(CNNNetReaderImplTest, canParseWithoutInput_1to2) {
    std::string model = R"V0G0N(
<net batch="1" name="SimpleNet" version="2">
    <layers>
        <layer id="1" name="Boo" precision="FP32" type="Split">
            <data operation="sum"/>
            <input>
                <port id="0">
                    <dim>2</dim>
                    <dim>16</dim>
                </port>
            </input>
            <output>
                <port id="1">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
                <port id="2">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
            </output>
        </layer>
    </layers>
</net>
    )V0G0N";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    sts = reader.ReadNetwork(model.data(), model.length(), &resp);
    ASSERT_EQ(GENERAL_ERROR, sts) << resp.msg;
}

TEST_F(CNNNetReaderImplTest, canParseWithoutInput_2to1) {
    std::string model = R"V0G0N(
<net batch="1" name="SimpleNet" version="2">
    <layers>
        <layer id="1" name="Foo" precision="FP32" type="Eltwise">
            <data operation="sum"/>
            <input>
                <port id="0">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
                <port id="1">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
            </input>
            <output>
                <port id="2">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
            </output>
        </layer>
    </layers>
</net>
    )V0G0N";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    sts = reader.ReadNetwork(model.data(), model.length(), &resp);
    ASSERT_EQ(GENERAL_ERROR, sts) << resp.msg;
}

TEST_F(CNNNetReaderImplTest, canParseSimpleTI) {
    std::string model = R"V0G0N(
<net batch="1" name="Simple_TI" version="4">
    <layers>
        <layer id="0" name="input" precision="FP32" type="Input">
            <output>
                <port id="0">
                    <dim>1</dim>
                    <dim>5</dim>
                    <dim>16</dim>
                </port>
            </output>
        </layer>
        <layer id="1" name="Bias" precision="FP32" type="Const">
            <output>
                <port id="0">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
            </output>
            <blobs>
                <custom offset="0" size="64"/>
            </blobs>
        </layer>
        <layer id="2" name="SomeTI" precision="FP32" type="TensorIterator">
            <input>
                <port id="0">
                    <dim>1</dim>
                    <dim>5</dim>
                    <dim>16</dim>
                </port>
                <port id="1">
                    <dim>1</dim>
                    <dim>16</dim>
                </port>
            </input>
            <output>
                <port id="3">
                    <dim>1</dim>
                    <dim>5</dim>
                    <dim>16</dim>
                </port>
            </output>
            <port_map>
                <input  external_port_id="0" internal_layer_id="0" internal_port_id="0" axis="1" />
                <input  external_port_id="1" internal_layer_id="1" internal_port_id="1"/>
                <output external_port_id="3" internal_layer_id="2" internal_port_id="1" axis="1" />
            </port_map>
            <back_edges>
                <edge from-layer="1" from-port="2" to-layer="1" to-port="1"/>
            </back_edges>
            <body>
                <layers>
                    <layer id="0" name="TI_reshape_in" precision="FP32" type="Reshape">
                        <data axis="0" dim="1,512" num_axes="-1"/>
                        <input>
                            <port id="0">
                                <dim>1</dim>
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </input>
                        <output>
                            <port id="1">
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </output>
                    </layer>
                    <layer id="1" name="TI_sum" precision="FP32" type="Eltwise">
                        <data operation="sum"/>
                        <input>
                            <port id="0">
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                            <port id="1">
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </input>
                        <output>
                            <port id="2">
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </output>
                    </layer>
                    <layer id="2" name="TI_reshape_out" precision="FP32" type="Reshape">
                        <data axis="0" dim="1,1,256" num_axes="-1"/>
                        <input>
                            <port id="0">
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </input>
                        <output>
                            <port id="1">
                                <dim>1</dim>
                                <dim>1</dim>
                                <dim>16</dim>
                            </port>
                        </output>
                    </layer>
                </layers>
                <edges>
                    <edge from-layer="0" from-port="1" to-layer="1" to-port="0"/>
                    <edge from-layer="1" from-port="2" to-layer="2" to-port="0"/>
                </edges>
            </body>
        </layer>
    </layers>
    <edges>
        <edge from-layer="0" from-port="0" to-layer="2" to-port="0"/>
        <edge from-layer="1" from-port="0" to-layer="2" to-port="1"/>
    </edges>
</net>
    )V0G0N";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    sts = reader.ReadNetwork(model.data(), model.length(), &resp);
    ASSERT_EQ(OK, sts) << resp.msg;

    auto network = reader.getNetwork(&resp);
    ASSERT_NE(nullptr, network) << resp.msg;

    CNNLayerPtr layer;
    sts = network->getLayerByName("SomeTI", layer, &resp);
    ASSERT_EQ(OK, sts) << resp.msg;

    auto* ti = dynamic_cast<TensorIterator*>(layer.get());
    ASSERT_NE(nullptr, ti);
    ASSERT_EQ(ti->type, "TensorIterator");

    //  Check Input port mapping
    ASSERT_EQ(ti->input_port_map.size(), 2);
    int i = ti->input_port_map[0].axis == 1 ? 0 : 1;
    ASSERT_EQ(ti->input_port_map[i].axis, 1);
    ASSERT_EQ(ti->input_port_map[i].stride, 1);
    ASSERT_EQ(ti->input_port_map[i].start, 0);
    ASSERT_EQ(ti->input_port_map[i].end, -1);
    ASSERT_EQ(ti->input_port_map[i].part_size, 1);
    ASSERT_EQ(ti->input_port_map[1 - i].axis, -1);
    ASSERT_EQ(ti->input_port_map[1 - i].stride, 1);
    ASSERT_EQ(ti->input_port_map[1 - i].start, 0);
    ASSERT_EQ(ti->input_port_map[1 - i].end, -1);
    ASSERT_EQ(ti->input_port_map[1 - i].part_size, 1);

    //  Check Output port mapping
    ASSERT_EQ(ti->output_port_map.size(), 1);
    ASSERT_EQ(ti->output_port_map[0].axis, 1);
    ASSERT_EQ(ti->output_port_map[0].stride, 1);
    ASSERT_EQ(ti->output_port_map[0].start, 0);
    ASSERT_EQ(ti->output_port_map[0].end, -1);
    ASSERT_EQ(ti->output_port_map[0].part_size, 1);

    //  No back edges
    ASSERT_EQ(ti->back_edges.size(), 1);
    ASSERT_EQ(ti->back_edges[0].from, 0);
    ASSERT_EQ(ti->back_edges[0].to, 1);
    ASSERT_EQ(ti->back_edges[0].axis, -1);
    ASSERT_EQ(ti->back_edges[0].stride, 1);
    ASSERT_EQ(ti->back_edges[0].start, 0);
    ASSERT_EQ(ti->back_edges[0].end, -1);
    ASSERT_EQ(ti->back_edges[0].part_size, 1);
}

TEST_F(CNNNetReaderImplTest, canParseScalar) {
    std::string model = R"V0G0N(
<net batch="1" name="SimpleNet" version="2">
    <layers>
        <layer id="0" name="input" precision="FP32" type="Input">
            <output>
                <port id="0">
                    <dim>1</dim>
                    <dim>5</dim>
                    <dim>16</dim>
                </port>
            </output>
        </layer>
        <layer id="1" name="scalar" precision="FP32" type="Const">
            <output>
                <port id="0"/>
            </output>
            <blobs>
                <custom offset="0" size="4"/>
            </blobs>
        </layer>
        <layer id="2" name="reshape" precision="FP32" type="Reshape">
            <input>
                <port id="0">
                    <dim>1</dim>
                    <dim>5</dim>
                    <dim>16</dim>
                </port>
                <port id="1"/>
            </input>
            <output>
                <port id="2">
                    <dim>90</dim>
                </port>
            </output>
        </layer>
    </layers>
    <edges>
        <edge from-layer="0" from-port="0" to-layer="2" to-port="0"/>
        <edge from-layer="1" from-port="0" to-layer="2" to-port="1"/>
    </edges>
</net>
    )V0G0N";

    CNNNetReaderImpl reader(make_shared<V2FormatParserCreator>());
    sts = reader.ReadNetwork(model.data(), model.length(), &resp);
    ASSERT_EQ(OK, sts) << resp.msg;
    auto blob = make_shared_blob<uint8_t>(TensorDesc(Precision::U8, {4}, Layout::C));
    blob->allocate();
    auto buffer = blob->buffer().as<float*>();
    float SCALAR_VALUE = 90;
    buffer[0] = SCALAR_VALUE;

    sts = reader.SetWeights(blob, &resp);
    ASSERT_EQ(OK, sts) << resp.msg;

    auto net = reader.getNetwork(&resp);

    ASSERT_NE(nullptr, net) << resp.msg;
    CNNLayerPtr layer;
    sts = net->getLayerByName("scalar", layer, &resp);
    ASSERT_EQ(OK, sts) << resp.msg;
    ASSERT_NE(nullptr, layer.get());
    ASSERT_EQ(layer->type, "Const");
    auto actualBlob = layer->blobs.begin()->second;
    ASSERT_EQ(actualBlob->buffer().as<float*>()[0], SCALAR_VALUE);
    auto scalarDesc = layer->outData[0]->getTensorDesc();
    ASSERT_TRUE(scalarDesc.getDims().empty());
    ASSERT_EQ(scalarDesc.getLayout(), SCALAR);
    ASSERT_EQ(scalarDesc.getPrecision(), Precision::FP32);
}

TEST_F(CNNNetReaderImplTest, ReadInThreads) {
    std::string model =
            "<net name=\"PVANET\" version=\"6\" batch=\"1\">"
            "    <layers>"
            "        <layer name=\"data\" type=\"Input\" precision=\"FP32\" id=\"0\">"
            "            <output>"
            "                <port id=\"0\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>544</dim>"
            "                    <dim>992</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_conv\" type=\"Convolution\" precision=\"FP32\" id=\"2\">"
            "            <convolution_data stride-x=\"2\" stride-y=\"2\" pad-x=\"3\" pad-y=\"3\" kernel-x=\"7\" kernel-y=\"7\" output=\"16\" group=\"1\"/>"
            "            <input>"
            "                <port id=\"2\">"
            "                    <dim>1</dim>"
            "                    <dim>3</dim>"
            "                    <dim>544</dim>"
            "                    <dim>992</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"3\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "            <weights offset=\"0\" size=\"9408\"/>"
            "            <biases offset=\"9408\" size=\"64\"/>"
            "        </layer>"
            "        <layer name=\"conv1_1_neg\" type=\"Power\" precision=\"FP32\" id=\"3\">"
            "            <power_data power=\"1\" scale=\"-1\" shift=\"0\"/>"
            "            <input>"
            "                <port id=\"4\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"5\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_concat\" type=\"Concat\" precision=\"FP32\" id=\"4\">"
            "            <concat_data axis=\"1\"/>"
            "            <input>"
            "                <port id=\"6\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "                <port id=\"7\">"
            "                    <dim>1</dim>"
            "                    <dim>16</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"8\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"conv1_1_scale\" type=\"ScaleShift\" precision=\"FP32\" id=\"5\">"
            "            <input>"
            "                <port id=\"9\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"10\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "            <weights offset=\"9472\" size=\"128\"/>"
            "            <biases offset=\"9600\" size=\"128\"/>"
            "        </layer>"
            "        <layer name=\"conv1_1_relu\" type=\"ReLU\" precision=\"FP32\" id=\"6\">"
            "            <data negative_slope=\"0\" engine=\"caffe.ReLUParameter.DEFAULT\"/>"
            "            <input>"
            "                <port id=\"11\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"12\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "        <layer name=\"pool1\" type=\"Pooling\" precision=\"FP32\" id=\"7\">"
            "            <pooling_data kernel-x=\"3\" kernel-y=\"3\" pad-x=\"0\" pad-y=\"0\" stride-x=\"2\" stride-y=\"2\" rounding-type=\"ceil\" pool-method=\"max\"/>"
            "            <input>"
            "                <port id=\"13\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>272</dim>"
            "                    <dim>496</dim>"
            "                </port>"
            "            </input>"
            "            <output>"
            "                <port id=\"14\">"
            "                    <dim>1</dim>"
            "                    <dim>32</dim>"
            "                    <dim>136</dim>"
            "                    <dim>248</dim>"
            "                </port>"
            "            </output>"
            "        </layer>"
            "    </layers>"
            "    <edges>"
            "        <edge from-layer=\"0\" from-port=\"0\" to-layer=\"2\" to-port=\"2\"/>"
            "        <edge from-layer=\"2\" from-port=\"3\" to-layer=\"3\" to-port=\"4\"/>"
            "        <edge from-layer=\"2\" from-port=\"3\" to-layer=\"4\" to-port=\"6\"/>"
            "        <edge from-layer=\"3\" from-port=\"5\" to-layer=\"4\" to-port=\"7\"/>"
            "        <edge from-layer=\"4\" from-port=\"8\" to-layer=\"5\" to-port=\"9\"/>"
            "        <edge from-layer=\"5\" from-port=\"10\" to-layer=\"6\" to-port=\"11\"/>"
            "        <edge from-layer=\"6\" from-port=\"12\" to-layer=\"7\" to-port=\"13\"/>"
            "    </edges>"
            "</net>";
    std::vector<std::thread> threads;
    for (int i = 0; i < 20; i++) {
        threads.push_back(std::thread([i, model]{
                    CNNNetReader networkReader;
                    /** Read network model **/
                    networkReader.ReadNetwork(model.data(), model.length());
                    CNNNetwork network = networkReader.getNetwork();
                    // -----------------------------------------------------------------------------------------------------

                    // --------------------------- 3. Configure input & output ---------------------------------------------

                    // --------------------------- Prepare input blobs -----------------------------------------------------

                    auto input_shapes = network.getInputShapes();
                    std::string input_name;
                    InferenceEngine::SizeVector input_shape;

                    std::tie(input_name, input_shape) = *input_shapes.begin();
                    input_shape[0] = i;  // batch
                    input_shapes[input_name] = input_shape;
                    network.reshape(input_shapes);    // Not synchronized for reshape. This fails due to concurrency.
                    std::this_thread::sleep_for(std::chrono::milliseconds(500));
                    }));
    }

    for (auto& thr : threads)
        thr.join();
}
