require 'rails_helper'

RSpec.describe Api::V1::StatusesController, type: :controller do
  render_views

  let(:user)  { Fabricate(:user) }
  let(:app)   { Fabricate(:application, name: 'Test app', website: 'http://testapp.com') }
  let(:token) { Fabricate(:accessible_access_token, resource_owner_id: user.id, application: app, scopes: scopes) }

  context 'with an oauth token' do
    before do
      allow(controller).to receive(:doorkeeper_token) { token }
    end

    describe 'GET #show' do
      let(:scopes) { 'read:statuses' }
      let(:status) { Fabricate(:status, account: user.account) }

      it 'returns http success' do
        get :show, params: { id: status.id }
        expect(response).to have_http_status(200)
      end

      context 'when post includes filtered terms' do
        let(:status) { Fabricate(:status, text: 'this toot is about that banned word') }

        before do
          user.account.custom_filters.create!(phrase: 'filter1', context: %w(home), action: :hide, keywords_attributes: [{ keyword: 'banned' }, { keyword: 'irrelevant' }])
        end

        it 'returns http success' do
          get :show, params: { id: status.id }
          expect(response).to have_http_status(200)
        end

        it 'returns filter information' do
          get :show, params: { id: status.id }
          json = body_as_json
          expect(json[:filtered][0]).to include({
            filter: a_hash_including({
              id: user.account.custom_filters.first.id.to_s,
              title: 'filter1',
              filter_action: 'hide',
            }),
            keyword_matches: ['banned'],
          })
        end
      end

      context 'when post is explicitly filtered' do
        let(:status) { Fabricate(:status, text: 'hello world') }

        before do
          filter = user.account.custom_filters.create!(phrase: 'filter1', context: %w(home), action: :hide)
          filter.statuses.create!(status_id: status.id)
        end

        it 'returns http success' do
          get :show, params: { id: status.id }
          expect(response).to have_http_status(200)
        end

        it 'returns filter information' do
          get :show, params: { id: status.id }
          json = body_as_json
          expect(json[:filtered][0]).to include({
            filter: a_hash_including({
              id: user.account.custom_filters.first.id.to_s,
              title: 'filter1',
              filter_action: 'hide',
            }),
            status_matches: [status.id.to_s],
          })
        end
      end

      context 'when reblog includes filtered terms' do
        let(:status) { Fabricate(:status, reblog: Fabricate(:status, text: 'this toot is about that banned word')) }

        before do
          user.account.custom_filters.create!(phrase: 'filter1', context: %w(home), action: :hide, keywords_attributes: [{ keyword: 'banned' }, { keyword: 'irrelevant' }])
        end

        it 'returns http success' do
          get :show, params: { id: status.id }
          expect(response).to have_http_status(200)
        end

        it 'returns filter information' do
          get :show, params: { id: status.id }
          json = body_as_json
          expect(json[:reblog][:filtered][0]).to include({
            filter: a_hash_including({
              id: user.account.custom_filters.first.id.to_s,
              title: 'filter1',
              filter_action: 'hide',
            }),
            keyword_matches: ['banned'],
          })
        end
      end
    end

    describe 'GET #context' do
      let(:scopes) { 'read:statuses' }
      let(:status) { Fabricate(:status, account: user.account) }

      before do
        Fabricate(:status, account: user.account, thread: status)
      end

      it 'returns http success' do
        get :context, params: { id: status.id }
        expect(response).to have_http_status(200)
      end
    end

    describe 'POST #create' do
      let(:scopes) { 'write:statuses' }

      context do
        before do
          post :create, params: { status: 'Hello world' }
        end

        it 'returns http success' do
          expect(response).to have_http_status(200)
        end

        it 'returns rate limit headers' do
          expect(response.headers['X-RateLimit-Limit']).to eq RateLimiter::FAMILIES[:statuses][:limit].to_s
          expect(response.headers['X-RateLimit-Remaining']).to eq (RateLimiter::FAMILIES[:statuses][:limit] - 1).to_s
        end
      end

      context 'with missing parameters' do
        before do
          post :create, params: {}
        end

        it 'returns http unprocessable entity' do
          expect(response).to have_http_status(422)
        end

        it 'returns rate limit headers' do
          expect(response.headers['X-RateLimit-Limit']).to eq RateLimiter::FAMILIES[:statuses][:limit].to_s
        end
      end

      context 'when exceeding rate limit' do
        before do
          rate_limiter = RateLimiter.new(user.account, family: :statuses)
          300.times { rate_limiter.record! }
          post :create, params: { status: 'Hello world' }
        end

        it 'returns http too many requests' do
          expect(response).to have_http_status(429)
        end

        it 'returns rate limit headers' do
          expect(response.headers['X-RateLimit-Limit']).to eq RateLimiter::FAMILIES[:statuses][:limit].to_s
          expect(response.headers['X-RateLimit-Remaining']).to eq '0'
        end
      end

      context 'with missing thread' do
        subject { post :create, params: params }

        let(:params) { { status: 'Hello world', in_reply_to_id: 0 } }

        it 'returns http not found' do
          subject

          expect(response).to have_http_status(404)
        end
      end

      context 'when scheduling a status' do
        subject { post :create, params: params }

        let(:params) { { status: 'Hello world', scheduled_at: 10.minutes.from_now } }
        let(:account) { user.account }

        it 'returns HTTP 200' do
          subject

          expect(response).to have_http_status(200)
        end

        it 'creates a scheduled status' do
          expect { subject }.to change { account.scheduled_statuses.count }.from(0).to(1)
        end

        context 'when the scheduling time is less than 5 minutes' do
          let(:params) { { status: 'Hello world', scheduled_at: 4.minutes.from_now } }

          it 'does not create a scheduled status', :aggregate_failures do
            subject

            expect(response).to have_http_status(422)
            expect(account.scheduled_statuses).to be_empty
          end
        end
      end
    end

    describe 'DELETE #destroy' do
      let(:scopes) { 'write:statuses' }
      let(:status) { Fabricate(:status, account: user.account) }

      before do
        post :destroy, params: { id: status.id }
      end

      it 'returns http success' do
        expect(response).to have_http_status(200)
      end

      it 'removes the status' do
        expect(Status.find_by(id: status.id)).to be nil
      end
    end

    describe 'PUT #update' do
      let(:scopes) { 'write:statuses' }
      let(:status) { Fabricate(:status, account: user.account) }

      before do
        put :update, params: { id: status.id, status: 'I am updated' }
      end

      it 'returns http success' do
        expect(response).to have_http_status(200)
      end

      it 'updates the status' do
        expect(status.reload.text).to eq 'I am updated'
      end
    end
  end

  context 'without an oauth token' do
    before do
      allow(controller).to receive(:doorkeeper_token) { nil }
    end

    context 'with a private status' do
      let(:status) { Fabricate(:status, account: user.account, visibility: :private) }

      describe 'GET #show' do
        it 'returns http unauthorized' do
          get :show, params: { id: status.id }
          expect(response).to have_http_status(404)
        end
      end

      describe 'GET #context' do
        before do
          Fabricate(:status, account: user.account, thread: status)
        end

        it 'returns http unauthorized' do
          get :context, params: { id: status.id }
          expect(response).to have_http_status(404)
        end
      end
    end

    context 'with a public status' do
      let(:status) { Fabricate(:status, account: user.account, visibility: :public) }

      describe 'GET #show' do
        it 'returns http success' do
          get :show, params: { id: status.id }
          expect(response).to have_http_status(200)
        end
      end

      describe 'GET #context' do
        before do
          Fabricate(:status, account: user.account, thread: status)
        end

        it 'returns http success' do
          get :context, params: { id: status.id }
          expect(response).to have_http_status(200)
        end
      end
    end
  end
end
