﻿// Copyright (c) Microsoft Open Technologies, Inc. All rights reserved. See License.txt in the project root for license information.

using System.Net.Http;
using System.Web.Http.Hosting;
using System.Web.Http.Routing;
using Microsoft.TestCommon;

namespace System.Web.Http.OData.Routing
{
    public class ODataUrlHelperExtensionsTest
    {
        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfHelperRequestHasNoConfiguration()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            UrlHelper urlHelper = new UrlHelper(request);

            Assert.Null(urlHelper.GenerateLinkDirectly("OData", "odataPath"));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfNoRouteCalledODataFound()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            config.Routes.MapHttpRoute("NotOData", "{controller}");
            request.SetConfiguration(config);
            UrlHelper urlHelper = new UrlHelper(request);

            Assert.Null(urlHelper.GenerateLinkDirectly("OData", "odataPath"));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfRouteTemplateDoesNotEndInODataPath()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            config.Routes.MapHttpRoute("OData", "prefix/{*notODataPath}");
            request.SetConfiguration(config);
            UrlHelper urlHelper = new UrlHelper(request);

            Assert.Null(urlHelper.GenerateLinkDirectly("OData", "odataPath"));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfRouteTemplateHasParameterInPrefix()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            config.Routes.MapHttpRoute("OData", "{prefix}/{*odataPath}");
            request.SetConfiguration(config);
            UrlHelper urlHelper = new UrlHelper(request);

            Assert.Null(urlHelper.GenerateLinkDirectly("OData", "odataPath"));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsUri_IfConditionsSatisfied()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            config.Routes.MapHttpRoute("OData", "prefix/{*odataPath}");
            request.SetConfiguration(config);
            request.Properties["MS_HttpRouteData"] = new HttpRouteData(new HttpRoute());
            UrlHelper urlHelper = new UrlHelper(request);
            string odataPath = "odataPath";

            // Test that the link generated by ODataLink matches the one generated by Url.Link
            Assert.Equal(
                urlHelper.Link("OData", new { odataPath = odataPath }),
                urlHelper.GenerateLinkDirectly("OData", odataPath));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsEncodedUri_IfNecessary()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/"));
            config.Routes.MapHttpRoute("OData", "{*odataPath}");
            request.Properties["MS_HttpConfiguration"] = config;
            request.Properties["MS_HttpRouteData"] = new HttpRouteData(new HttpRoute());
            UrlHelper urlHelper = new UrlHelper(request);
            string odataPath = "Customers('$&+,/:;=?@ <>#%{}|\\^~[]` ')";

            // Test that the link generated by ODataLink matches the one generated by Url.Link
            Assert.Equal(
                urlHelper.Link("OData", new { odataPath = odataPath }),
                urlHelper.GenerateLinkDirectly("OData", odataPath));
        }
    }
}
