/**
 * Copyright 2015 Mohiva Organisation (license at mohiva dot com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mohiva.play.silhouette.api

import play.api.http.HttpErrorHandler
import play.api.i18n.Messages
import play.api.mvc.{ RequestHeader, Result }

import scala.concurrent.Future

/**
 * Can be mixed into the `HttpErrorHandler` object to define a global behaviour
 * for unauthorized and forbidden endpoints.
 */
trait SecuredErrorHandler {
  this: HttpErrorHandler =>

  /**
   * Called when a user is not authenticated.
   *
   * As defined by RFC 2616, the status code of the response should be 401 Unauthorized.
   *
   * @param request The request header.
   * @param messages The messages for the current language.
   * @return The result to send to the client.
   */
  def onNotAuthenticated(request: RequestHeader, messages: Messages): Option[Future[Result]] = None

  /**
   * Called when a user is authenticated but not authorized.
   *
   * As defined by RFC 2616, the status code of the response should be 403 Forbidden.
   *
   * @param request The request header.
   * @param messages The messages for the current language.
   * @return The result to send to the client.
   */
  def onNotAuthorized(request: RequestHeader, messages: Messages): Option[Future[Result]] = None
}
