/* eslint-env jest */
import 'flat-map-polyfill'
import { readdir, readFile, remove } from 'fs-extra'
import { nextBuild } from 'next-test-utils'
import { join } from 'path'

const fixturesDir = join(__dirname, '../..', 'css-fixtures')

describe('Invalid CSS in _document', () => {
  const appDir = join(fixturesDir, 'invalid-module-document')

  beforeAll(async () => {
    await remove(join(appDir, '.next'))
  })

  it('should fail to build', async () => {
    const { code, stderr } = await nextBuild(appDir, [], {
      stderr: true,
    })
    expect(code).not.toBe(0)
    expect(stderr).toContain('Failed to compile')
    expect(stderr).toContain('styles.module.css')
    expect(stderr).toMatch(
      /CSS.*cannot.*be imported within.*pages[\\/]_document\.js/
    )
    expect(stderr).toMatch(/Location:.*pages[\\/]_document\.js/)
  })
})

describe('Invalid Global CSS', () => {
  const appDir = join(fixturesDir, 'invalid-global')

  beforeAll(async () => {
    await remove(join(appDir, '.next'))
  })

  it('should fail to build', async () => {
    const { code, stderr } = await nextBuild(appDir, [], {
      stderr: true,
    })
    expect(code).not.toBe(0)
    expect(stderr).toContain('Failed to compile')
    expect(stderr).toContain('styles/global.css')
    expect(stderr).toMatch(
      /Please move all first-party global CSS imports.*?pages(\/|\\)_app/
    )
    expect(stderr).toMatch(/Location:.*pages[\\/]index\.js/)
  })
})

describe('Valid Global CSS from npm', () => {
  const appDir = join(fixturesDir, 'import-global-from-module')

  beforeAll(async () => {
    await remove(join(appDir, '.next'))
  })

  it('should compile successfully', async () => {
    const { code, stdout } = await nextBuild(appDir, [], {
      stdout: true,
    })
    expect(code).toBe(0)
    expect(stdout).toMatch(/Compiled successfully/)
  })

  it(`should've emitted a single CSS file`, async () => {
    const cssFolder = join(appDir, '.next/static/css')

    const files = await readdir(cssFolder)
    const cssFiles = files.filter((f) => /\.css$/.test(f))

    expect(cssFiles.length).toBe(1)
    const cssContent = await readFile(join(cssFolder, cssFiles[0]), 'utf8')
    expect(cssContent.replace(/\/\*.*?\*\//g, '').trim()).toMatchInlineSnapshot(
      `".red-text{color:\\"red\\"}"`
    )
  })
})

describe('Invalid Global CSS with Custom App', () => {
  const appDir = join(fixturesDir, 'invalid-global-with-app')

  beforeAll(async () => {
    await remove(join(appDir, '.next'))
  })

  it('should fail to build', async () => {
    const { code, stderr } = await nextBuild(appDir, [], {
      stderr: true,
    })
    expect(code).not.toBe(0)
    expect(stderr).toContain('Failed to compile')
    expect(stderr).toContain('styles/global.css')
    expect(stderr).toMatch(
      /Please move all first-party global CSS imports.*?pages(\/|\\)_app/
    )
    expect(stderr).toMatch(/Location:.*pages[\\/]index\.js/)
  })
})

describe('Valid and Invalid Global CSS with Custom App', () => {
  const appDir = join(fixturesDir, 'valid-and-invalid-global')

  beforeAll(async () => {
    await remove(join(appDir, '.next'))
  })

  it('should fail to build', async () => {
    const { code, stderr } = await nextBuild(appDir, [], {
      stderr: true,
    })
    expect(code).not.toBe(0)
    expect(stderr).toContain('Failed to compile')
    expect(stderr).toContain('styles/global.css')
    expect(stderr).toContain('Please move all first-party global CSS imports')
    expect(stderr).toMatch(/Location:.*pages[\\/]index\.js/)
  })
})
