/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2021 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifndef TGUI_BACKEND_TEXT_SFML_HPP
#define TGUI_BACKEND_TEXT_SFML_HPP

#include <TGUI/BackendText.hpp>
#include <SFML/Graphics/Text.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Text implementation that makes use of SFML
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API BackendTextSFML : public BackendTextBase
    {
    public:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the size of the text
        /// @return Size of the bounding box around the text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Vector2f getSize() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text
        /// @param string  Text that should be displayed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setString(const String& string) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the size of the characters
        /// @param characterSize  Maximum size available for characters above the baseline
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setCharacterSize(unsigned int characterSize) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the color of the text
        /// @param color  Text color
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setFillColor(const Color& color) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the color of the text outline
        /// @param color  Outline color
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setOutlineColor(const Color& color) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the thickness of the text outline
        /// @param thickness  Outline thickness
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setOutlineThickness(float thickness) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text style
        /// @param style  New text style
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setStyle(TextStyles style) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the font used by the text
        /// @param font  New text font
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setFont(const Font& font) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the top-left position of the character at the provided index
        /// @param index  Index of the character for which the position should be returned
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Vector2f findCharacterPos(std::size_t index) const override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a reference to the internal SFML text
        /// @return Reference to internal text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        sf::Text& getInternalText();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a const reference to the internal SFML text
        /// @return Const reference to internal text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        const sf::Text& getInternalText() const;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Updates the caches text dimensions
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void recalculateSize();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Calculates the extra vertical space that should be included between each line of text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        float calculateExtraLineSpace(const sf::Font* font, unsigned int characterSize, unsigned int style);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        sf::Text m_text;
        Vector2f m_size;
    };
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_BACKEND_TEXT_SFML_HPP
