/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2021 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifndef TGUI_BACKEND_RENDER_TARGET_SFML_HPP
#define TGUI_BACKEND_RENDER_TARGET_SFML_HPP

#include <TGUI/BackendRenderTarget.hpp>

#include <SFML/Graphics/View.hpp>
#include <SFML/Graphics/RenderStates.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace sf
{
    class RenderTarget;
}

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Render target that uses SFML to draw the gui
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API BackendRenderTargetSFML : public BackendRenderTargetBase
    {
    public:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructs the render target
        ///
        /// @param target  The SFML render target (typically sf::RenderWindow) on which the gui should be drawn
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        BackendRenderTargetSFML(sf::RenderTarget& target);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the SFML render target on which the gui is being drawn
        ///
        /// @return The SFML render target that is used by the gui
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        sf::RenderTarget* getTarget() const;

#ifndef TGUI_REMOVE_DEPRECATED_CODE
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Informs the render target about which part of the window is used for rendering
        ///
        /// @param view     Defines which part of the gui is being shown
        /// @param viewport Defines which part of the window is being rendered to
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setView(FloatRect view, FloatRect viewport) override;
        using BackendRenderTargetBase::setView;
#endif

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draws the gui and all of its widgets
        ///
        /// @param root  Root container that holds all widgets in the gui
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void drawGui(const std::shared_ptr<RootContainer>& root) override;

#ifndef TGUI_REMOVE_DEPRECATED_CODE
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds another clipping region
        ///
        /// @param states  Render states to use for drawing
        /// @param rect  The clipping region
        ///
        /// If multiple clipping regions were added then contents is only shown in the intersection of all regions.
        ///
        /// @warning Every call to addClippingLayer must have a matching call to removeClippingLayer.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void addClippingLayer(const RenderStates& states, FloatRect rect) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes the last added clipping region
        ///
        /// @warning The addClippingLayer function must have been called before calling this function.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void removeClippingLayer() override;
#endif

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draws a texture
        ///
        /// @param states  Render states to use for drawing
        /// @param sprite  Image to draw
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void drawSprite(const RenderStates& states, const Sprite& sprite) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draws some text
        ///
        /// @param states  Render states to use for drawing
        /// @param text    Text to draw
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void drawText(const RenderStates& states, const Text& text) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draws one or more triangles (using the color that is specified in the vertices)
        ///
        /// @param states       Render states to use for drawing
        /// @param vertices     Pointer to first element in array of vertices
        /// @param vertexCount  Amount of elements in the vertex array
        /// @param indices      Pointer to first element in array of indices
        /// @param indexCount   Amount of elements in the indices array
        ///
        /// If indices is a nullptr then vertexCount must be a multiple of 3 and each set of 3 vertices will be seen as a triangle.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void drawTriangles(const RenderStates& states, const Vertex* vertices, std::size_t vertexCount, const int* indices = nullptr, std::size_t indexCount = 0) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Called from addClippingLayer and removeClippingLayer to apply the clipping
        ///
        /// @param clipRect      View rectangle to apply
        /// @param clipViewport  Viewport to apply
        ///
        /// Both rectangles may be empty when nothing that will be drawn is going to be visible.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateClipping(FloatRect clipRect, FloatRect clipViewport) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Converts tgui::RenderStates to sf::RenderStates
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        sf::RenderStates convertRenderStates(const RenderStates& states);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        sf::RenderTarget* m_target = nullptr;

#ifndef TGUI_REMOVE_DEPRECATED_CODE
        TGUI_DEPRECATED("Calculate using m_viewRect, m_viewport and m_targetSize instead. See new updateClipping function.") sf::View m_view;
        TGUI_DEPRECATED("Use m_clipLayers instead") std::vector<std::pair<FloatRect, sf::View>> m_clippingLayers;
#endif
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_BACKEND_RENDER_TARGET_SFML_HPP
