/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2021 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifndef TGUI_BACKEND_GUI_SDL_HPP
#define TGUI_BACKEND_GUI_SDL_HPP


#include <TGUI/GuiBase.hpp>

#include <SDL.h>
#include <SDL_touch.h>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    class BackendRenderTargetSDL;

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Gui class for the SDL backend
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API GuiSDL : public GuiBase
    {
    public:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default constructor
        ///
        /// @warning If you use this constructor then you will still have to call setWindow before using the gui.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GuiSDL();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructs the gui and set the window on which the gui should be drawn
        ///
        /// @param window  The SDL window that will be used by the gui
        ///
        /// If you use this constructor then you will no longer have to call setWindow yourself.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GuiSDL(SDL_Window* window);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the window on which the gui should be drawn
        ///
        /// @param window  The SDL window that will be used by the gui
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setWindow(SDL_Window* window);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Passes the event to the widgets
        ///
        /// @param event  The event that was polled from the window
        ///
        /// @return Has the event been consumed?
        ///         When this function returns false, then the event was ignored by all widgets.
        ///
        /// You should call this function in your event loop.
        /// @code
        /// SDL_Event event;
        /// while (SDL_PollEvent(&event) != 0) {
        ///     gui.handleEvent(event);
        /// }
        /// @endcode
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool handleEvent(const SDL_Event& event);
        using GuiBase::handleEvent;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draws all the widgets that were added to the gui
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void draw() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Give the gui control over the main loop
        ///
        /// This function is only intended in cases where your program only needs to respond to gui events.
        /// For multimedia applications, games, or other programs where you want a high framerate or do a lot of processing
        /// in the main loop, you should use your own main loop.
        ///
        /// You can consider this function to execute something similar (but not identical) to the following code:
        /// @code
        /// while (!quit)
        /// {
        ///     SDL_Event event;
        ///     while (SDL_PollEvent(&event) != 0)
        ///     {
        ///         gui.handleEvent(event);
        ///         if (e.type == SDL_QUIT)
        ///             quit = true;
        ///     }
        ///
        ///     glClear(GL_COLOR_BUFFER_BIT);
        ///     gui.draw();
        ///     SDL_GL_SwapWindow(window);
        /// }
        /// @endcode
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mainLoop() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Helper function that converts an SDL event to a TGUI event
        /// @param eventSFML  The input SDL event
        /// @param eventTGUI  The output TGUI event
        /// @return Did the SDL event match on a TGUI event and has the output event been written to?
        ///
        /// You typically don't need this function as you can call handleEvent directly with an SDL_Event.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool convertEvent(const SDL_Event& eventSDL, Event& eventTGUI);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Function that is called when constructing the Gui
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void init() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Updates the view and changes the size of the root container when needed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateContainerSize() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        std::shared_ptr<BackendRenderTargetSDL> m_renderTarget = nullptr;

        Vector2i m_windowSize;

        bool m_touchFirstFingerDown = false;
        SDL_FingerID m_touchFirstFingerId = 0;      // Only valid if m_touchFirstFingerDown is true
        SDL_TouchID  m_touchFirstFingerTouchId = 0; // Only valid if m_touchFirstFingerDown is true

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    };
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_BACKEND_GUI_SDL_HPP
