/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2021 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifndef TGUI_BACKEND_TEXT_SDL_HPP
#define TGUI_BACKEND_TEXT_SDL_HPP

#include <TGUI/BackendText.hpp>
#include <vector>

#include <SDL.h>
#include <SDL_ttf.h>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

typedef unsigned int GLuint;

namespace tgui
{
    class BackendFontSDL;

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Text implementation that makes use of SDL
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API BackendTextSDL : public BackendTextBase
    {
    public:

        /// Internal texture per line of text
        struct LineTexture
        {
            SDL_Rect bounding;
            GLuint textureId;
        };


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        BackendTextSDL();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ~BackendTextSDL();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the size of the text
        /// @return Size of the bounding box around the text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Vector2f getSize() override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text
        /// @param string  Text that should be displayed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setString(const String& string) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the size of the characters
        /// @param characterSize  Maximum size available for characters above the baseline
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setCharacterSize(unsigned int characterSize) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the color of the text
        /// @param color  Text color
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setFillColor(const Color& color) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the color of the text outline
        /// @param color  Outline color
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setOutlineColor(const Color& color) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the thickness of the text outline
        /// @param thickness  Outline thickness
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setOutlineThickness(float thickness) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text style
        /// @param style  New text style
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setStyle(TextStyles style) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the font used by the text
        /// @param font  New text font
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setFont(const Font& font) override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the top-left position of the character at the provided index
        /// @param index  Index of the character for which the position should be returned
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Vector2f findCharacterPos(std::size_t index) const override;


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a reference to the internal SDL textures
        /// @return Reference to internal texture
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        const std::vector<LineTexture>& getInternalTextures();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns a reference to the internal SDL outline textures
        /// @return Reference to internal outline texture
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        const std::vector<LineTexture>& getInternalOutlineTextures();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Updates the internal surfaces and textures.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool updateTextures();


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Create a texture for a single line of text
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        LineTexture createLineTexture(TTF_Font* font, int verticalOffset, std::string line, const SDL_Color& color, int outline);


        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        std::vector<LineTexture> m_textures;
        std::vector<LineTexture> m_outlineTextures;

        std::shared_ptr<BackendFontSDL> m_font = nullptr;
        unsigned int m_characterSize = 0;
        int m_fontOutline = 0;
        int m_fontStyle = TTF_STYLE_NORMAL;
        SDL_Color m_textColor = {0, 0, 0, 255};
        SDL_Color m_outlineColor = {0, 0, 0, 255};
        std::vector<String> m_lines;

        Vector2f m_size;
        bool m_texturesValid = false;
    };
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_BACKEND_TEXT_SDL_HPP
