/*
 *
 *  Copyright 2012 Netflix, Inc.
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 */
package com.netflix.simianarmy.basic.chaos;

import com.netflix.simianarmy.MonkeyRecorder.Event;
import com.netflix.simianarmy.chaos.ChaosCrawler.InstanceGroup;
import com.netflix.simianarmy.chaos.ChaosType;

/**
 * The Class CloudFormationChaosMonkey. Strips out the random string generated by the CloudFormation api in
 * the instance group name of the ASG we want to kill instances on
 */
public class CloudFormationChaosMonkey extends BasicChaosMonkey {

    /**
     * Instantiates a new cloud formation chaos monkey.
     * @param ctx
     *            the ctx
     */
    public CloudFormationChaosMonkey(Context ctx) {
        super(ctx);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean isGroupEnabled(InstanceGroup group) {
        InstanceGroup noSuffixGroup = noSuffixInstanceGroup(group);
        return super.isGroupEnabled(noSuffixGroup);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Event terminateInstance(InstanceGroup group, String inst, ChaosType chaosType) {
        InstanceGroup noSuffixGroup = noSuffixInstanceGroup(group);
        return super.terminateInstance(noSuffixGroup, inst, chaosType);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean isMaxTerminationCountExceeded(InstanceGroup group) {
        InstanceGroup noSuffixGroup = noSuffixInstanceGroup(group);
        return super.isMaxTerminationCountExceeded(noSuffixGroup);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected double getEffectiveProbability(InstanceGroup group) {
        InstanceGroup noSuffixGroup = noSuffixInstanceGroup(group);
        if (!super.isGroupEnabled(noSuffixGroup)) {
            return 0;
        }
        return getEffectiveProbabilityFromCfg(noSuffixGroup);
    }

    /**
     * Returns the lastOptInTimeInMilliseconds parameter for a group omitting the
     * randomly generated suffix.
     */
    @Override
    protected long getLastOptInMilliseconds(InstanceGroup group) {
        InstanceGroup noSuffixGroup = noSuffixInstanceGroup(group);
        return super.getLastOptInMilliseconds(noSuffixGroup);
    }

    /**
     * Return a copy of the instance group removing the randomly generated suffix from
     * its name.
     */
    public InstanceGroup noSuffixInstanceGroup(InstanceGroup group) {
        String newName = group.name().replaceAll("(-)([^-]*$)", "");
        InstanceGroup noSuffixGroup = group.copyAs(newName);
        return noSuffixGroup;
    }
}
