/*
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package genai.imagegeneration;

// [START googlegenaisdk_imggen_style_reference_with_txt_img]

import com.google.genai.Client;
import com.google.genai.types.EditImageConfig;
import com.google.genai.types.EditImageResponse;
import com.google.genai.types.GeneratedImage;
import com.google.genai.types.Image;
import com.google.genai.types.StyleReferenceConfig;
import com.google.genai.types.StyleReferenceImage;
import java.util.List;
import java.util.Optional;

public class ImageGenStyleReferenceWithTextAndImage {

  public static void main(String[] args) {
    // TODO(developer): Replace these variables before running the sample.
    String modelId = "imagen-3.0-capability-001";
    String outputGcsUri = "gs://your-bucket/your-prefix";
    styleCustomization(modelId, outputGcsUri);
  }

  // Generates an image using style customization with a style reference image and text prompt.
  public static Optional<String> styleCustomization(String modelId, String outputGcsUri) {
    // Client Initialization. Once created, it can be reused for multiple requests.
    try (Client client = Client.builder().location("global").vertexAI(true).build()) {
      // Create a style reference image of a neon sign stored in Google Cloud Storage
      // using https://storage.googleapis.com/cloud-samples-data/generative-ai/image/neon.png
      StyleReferenceImage styleReferenceImage =
          StyleReferenceImage.builder()
              .referenceId(1)
              .referenceImage(
                  Image.builder()
                      .gcsUri("gs://cloud-samples-data/generative-ai/image/neon.png")
                      .build())
              .config(StyleReferenceConfig.builder().styleDescription("neon sign").build())
              .build();

      // The `[1]` in the prompt refers to the `referenceId` assigned to the style reference image.
      EditImageResponse imageResponse =
          client.models.editImage(
              modelId,
              "generate an image of a neon sign [1] with the words: have a great day",
              List.of(styleReferenceImage),
              EditImageConfig.builder()
                  .editMode("EDIT_MODE_DEFAULT")
                  .numberOfImages(1)
                  .safetyFilterLevel("BLOCK_MEDIUM_AND_ABOVE")
                  .personGeneration("ALLOW_ADULT")
                  .outputGcsUri(outputGcsUri)
                  .build());

      Image generatedImage =
          imageResponse
              .generatedImages()
              .flatMap(generatedImages -> generatedImages.stream().findFirst())
              .flatMap(GeneratedImage::image)
              .orElseThrow(() -> new IllegalStateException("No image was generated by the model."));

      generatedImage.gcsUri().ifPresent(System.out::println);
      // Example response:
      // gs://your-bucket/your-prefix

      return generatedImage.gcsUri();
    }
  }
}
// [END googlegenaisdk_imggen_style_reference_with_txt_img]
