// Copyright 2020 The MediaPipe Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#if defined(__ANDROID__)
#include "mediapipe/util/android/asset_manager_util.h"
#else
#include <fstream>
#include <iostream>
#endif

#include "mediapipe/framework/calculator_framework.h"
#include "mediapipe/framework/port/ret_check.h"
#include "mediapipe/framework/port/status.h"
#include "mediapipe/gpu/gl_calculator_helper.h"
#include "mediapipe/gpu/shader_util.h"
#include "mediapipe/graphs/object_detection_3d/calculators/camera_parameters.pb.h"
#include "mediapipe/graphs/object_detection_3d/calculators/gl_animation_overlay_calculator.pb.h"
#include "mediapipe/graphs/object_detection_3d/calculators/model_matrix.pb.h"

namespace mediapipe {

namespace {

#if defined(GL_DEBUG)
#define GLCHECK(command) \
  command;               \
  if (int err = glGetError()) LOG(ERROR) << "GL error detected: " << err;
#else
#define GLCHECK(command) command
#endif

// For ease of use, we prefer ImageFrame on Android and GpuBuffer otherwise.
#if defined(__ANDROID__)
typedef ImageFrame AssetTextureFormat;
#else
typedef GpuBuffer AssetTextureFormat;
#endif

enum { ATTRIB_VERTEX, ATTRIB_TEXTURE_POSITION, NUM_ATTRIBUTES };
static const int kNumMatrixEntries = 16;

// Hard-coded MVP Matrix for testing.
static const float kModelMatrix[] = {0.83704215,  -0.36174262, 0.41049102, 0.0,
                                     0.06146407,  0.8076706,   0.5864218,  0.0,
                                     -0.54367524, -0.4656292,  0.69828844, 0.0,
                                     0.0,         0.0,         -98.64117,  1.0};

// Loads a texture from an input side packet, and streams in an animation file
// from a filename given in another input side packet, and renders the animation
// over the screen according to the input timestamp and desired animation FPS.
//
// Inputs:
//   VIDEO (GpuBuffer, optional):
//     If provided, the input buffer will be assumed to be unique, and will be
//     consumed by this calculator and rendered to directly.  The output video
//     buffer will then be the released reference to the input video buffer.
//   MODEL_MATRICES (TimedModelMatrixProtoList, optional):
//     If provided, will set the model matrices for the objects to be rendered
//     during future rendering calls.
//
// Input side packets:
//   TEXTURE (ImageFrame on Android / GpuBuffer on iOS, required):
//     Texture to use with animation file.
//   ANIMATION_ASSET (String, required):
//     Path of animation file to load and render. Should be generated by
//     //java/com/google/android/apps/motionstills/SimpleObjEncryptor with
//     --compressed_mode=true.  See comments and documentation there for more
//     information on custom .obj.uuu file format.
//   CAMERA_PARAMETERS_PROTO_STRING (String, optional):
//     Serialized proto std::string of CameraParametersProto. We need this to
//     get the right aspect ratio and field of view.
// Options:
//   aspect_ratio: the ratio between the rendered image width and height.
//     It will be ignored if CAMERA_PARAMETERS_PROTO_STRING input side packet
//     is provided.
//   vertical_fov_degrees: vertical field of view in degrees.
//     It will be ignored if CAMERA_PARAMETERS_PROTO_STRING input side packet
//     is provided.
//   z_clipping_plane_near: near plane value for z-clipping.
//   z_clipping_plane_far: far plane value for z-clipping.
//   animation_speed_fps: speed at which to cycle through animation frames (in
//     frames per second).
//
// Outputs:
//   OUTPUT, or index 0 (GpuBuffer):
//     Frames filled with the given texture.

// Simple helper-struct for containing the parsed geometry data from a 3D
// animation frame for rendering.

struct TriangleMesh {
  int index_count = 0;  // Needed for glDrawElements rendering call
  std::unique_ptr<float[]> vertices = nullptr;
  std::unique_ptr<float[]> texture_coords = nullptr;
  std::unique_ptr<int16[]> triangle_indices = nullptr;
};

typedef std::unique_ptr<float[]> ModelMatrix;

}  // namespace

class GlAnimationOverlayCalculator : public CalculatorBase {
 public:
  GlAnimationOverlayCalculator() {}
  ~GlAnimationOverlayCalculator();

  static ::mediapipe::Status GetContract(CalculatorContract *cc);

  ::mediapipe::Status Open(CalculatorContext *cc) override;
  ::mediapipe::Status Process(CalculatorContext *cc) override;

 private:
  bool has_video_stream_ = false;
  bool has_model_matrix_stream_ = false;
  bool has_mask_model_matrix_stream_ = false;
  bool has_occlusion_mask_ = false;

  GlCalculatorHelper helper_;
  bool initialized_ = false;
  GlTexture texture_;
  GlTexture mask_texture_;

  GLuint renderbuffer_ = 0;
  bool depth_buffer_created_ = false;

  GLuint program_ = 0;
  GLint texture_uniform_ = -1;
  GLint perspective_matrix_uniform_ = -1;
  GLint model_matrix_uniform_ = -1;

  std::vector<TriangleMesh> triangle_meshes_;
  std::vector<TriangleMesh> mask_meshes_;
  Timestamp animation_start_time_;
  int frame_count_ = 0;
  float animation_speed_fps_;

  std::vector<ModelMatrix> current_model_matrices_;
  std::vector<ModelMatrix> current_mask_model_matrices_;

  // Perspective matrix for rendering, to be applied to all model matrices
  // prior to passing through to the shader as a MVP matrix.  Initialized during
  // first image packet read.
  float perspective_matrix_[kNumMatrixEntries];

  void ComputeAspectRatioAndFovFromCameraParameters(
      const CameraParametersProto &camera_parameters, float *aspect_ratio,
      float *vertical_fov_degrees);
  int GetAnimationFrameIndex(Timestamp timestamp);
  ::mediapipe::Status GlSetup();
  ::mediapipe::Status GlBind(const TriangleMesh &triangle_mesh,
                             const GlTexture &texture);
  ::mediapipe::Status GlRender(const TriangleMesh &triangle_mesh,
                               const float *model_matrix);
  void InitializePerspectiveMatrix(float aspect_ratio,
                                   float vertical_fov_degrees, float z_near,
                                   float z_far);
  void LoadModelMatrices(const TimedModelMatrixProtoList &model_matrices,
                         std::vector<ModelMatrix> *current_model_matrices);

#if !defined(__ANDROID__)
  // Asset loading routine for all non-Android platforms.
  bool LoadAnimation(const std::string &filename);
#else
  // Asset loading for all Android platforms.
  bool LoadAnimationAndroid(const std::string &filename,
                            std::vector<TriangleMesh> *mesh);
  bool ReadBytesFromAsset(AAsset *asset, void *buffer, int num_bytes_to_read);
#endif
};
REGISTER_CALCULATOR(GlAnimationOverlayCalculator);

// static
::mediapipe::Status GlAnimationOverlayCalculator::GetContract(
    CalculatorContract *cc) {
  MP_RETURN_IF_ERROR(
      GlCalculatorHelper::SetupInputSidePackets(&(cc->InputSidePackets())));
  if (cc->Inputs().HasTag("VIDEO")) {
    // Currently used only for size and timestamp.
    cc->Inputs().Tag("VIDEO").Set<GpuBuffer>();
  }
  TagOrIndex(&(cc->Outputs()), "OUTPUT", 0).Set<GpuBuffer>();

  if (cc->Inputs().HasTag("MODEL_MATRICES")) {
    cc->Inputs().Tag("MODEL_MATRICES").Set<TimedModelMatrixProtoList>();
  }
  if (cc->Inputs().HasTag("MASK_MODEL_MATRICES")) {
    cc->Inputs().Tag("MASK_MODEL_MATRICES").Set<TimedModelMatrixProtoList>();
  }

  cc->InputSidePackets().Tag("TEXTURE").Set<AssetTextureFormat>();
  cc->InputSidePackets().Tag("ANIMATION_ASSET").Set<std::string>();
  if (cc->InputSidePackets().HasTag("CAMERA_PARAMETERS_PROTO_STRING")) {
    cc->InputSidePackets()
        .Tag("CAMERA_PARAMETERS_PROTO_STRING")
        .Set<std::string>();
  }

  if (cc->InputSidePackets().HasTag("MASK_TEXTURE")) {
    cc->InputSidePackets().Tag("MASK_TEXTURE").Set<AssetTextureFormat>();
  }
  if (cc->InputSidePackets().HasTag("MASK_ASSET")) {
    cc->InputSidePackets().Tag("MASK_ASSET").Set<std::string>();
  }

  return ::mediapipe::OkStatus();
}

// Helper function for initializing our perspective matrix.
void GlAnimationOverlayCalculator::InitializePerspectiveMatrix(
    float aspect_ratio, float fov_degrees, float z_near, float z_far) {
  // Standard perspective projection matrix calculations.
  const float f = 1.0f / std::tan(fov_degrees * M_PI / 360.0f);
  for (int i = 0; i < kNumMatrixEntries; i++) {
    perspective_matrix_[i] = 0;
  }
  const float denom = 1.0f / (z_near - z_far);
  perspective_matrix_[0] = f / aspect_ratio;
  perspective_matrix_[5] = f;
  perspective_matrix_[10] = (z_near + z_far) * denom;
  perspective_matrix_[11] = -1.0f;
  perspective_matrix_[14] = 2.0f * z_far * z_near * denom;
}

#if defined(__ANDROID__)
// Helper function for reading in a specified number of bytes from an Android
// asset.  Returns true if successfully reads in all bytes into buffer.
bool GlAnimationOverlayCalculator::ReadBytesFromAsset(AAsset *asset,
                                                      void *buffer,
                                                      int num_bytes_to_read) {
  // Most file systems use block sizes of 4KB or 8KB; ideally we'd choose a
  // small multiple of the block size for best input streaming performance, so
  // we go for a reasobably safe buffer size of 8KB = 8*1024 bytes.
  static const int kMaxChunkSize = 8192;

  int bytes_left = num_bytes_to_read;
  int bytes_read = 1;  // any value > 0 here just to start looping.

  // Treat as uint8_t array so we can deal in single byte arithmetic easily.
  uint8_t *currBufferIndex = reinterpret_cast<uint8_t *>(buffer);
  while (bytes_read > 0 && bytes_left > 0) {
    bytes_read = AAsset_read(asset, (void *)currBufferIndex,
                             std::min(bytes_left, kMaxChunkSize));
    bytes_left -= bytes_read;
    currBufferIndex += bytes_read;
  }
  // At least log any I/O errors encountered.
  if (bytes_read < 0) {
    LOG(ERROR) << "Error reading from AAsset: " << bytes_read;
    return false;
  }
  if (bytes_left > 0) {
    // Reached EOF before reading in specified number of bytes.
    LOG(WARNING) << "Reached EOF before reading in specified number of bytes.";
    return false;
  }
  return true;
}

// The below asset streaming code is Android-only, making use of the platform
// JNI helper classes AAssetManager and AAsset.
bool GlAnimationOverlayCalculator::LoadAnimationAndroid(
    const std::string &filename, std::vector<TriangleMesh> *meshes) {
  mediapipe::AssetManager *mediapipe_asset_manager =
      Singleton<mediapipe::AssetManager>::get();
  AAssetManager *asset_manager = mediapipe_asset_manager->GetAssetManager();
  if (!asset_manager) {
    LOG(ERROR) << "Failed to access Android asset manager.";
    return false;
  }

  // New read-bytes stuff here!  First we open file for streaming.
  AAsset *asset = AAssetManager_open(asset_manager, filename.c_str(),
                                     AASSET_MODE_STREAMING);
  if (!asset) {
    LOG(ERROR) << "Failed to open animation asset: " << filename;
    return false;
  }

  // And now, while we are able to stream in more frames, we do so.
  frame_count_ = 0;
  int32 lengths[3];
  while (ReadBytesFromAsset(asset, (void *)lengths, sizeof(lengths[0]) * 3)) {
    // About to start reading the next animation frame.  Stream it in here.
    // Each frame stores first the object counts of its three arrays
    // (vertices, texture coordinates, triangle indices; respectively), and
    // then stores each of those arrays as a byte dump, in order.
    meshes->emplace_back();
    TriangleMesh &triangle_mesh = meshes->back();
    // Try to read in vertices (4-byte floats)
    triangle_mesh.vertices.reset(new float[lengths[0]]);
    if (!ReadBytesFromAsset(asset, (void *)triangle_mesh.vertices.get(),
                            sizeof(float) * lengths[0])) {
      LOG(ERROR) << "Failed to read vertices for frame " << frame_count_;
      return false;
    }
    // Try to read in texture coordinates (4-byte floats)
    triangle_mesh.texture_coords.reset(new float[lengths[1]]);
    if (!ReadBytesFromAsset(asset, (void *)triangle_mesh.texture_coords.get(),
                            sizeof(float) * lengths[1])) {
      LOG(ERROR) << "Failed to read tex-coords for frame " << frame_count_;
      return false;
    }
    // Try to read in indices (2-byte shorts)
    triangle_mesh.index_count = lengths[2];
    triangle_mesh.triangle_indices.reset(new int16[lengths[2]]);
    if (!ReadBytesFromAsset(asset, (void *)triangle_mesh.triangle_indices.get(),
                            sizeof(int16) * lengths[2])) {
      LOG(ERROR) << "Failed to read indices for frame " << frame_count_;
      return false;
    }
    frame_count_++;
  }
  AAsset_close(asset);

  LOG(INFO) << "Finished parsing " << frame_count_ << " animation frames.";
  if (meshes->empty()) {
    LOG(ERROR) << "No animation frames were parsed!  Erroring out calculator.";
    return false;
  }
  return true;
}

#else  // defined(__ANDROID__)

bool GlAnimationOverlayCalculator::LoadAnimation(const std::string &filename) {
  std::ifstream infile(filename.c_str(), std::ifstream::binary);
  if (!infile) {
    LOG(ERROR) << "Error opening asset with filename: " << filename;
    return false;
  }

  frame_count_ = 0;
  int32 lengths[3];
  while (true) {
    // See if we have more initial size counts to read in.
    infile.read((char *)(lengths), sizeof(lengths[0]) * 3);
    if (!infile) {
      // No more frames to read.  Close out.
      infile.close();
      break;
    }

    triangle_meshes_.emplace_back();
    TriangleMesh &triangle_mesh = triangle_meshes_.back();

    // Try to read in vertices (4-byte floats).
    triangle_mesh.vertices.reset(new float[lengths[0]]);
    infile.read((char *)(triangle_mesh.vertices.get()),
                sizeof(float) * lengths[0]);
    if (!infile) {
      LOG(ERROR) << "Failed to read vertices for frame " << frame_count_;
      return false;
    }

    // Try to read in texture coordinates (4-byte floats)
    triangle_mesh.texture_coords.reset(new float[lengths[1]]);
    infile.read((char *)(triangle_mesh.texture_coords.get()),
                sizeof(float) * lengths[1]);
    if (!infile) {
      LOG(ERROR) << "Failed to read texture coordinates for frame "
                 << frame_count_;
      return false;
    }

    // Try to read in the triangle indices (2-byte shorts)
    triangle_mesh.index_count = lengths[2];
    triangle_mesh.triangle_indices.reset(new int16[lengths[2]]);
    infile.read((char *)(triangle_mesh.triangle_indices.get()),
                sizeof(int16) * lengths[2]);
    if (!infile) {
      LOG(ERROR) << "Failed to read triangle indices for frame "
                 << frame_count_;
      return false;
    }
    frame_count_++;
  }

  LOG(INFO) << "Finished parsing " << frame_count_ << " animation frames.";
  if (triangle_meshes_.empty()) {
    LOG(ERROR) << "No animation frames were parsed!  Erroring out calculator.";
    return false;
  }
  return true;
}

#endif

void GlAnimationOverlayCalculator::ComputeAspectRatioAndFovFromCameraParameters(
    const CameraParametersProto &camera_parameters, float *aspect_ratio,
    float *vertical_fov_degrees) {
  CHECK(aspect_ratio != nullptr);
  CHECK(vertical_fov_degrees != nullptr);
  *aspect_ratio =
      camera_parameters.portrait_width() / camera_parameters.portrait_height();
  *vertical_fov_degrees =
      std::atan(camera_parameters.portrait_height() * 0.5f) * 2 * 180 / M_PI;
}

::mediapipe::Status GlAnimationOverlayCalculator::Open(CalculatorContext *cc) {
  cc->SetOffset(TimestampDiff(0));
  MP_RETURN_IF_ERROR(helper_.Open(cc));

  const auto &options = cc->Options<GlAnimationOverlayCalculatorOptions>();

  animation_speed_fps_ = options.animation_speed_fps();

  // Construct projection matrix using input side packets or option
  float aspect_ratio;
  float vertical_fov_degrees;
  if (cc->InputSidePackets().HasTag("CAMERA_PARAMETERS_PROTO_STRING")) {
    const std::string &camera_parameters_proto_string =
        cc->InputSidePackets()
            .Tag("CAMERA_PARAMETERS_PROTO_STRING")
            .Get<std::string>();
    CameraParametersProto camera_parameters_proto;
    camera_parameters_proto.ParseFromString(camera_parameters_proto_string);
    ComputeAspectRatioAndFovFromCameraParameters(
        camera_parameters_proto, &aspect_ratio, &vertical_fov_degrees);
  } else {
    aspect_ratio = options.aspect_ratio();
    vertical_fov_degrees = options.vertical_fov_degrees();
  }

  // when constructing projection matrix.
  InitializePerspectiveMatrix(aspect_ratio, vertical_fov_degrees,
                              options.z_clipping_plane_near(),
                              options.z_clipping_plane_far());

  // See what streams we have.
  has_video_stream_ = cc->Inputs().HasTag("VIDEO");
  has_model_matrix_stream_ = cc->Inputs().HasTag("MODEL_MATRICES");
  has_mask_model_matrix_stream_ = cc->Inputs().HasTag("MASK_MODEL_MATRICES");

  // Try to load in the animation asset in a platform-specific manner.
  const std::string &asset_name =
      cc->InputSidePackets().Tag("ANIMATION_ASSET").Get<std::string>();
  bool loaded_animation = false;
#if defined(__ANDROID__)
  if (cc->InputSidePackets().HasTag("MASK_ASSET")) {
    has_occlusion_mask_ = true;
    const std::string &mask_asset_name =
        cc->InputSidePackets().Tag("MASK_ASSET").Get<std::string>();
    loaded_animation = LoadAnimationAndroid(mask_asset_name, &mask_meshes_);
    if (!loaded_animation) {
      LOG(ERROR) << "Failed to load mask asset.";
      return ::mediapipe::UnknownError("Failed to load mask asset.");
    }
  }
  loaded_animation = LoadAnimationAndroid(asset_name, &triangle_meshes_);
#else
  loaded_animation = LoadAnimation(asset_name);
#endif
  if (!loaded_animation) {
    LOG(ERROR) << "Failed to load animation asset.";
    return ::mediapipe::UnknownError("Failed to load animation asset.");
  }

  return helper_.RunInGlContext([this, &cc]() -> ::mediapipe::Status {
    if (cc->InputSidePackets().HasTag("MASK_TEXTURE")) {
      const auto &mask_texture =
          cc->InputSidePackets().Tag("MASK_TEXTURE").Get<AssetTextureFormat>();
      mask_texture_ = helper_.CreateSourceTexture(mask_texture);
    }

    // Load in our asset's texture data
    const auto &input_texture =
        cc->InputSidePackets().Tag("TEXTURE").Get<AssetTextureFormat>();
    texture_ = helper_.CreateSourceTexture(input_texture);
    VLOG(2) << "Input texture size: " << texture_.width() << ", "
            << texture_.height() << std::endl;

    return ::mediapipe::OkStatus();
  });
}

int GlAnimationOverlayCalculator::GetAnimationFrameIndex(Timestamp timestamp) {
  double seconds_delta = timestamp.Seconds() - animation_start_time_.Seconds();
  int64_t frame_index =
      static_cast<int64_t>(seconds_delta * animation_speed_fps_);
  frame_index %= frame_count_;
  return static_cast<int>(frame_index);
}

void GlAnimationOverlayCalculator::LoadModelMatrices(
    const TimedModelMatrixProtoList &model_matrices,
    std::vector<ModelMatrix> *current_model_matrices) {
  current_model_matrices->clear();
  for (int i = 0; i < model_matrices.model_matrix_size(); ++i) {
    const auto &model_matrix = model_matrices.model_matrix(i);
    CHECK(model_matrix.matrix_entries_size() == kNumMatrixEntries)
        << "Invalid Model Matrix";
    current_model_matrices->emplace_back();
    ModelMatrix &new_matrix = current_model_matrices->back();
    new_matrix.reset(new float[kNumMatrixEntries]);
    for (int j = 0; j < kNumMatrixEntries; j++) {
      // Model matrices streamed in using ROW-MAJOR format, but we want
      // COLUMN-MAJOR for rendering, so we transpose here.
      int col = j % 4;
      int row = j / 4;
      new_matrix[row + col * 4] = model_matrix.matrix_entries(j);
    }
  }
}

::mediapipe::Status GlAnimationOverlayCalculator::Process(
    CalculatorContext *cc) {
  return helper_.RunInGlContext([this, &cc]() -> ::mediapipe::Status {
    if (!initialized_) {
      MP_RETURN_IF_ERROR(GlSetup());
      initialized_ = true;
      animation_start_time_ = cc->InputTimestamp();
    }

    // Process model matrices, if any are being streamed in, and update our
    // list.
    if (has_model_matrix_stream_ &&
        !cc->Inputs().Tag("MODEL_MATRICES").IsEmpty()) {
      const TimedModelMatrixProtoList &model_matrices =
          cc->Inputs().Tag("MODEL_MATRICES").Get<TimedModelMatrixProtoList>();
      LoadModelMatrices(model_matrices, &current_model_matrices_);
    }
    if (has_mask_model_matrix_stream_ &&
        !cc->Inputs().Tag("MASK_MODEL_MATRICES").IsEmpty()) {
      const TimedModelMatrixProtoList &model_matrices =
          cc->Inputs()
              .Tag("MASK_MODEL_MATRICES")
              .Get<TimedModelMatrixProtoList>();
      LoadModelMatrices(model_matrices, &current_mask_model_matrices_);
    }

    // Arbitrary default width and height for output destination texture, in the
    // event that we don't have a valid and unique input buffer to overlay.
    int width = 640;
    int height = 480;

    GlTexture dst;
    std::unique_ptr<GpuBuffer> input_frame(nullptr);
    if (has_video_stream_ && !(cc->Inputs().Tag("VIDEO").IsEmpty())) {
      auto result = cc->Inputs().Tag("VIDEO").Value().Consume<GpuBuffer>();
      if (result.ok()) {
        input_frame = std::move(result).ValueOrDie();
#if !MEDIAPIPE_GPU_BUFFER_USE_CV_PIXEL_BUFFER
        input_frame->GetGlTextureBufferSharedPtr()->Reuse();
#endif
        width = input_frame->width();
        height = input_frame->height();
        dst = helper_.CreateSourceTexture(*input_frame);
      } else {
        LOG(ERROR) << "Unable to consume input video frame for overlay!";
        LOG(ERROR) << "Status returned was: " << result.status();
        dst = helper_.CreateDestinationTexture(width, height);
      }
    } else if (!has_video_stream_) {
      dst = helper_.CreateDestinationTexture(width, height);
    } else {
      // We have an input video stream, but not for this frame. Don't render!
      return ::mediapipe::OkStatus();
    }
    helper_.BindFramebuffer(dst);

    if (!depth_buffer_created_) {
      // Create our private depth buffer.
      GLCHECK(glGenRenderbuffers(1, &renderbuffer_));
      GLCHECK(glBindRenderbuffer(GL_RENDERBUFFER, renderbuffer_));
      GLCHECK(glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT16,
                                    width, height));
      GLCHECK(glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                                        GL_RENDERBUFFER, renderbuffer_));
      GLCHECK(glBindRenderbuffer(GL_RENDERBUFFER, 0));
      depth_buffer_created_ = true;
    }

    // Re-bind our depth renderbuffer to our FBO depth attachment here.
    GLCHECK(glBindRenderbuffer(GL_RENDERBUFFER, renderbuffer_));
    GLCHECK(glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                                      GL_RENDERBUFFER, renderbuffer_));
    GLenum status = GLCHECK(glCheckFramebufferStatus(GL_FRAMEBUFFER));
    if (status != GL_FRAMEBUFFER_COMPLETE) {
      LOG(ERROR) << "Incomplete framebuffer with status: " << status;
    }
    GLCHECK(glClear(GL_DEPTH_BUFFER_BIT));

    if (has_occlusion_mask_) {
      glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
      const TriangleMesh &mask_frame = mask_meshes_.front();
      MP_RETURN_IF_ERROR(GlBind(mask_frame, mask_texture_));
      // Draw objects using our latest model matrix stream packet.
      for (const ModelMatrix &model_matrix : current_mask_model_matrices_) {
        MP_RETURN_IF_ERROR(GlRender(mask_frame, model_matrix.get()));
      }
    }

    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    int frame_index = GetAnimationFrameIndex(cc->InputTimestamp());
    const TriangleMesh &current_frame = triangle_meshes_[frame_index];
    MP_RETURN_IF_ERROR(GlBind(current_frame, texture_));
    if (has_model_matrix_stream_) {
      // Draw objects using our latest model matrix stream packet.
      for (const ModelMatrix &model_matrix : current_model_matrices_) {
        MP_RETURN_IF_ERROR(GlRender(current_frame, model_matrix.get()));
      }
    } else {
      // Just draw one object to a static model matrix.
      MP_RETURN_IF_ERROR(GlRender(current_frame, kModelMatrix));
    }

    // Disable vertex attributes
    GLCHECK(glEnableVertexAttribArray(ATTRIB_VERTEX));
    GLCHECK(glEnableVertexAttribArray(ATTRIB_TEXTURE_POSITION));

    // Disable depth test
    GLCHECK(glDisable(GL_DEPTH_TEST));

    // Unbind texture
    GLCHECK(glActiveTexture(GL_TEXTURE1));
    GLCHECK(glBindTexture(texture_.target(), 0));

    // Unbind depth buffer
    GLCHECK(glBindRenderbuffer(GL_RENDERBUFFER, 0));

    GLCHECK(glFlush());

    auto output = dst.GetFrame<GpuBuffer>();
    dst.Release();
    TagOrIndex(&(cc->Outputs()), "OUTPUT", 0)
        .Add(output.release(), cc->InputTimestamp());
    GLCHECK(glFrontFace(GL_CCW));
    return ::mediapipe::OkStatus();
  });
}

::mediapipe::Status GlAnimationOverlayCalculator::GlSetup() {
  // Load vertex and fragment shaders
  const GLint attr_location[NUM_ATTRIBUTES] = {
      ATTRIB_VERTEX,
      ATTRIB_TEXTURE_POSITION,
  };
  const GLchar *attr_name[NUM_ATTRIBUTES] = {
      "position",
      "texture_coordinate",
  };

  const GLchar *vert_src = R"(
    // Perspective projection matrix for rendering / clipping
    uniform mat4 perspectiveMatrix;

    // Matrix defining the currently rendered object model
    uniform mat4 modelMatrix;

    // vertex position in threespace
    attribute vec4 position;

    // texture coordinate for each vertex in normalized texture space (0..1)
    attribute mediump vec4 texture_coordinate;

    // texture coordinate for fragment shader (will be interpolated)
    varying mediump vec2 sample_coordinate;

    void main() {
      sample_coordinate = texture_coordinate.xy;
      mat4 mvpMatrix = perspectiveMatrix * modelMatrix;
      gl_Position = mvpMatrix * position;
    }
  )";

  const GLchar *frag_src = R"(
    precision mediump float;

    varying vec2 sample_coordinate;  // texture coordinate (0..1)
    uniform sampler2D texture;  // texture to shade with

    void main() {
      gl_FragColor = texture2D(texture, sample_coordinate);
    }
  )";

  // Shader program
  GLCHECK(GlhCreateProgram(vert_src, frag_src, NUM_ATTRIBUTES,
                           (const GLchar **)&attr_name[0], attr_location,
                           &program_));
  RET_CHECK(program_) << "Problem initializing the program.";
  texture_uniform_ = GLCHECK(glGetUniformLocation(program_, "texture"));
  perspective_matrix_uniform_ =
      GLCHECK(glGetUniformLocation(program_, "perspectiveMatrix"));
  model_matrix_uniform_ =
      GLCHECK(glGetUniformLocation(program_, "modelMatrix"));
  return ::mediapipe::OkStatus();
}

::mediapipe::Status GlAnimationOverlayCalculator::GlBind(
    const TriangleMesh &triangle_mesh, const GlTexture &texture) {
  GLCHECK(glUseProgram(program_));

  // Disable backface culling to allow occlusion effects.
  // Some options for solid arbitrary 3D geometry rendering
  GLCHECK(glEnable(GL_BLEND));
  GLCHECK(glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA));
  GLCHECK(glEnable(GL_DEPTH_TEST));
  GLCHECK(glFrontFace(GL_CW));
  GLCHECK(glDepthMask(GL_TRUE));
  GLCHECK(glDepthFunc(GL_LESS));

  // Clear our depth buffer before starting draw calls
  GLCHECK(glVertexAttribPointer(ATTRIB_VERTEX, 3, GL_FLOAT, 0, 0,
                                triangle_mesh.vertices.get()));
  GLCHECK(glEnableVertexAttribArray(ATTRIB_VERTEX));
  GLCHECK(glVertexAttribPointer(ATTRIB_TEXTURE_POSITION, 2, GL_FLOAT, 0, 0,
                                triangle_mesh.texture_coords.get()));
  GLCHECK(glEnableVertexAttribArray(ATTRIB_TEXTURE_POSITION));
  GLCHECK(glActiveTexture(GL_TEXTURE1));
  GLCHECK(glBindTexture(texture.target(), texture.name()));

  // We previously bound it to GL_TEXTURE1
  GLCHECK(glUniform1i(texture_uniform_, 1));

  GLCHECK(glUniformMatrix4fv(perspective_matrix_uniform_, 1, GL_FALSE,
                             perspective_matrix_));
  return ::mediapipe::OkStatus();
}

::mediapipe::Status GlAnimationOverlayCalculator::GlRender(
    const TriangleMesh &triangle_mesh, const float *model_matrix) {
  GLCHECK(glUniformMatrix4fv(model_matrix_uniform_, 1, GL_FALSE, model_matrix));
  GLCHECK(glDrawElements(GL_TRIANGLES, triangle_mesh.index_count,
                         GL_UNSIGNED_SHORT,
                         triangle_mesh.triangle_indices.get()));
  return ::mediapipe::OkStatus();
}

GlAnimationOverlayCalculator::~GlAnimationOverlayCalculator() {
  helper_.RunInGlContext([this] {
    if (program_) {
      GLCHECK(glDeleteProgram(program_));
      program_ = 0;
    }
    if (depth_buffer_created_) {
      GLCHECK(glDeleteRenderbuffers(1, &renderbuffer_));
      renderbuffer_ = 0;
    }
    if (texture_.width() > 0) {
      texture_.Release();
    }
    if (mask_texture_.width() > 0) {
      mask_texture_.Release();
    }
  });
}

}  // namespace mediapipe
