package scala

import scala.reflect.io.File
import java.util.Base64
import org.junit.Test

// This test is self-modifying when run as follows:
//
//    junit/testOnly scala.SerializationStabilityTest -- -Doverwrite.source=test/junit/scala/SerializationStabilityTest.scala
//
// Use this to re-establish a baseline for serialization compatibility.

// based on run/t8549.scala partest
object SerializationStability {

  def parseBase64Binary(s: String): Array[Byte] = Base64.getDecoder.decode(s)

  def printBase64Binary(data: Array[Byte]): String = Base64.getEncoder.encode(data).map(_.toChar).mkString

  val overwrite: Option[File] = sys.props.get("overwrite.source").map(s => new File(new java.io.File(s)))

  def serialize(o: AnyRef): String = {
    val bos = new java.io.ByteArrayOutputStream()
    val out = new java.io.ObjectOutputStream(bos)
    out.writeObject(o)
    out.flush()
    printBase64Binary(bos.toByteArray())
  }

  def amend(file: File)(f: String => String): Unit = {
    file.writeAll(f(file.slurp))
  }
  def quote(s: String) = List("\"", s, "\"").mkString

  def patch(file: File, line: Int, prevResult: String, result: String): Unit = {
    amend(file) {
      content =>
        content.lines.toList.zipWithIndex.map {
          case (content, i) if i == line - 1 =>
            val newContent = content.replaceAllLiterally(quote(prevResult), quote(result))
            if (newContent != content)
              println(s"- $content\n+ $newContent\n")
            newContent
          case (content, _) => content
        }.mkString("\n")
    }
  }

  def updateComment(file: File): Unit = {
    val timestamp = {
      import java.text.SimpleDateFormat
      val sdf = new SimpleDateFormat("yyyyMMdd-HH:mm:ss")
      sdf.format(new java.util.Date)
    }
    val newComment = s"  // Generated on $timestamp with Scala ${scala.util.Properties.versionString})"
    amend(file) {
      content =>
        content.lines.toList.map {
          f => f.replaceAll("""^ +// Generated on.*""", newComment)
        }.mkString("\n")
    }
  }

  def deserialize(string: String): AnyRef = {
    val bis = new java.io.ByteArrayInputStream(parseBase64Binary(string))
    val in = new java.io.ObjectInputStream(bis)
    in.readObject()
  }

  def checkRoundTrip[T <: AnyRef](instance: T)(f: T => AnyRef): Unit = {
    val result = serialize(instance)
    val reconstituted = deserialize(result).asInstanceOf[T]
    assert(f(instance) == f(reconstituted), (f(instance), f(reconstituted)))
  }

  def check[T <: AnyRef](stack: Array[StackTraceElement])(instance: => T)(prevResult: String, f: T => AnyRef = (x: T) => x): Unit = {
    val result = serialize(instance)
    overwrite match {
      case Some(f) =>
        val lineNumberOfLiteralString = stack.apply(2).getLineNumber
        patch(f, lineNumberOfLiteralString, prevResult, result)
      case None =>
        checkRoundTrip(instance)(f)
        assert(f(deserialize(prevResult).asInstanceOf[T]) == f(instance), s"$instance != f(deserialize(prevResult))")
        assert(prevResult == result, s"instance = $instance : ${instance.getClass}\n serialization unstable: ${prevResult}\n   found: ${result}")
    }
  }

  // Generated on 20180814-19:56:47 with Scala version 2.13.0-20180814-124737-ca5a07d)
  def main(args: Array[String]): Unit = {
    overwrite.foreach(updateComment)
    def g = Thread.currentThread.getStackTrace

    check(g)(Some(1))("rO0ABXNyAApzY2FsYS5Tb21lESLyaV6hi3QCAAFMAAV2YWx1ZXQAEkxqYXZhL2xhbmcvT2JqZWN0O3hyAAxzY2FsYS5PcHRpb27+aTf92w5mdAIAAHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAE=")
    check(g)(None)("rO0ABXNyAAtzY2FsYS5Ob25lJEZQJPZTypSsAgAAeHIADHNjYWxhLk9wdGlvbv5pN/3bDmZ0AgAAeHA=")

    check(g)(List(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAIHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3QkAAAAAAAAAAMDAAFMABFwYXJ0aWFsTm90QXBwbGllZHQAEUxzY2FsYS9GdW5jdGlvbjE7eHB4dwT/////c3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAkAAAACc3EAfgAJAAAAA3NyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(Nil)(           "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAIHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3QkAAAAAAAAAAMDAAFMABFwYXJ0aWFsTm90QXBwbGllZHQAEUxzY2FsYS9GdW5jdGlvbjE7eHB4dwQAAAAAeA==")

    // TODO scala/bug#8576 unstable under -Xcheckinit
    // check(g)(Vector(1))(     "rO0ABXNyACFzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5WZWN0b3Lkd3dcHq6PXAIAC0kABWRlcHRoWgAFZGlydHlJAAhlbmRJbmRleEkABWZvY3VzSQAKc3RhcnRJbmRleFsACGRpc3BsYXkwdAATW0xqYXZhL2xhbmcvT2JqZWN0O1sACGRpc3BsYXkxcQB+AAFbAAhkaXNwbGF5MnEAfgABWwAIZGlzcGxheTNxAH4AAVsACGRpc3BsYXk0cQB+AAFbAAhkaXNwbGF5NXEAfgABeHAAAAABAAAAAAEAAAAAAAAAAHVyABNbTGphdmEubGFuZy5PYmplY3Q7kM5YnxBzKWwCAAB4cAAAACBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXBwcHBwcHBwcHBwcHBwcHBwcHBwcHBwcHBwcHBwcHBwcHBwcA==")
    // check(g)(Vector())(      "rO0ABXNyACFzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5WZWN0b3Lkd3dcHq6PXAIAC0kABWRlcHRoWgAFZGlydHlJAAhlbmRJbmRleEkABWZvY3VzSQAKc3RhcnRJbmRleFsACGRpc3BsYXkwdAATW0xqYXZhL2xhbmcvT2JqZWN0O1sACGRpc3BsYXkxcQB+AAFbAAhkaXNwbGF5MnEAfgABWwAIZGlzcGxheTNxAH4AAVsACGRpc3BsYXk0cQB+AAFbAAhkaXNwbGF5NXEAfgABeHAAAAAAAAAAAAAAAAAAAAAAAHBwcHBwcA==")

    import collection.{ mutable, immutable }

    class C
    check(g)(reflect.classTag[C])("rO0ABXNyACZzY2FsYS5yZWZsZWN0LkNsYXNzVGFnJEdlbmVyaWNDbGFzc1RhZwAAAAAAAAABAgABTAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACBzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JEMkMQAAAAAAAAAAAAAAeHA=")
    check(g)(reflect.classTag[Int])("rO0ABXNyAClzY2FsYS5yZWZsZWN0Lk1hbmlmZXN0RmFjdG9yeSRJbnRNYW5pZmVzdAAAAAAAAAABAgAAeHIAHHNjYWxhLnJlZmxlY3QuQW55VmFsTWFuaWZlc3QAAAAAAAAAAQIAAUwACHRvU3RyaW5ndAASTGphdmEvbGFuZy9TdHJpbmc7eHB0AANJbnQ=")
    check(g)(reflect.classTag[String])("rO0ABXNyACZzY2FsYS5yZWZsZWN0LkNsYXNzVGFnJEdlbmVyaWNDbGFzc1RhZwAAAAAAAAABAgABTAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyABBqYXZhLmxhbmcuU3RyaW5noPCkOHo7s0ICAAB4cA==")
    check(g)(reflect.classTag[Object])("rO0ABXNyACxzY2FsYS5yZWZsZWN0Lk1hbmlmZXN0RmFjdG9yeSRPYmplY3RNYW5pZmVzdAAAAAAAAAABAgAAeHIALXNjYWxhLnJlZmxlY3QuTWFuaWZlc3RGYWN0b3J5JFBoYW50b21NYW5pZmVzdAAAAAAAAAABAgABTAAIdG9TdHJpbmd0ABJMamF2YS9sYW5nL1N0cmluZzt4cgAvc2NhbGEucmVmbGVjdC5NYW5pZmVzdEZhY3RvcnkkQ2xhc3NUeXBlTWFuaWZlc3QAAAAAAAAAAQIAA0wABnByZWZpeHQADkxzY2FsYS9PcHRpb247TAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMAA10eXBlQXJndW1lbnRzdAAhTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0xpc3Q7eHBzcgALc2NhbGEuTm9uZSRGUCT2U8qUrAIAAHhyAAxzY2FsYS5PcHRpb27+aTf92w5mdAIAAHhwdnIAEGphdmEubGFuZy5PYmplY3QAAAAAAAAAAAAAAHhwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyACpzY2FsYS5jb2xsZWN0aW9uLkl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAUwAB2ZhY3Rvcnl0ACJMc2NhbGEvY29sbGVjdGlvbi9JdGVyYWJsZUZhY3Rvcnk7eHBzcgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwMAAUwAEXBhcnRpYWxOb3RBcHBsaWVkdAARTHNjYWxhL0Z1bmN0aW9uMTt4cHh3BAAAAAB4dAAGT2JqZWN0")

    // TODO scala/bug#8576 unstable under -Xcheckinit
    // check(g)(Enum)(         "rO0ABXNyAApUZXN0JEVudW0ketCIyQ8C23MCAAJMAAJWMXQAGUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZTtMAAJWMnQAF0xzY2FsYS9FbnVtZXJhdGlvbiRWYWw7eHIAEXNjYWxhLkVudW1lcmF0aW9udaDN3ZgOWY4CAAhJAAZuZXh0SWRJABtzY2FsYSRFbnVtZXJhdGlvbiQkYm90dG9tSWRJABhzY2FsYSRFbnVtZXJhdGlvbiQkdG9wSWRMABRWYWx1ZU9yZGVyaW5nJG1vZHVsZXQAIkxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZU9yZGVyaW5nJDtMAA9WYWx1ZVNldCRtb2R1bGV0AB1Mc2NhbGEvRW51bWVyYXRpb24kVmFsdWVTZXQkO0wACG5leHROYW1ldAAbTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmF0b3I7TAAXc2NhbGEkRW51bWVyYXRpb24kJG5tYXB0AB5Mc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL01hcDtMABdzY2FsYSRFbnVtZXJhdGlvbiQkdm1hcHEAfgAHeHAAAAArAAAAAAAAACtwcHBzcgAgc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAAAAAAAAAAAQMAAHhwdw0AAALuAAAAAAAAAAQAeHNxAH4ACXcNAAAC7gAAAAEAAAAEAHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAqc3IAFXNjYWxhLkVudW1lcmF0aW9uJFZhbM9pZ6/J/O1PAgACSQAYc2NhbGEkRW51bWVyYXRpb24kVmFsJCRpTAAEbmFtZXQAEkxqYXZhL2xhbmcvU3RyaW5nO3hyABdzY2FsYS5FbnVtZXJhdGlvbiRWYWx1ZWJpfC/tIR1RAgACTAAGJG91dGVydAATTHNjYWxhL0VudW1lcmF0aW9uO0wAHHNjYWxhJEVudW1lcmF0aW9uJCRvdXRlckVudW1xAH4AEnhwcQB+AAhxAH4ACAAAACpweHNyABFUZXN0JEVudW0kJGFub24kMVlIjlmE1sXaAgAAeHEAfgARcQB+AAhxAH4ACHEAfgAT")
    // check(g)(Enum.V1)(      "rO0ABXNyABFUZXN0JEVudW0kJGFub24kMVlIjlmE1sXaAgAAeHIAF3NjYWxhLkVudW1lcmF0aW9uJFZhbHVlYml8L+0hHVECAAJMAAYkb3V0ZXJ0ABNMc2NhbGEvRW51bWVyYXRpb247TAAcc2NhbGEkRW51bWVyYXRpb24kJG91dGVyRW51bXEAfgACeHBzcgAKVGVzdCRFbnVtJHrQiMkPAttzAgACTAACVjF0ABlMc2NhbGEvRW51bWVyYXRpb24kVmFsdWU7TAACVjJ0ABdMc2NhbGEvRW51bWVyYXRpb24kVmFsO3hyABFzY2FsYS5FbnVtZXJhdGlvbnWgzd2YDlmOAgAISQAGbmV4dElkSQAbc2NhbGEkRW51bWVyYXRpb24kJGJvdHRvbUlkSQAYc2NhbGEkRW51bWVyYXRpb24kJHRvcElkTAAUVmFsdWVPcmRlcmluZyRtb2R1bGV0ACJMc2NhbGEvRW51bWVyYXRpb24kVmFsdWVPcmRlcmluZyQ7TAAPVmFsdWVTZXQkbW9kdWxldAAdTHNjYWxhL0VudW1lcmF0aW9uJFZhbHVlU2V0JDtMAAhuZXh0TmFtZXQAG0xzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhdG9yO0wAF3NjYWxhJEVudW1lcmF0aW9uJCRubWFwdAAeTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9NYXA7TAAXc2NhbGEkRW51bWVyYXRpb24kJHZtYXBxAH4AC3hwAAAAKwAAAAAAAAArcHBwc3IAIHNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5IYXNoTWFwAAAAAAAAAAEDAAB4cHcNAAAC7gAAAAAAAAAEAHhzcQB+AA13DQAAAu4AAAABAAAABABzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAKnNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAGHNjYWxhJEVudW1lcmF0aW9uJFZhbCQkaUwABG5hbWV0ABJMamF2YS9sYW5nL1N0cmluZzt4cQB+AAFxAH4ADHEAfgAMAAAAKnB4cQB+AANxAH4AFXEAfgAM")
    // check(g)(Enum.V2)(      "rO0ABXNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAGHNjYWxhJEVudW1lcmF0aW9uJFZhbCQkaUwABG5hbWV0ABJMamF2YS9sYW5nL1N0cmluZzt4cgAXc2NhbGEuRW51bWVyYXRpb24kVmFsdWViaXwv7SEdUQIAAkwABiRvdXRlcnQAE0xzY2FsYS9FbnVtZXJhdGlvbjtMABxzY2FsYSRFbnVtZXJhdGlvbiQkb3V0ZXJFbnVtcQB+AAN4cHNyAApUZXN0JEVudW0ketCIyQ8C23MCAAJMAAJWMXQAGUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZTtMAAJWMnQAF0xzY2FsYS9FbnVtZXJhdGlvbiRWYWw7eHIAEXNjYWxhLkVudW1lcmF0aW9udaDN3ZgOWY4CAAhJAAZuZXh0SWRJABtzY2FsYSRFbnVtZXJhdGlvbiQkYm90dG9tSWRJABhzY2FsYSRFbnVtZXJhdGlvbiQkdG9wSWRMABRWYWx1ZU9yZGVyaW5nJG1vZHVsZXQAIkxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZU9yZGVyaW5nJDtMAA9WYWx1ZVNldCRtb2R1bGV0AB1Mc2NhbGEvRW51bWVyYXRpb24kVmFsdWVTZXQkO0wACG5leHROYW1ldAAbTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmF0b3I7TAAXc2NhbGEkRW51bWVyYXRpb24kJG5tYXB0AB5Mc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL01hcDtMABdzY2FsYSRFbnVtZXJhdGlvbiQkdm1hcHEAfgAMeHAAAAArAAAAAAAAACtwcHBzcgAgc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAAAAAAAAAAAQMAAHhwdw0AAALuAAAAAAAAAAQAeHNxAH4ADncNAAAC7gAAAAEAAAAEAHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAqcQB+AAR4c3IAEVRlc3QkRW51bSQkYW5vbiQxWUiOWYTWxdoCAAB4cQB+AAJxAH4ADXEAfgANcQB+AARxAH4ADQAAACpw")

    // IndexedSeqLike#Elements
    // TODO scala/bug#8576 throws scala.UnitializedFieldError under -Xcheckinit
    // check(g)(new immutable.Range(0, 1, 1).iterator)("rO0ABXNyAChzY2FsYS5jb2xsZWN0aW9uLkluZGV4ZWRTZXFMaWtlJEVsZW1lbnRzGF+1cBwmcx0CAANJAANlbmRJAAVpbmRleEwABiRvdXRlcnQAIUxzY2FsYS9jb2xsZWN0aW9uL0luZGV4ZWRTZXFMaWtlO3hwAAAAAQAAAABzcgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuUmFuZ2Vpu6NUqxUyDQIAB0kAA2VuZFoAB2lzRW1wdHlJAAtsYXN0RWxlbWVudEkAEG51bVJhbmdlRWxlbWVudHNJAAVzdGFydEkABHN0ZXBJAA90ZXJtaW5hbEVsZW1lbnR4cAAAAAEAAAAAAAAAAAEAAAAAAAAAAQAAAAE="
    //   , _.toList)

    // check(g)(new collection.concurrent.TrieMap[Any, Any]())( "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmNvbmN1cnJlbnQuVHJpZU1hcKckxpgOIYHPAwAETAALZXF1YWxpdHlvYmp0ABJMc2NhbGEvbWF0aC9FcXVpdjtMAApoYXNoaW5nb2JqdAAcTHNjYWxhL3V0aWwvaGFzaGluZy9IYXNoaW5nO0wABHJvb3R0ABJMamF2YS9sYW5nL09iamVjdDtMAAtyb290dXBkYXRlcnQAOUxqYXZhL3V0aWwvY29uY3VycmVudC9hdG9taWMvQXRvbWljUmVmZXJlbmNlRmllbGRVcGRhdGVyO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uY29uY3VycmVudC5UcmllTWFwJE1hbmdsZWRIYXNoaW5nhTBoJQ/mgb0CAAB4cHNyABhzY2FsYS5tYXRoLkVxdWl2JCRhbm9uJDLBbyx4dy/qGwIAAHhwc3IANHNjYWxhLmNvbGxlY3Rpb24uY29uY3VycmVudC5UcmllTWFwU2VyaWFsaXphdGlvbkVuZCSbjdgbbGCt2gIAAHhweA==")
    // not sure why this one needs stable serialization.

    // TODO scala/bug#8576 unstable under -Xcheckinit
    check(g)(collection.convert.Wrappers)( "rO0ABXNyACJzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuV3JhcHBlcnMkAAAAAAAAAAMCABFMABhEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADZMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJERpY3Rpb25hcnlXcmFwcGVyJDtMABZJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA0THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRJdGVyYWJsZVdyYXBwZXIkO0wAFkl0ZXJhdG9yV3JhcHBlciRtb2R1bGV0ADRMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEl0ZXJhdG9yV3JhcHBlciQ7TAAZSkNvbGxlY3Rpb25XcmFwcGVyJG1vZHVsZXQAN0xzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkNvbGxlY3Rpb25XcmFwcGVyJDtMABxKQ29uY3VycmVudE1hcFdyYXBwZXIkbW9kdWxldAA6THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKQ29uY3VycmVudE1hcFdyYXBwZXIkO0wAGUpEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADdMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpEaWN0aW9uYXJ5V3JhcHBlciQ7TAAaSkVudW1lcmF0aW9uV3JhcHBlciRtb2R1bGV0ADhMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpFbnVtZXJhdGlvbldyYXBwZXIkO0wAF0pJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA1THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKSXRlcmFibGVXcmFwcGVyJDtMABdKSXRlcmF0b3JXcmFwcGVyJG1vZHVsZXQANUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkl0ZXJhdG9yV3JhcHBlciQ7TAATSkxpc3RXcmFwcGVyJG1vZHVsZXQAMUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkxpc3RXcmFwcGVyJDtMABlKUHJvcGVydGllc1dyYXBwZXIkbW9kdWxldAA3THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKUHJvcGVydGllc1dyYXBwZXIkO0wAEkpTZXRXcmFwcGVyJG1vZHVsZXQAMExzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSlNldFdyYXBwZXIkO0wAG011dGFibGVCdWZmZXJXcmFwcGVyJG1vZHVsZXQAOUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZUJ1ZmZlcldyYXBwZXIkO0wAGE11dGFibGVNYXBXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZU1hcFdyYXBwZXIkO0wAGE11dGFibGVTZXFXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNlcVdyYXBwZXIkO0wAGE11dGFibGVTZXRXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNldFdyYXBwZXIkO0wAEVNlcVdyYXBwZXIkbW9kdWxldAAvTHNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRTZXFXcmFwcGVyJDt4cHBwcHBwcHBwcHBwcHBwcHBw")

    check(g)(new collection.convert.Wrappers.SetWrapper(immutable.Set()))("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")
    check(g)(new collection.convert.Wrappers.SetWrapper(immutable.Set(1, 2, 3)))("rO0ABXNyACxzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuV3JhcHBlcnMkU2V0V3JhcHBlcgAAAAAAAAADAgACTAAGJG91dGVydAAjTHNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycztMADhzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkV3JhcHBlcnMkU2V0V3JhcHBlciQkdW5kZXJseWluZ3QAFkxzY2FsYS9jb2xsZWN0aW9uL1NldDt4cHNyACJzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuV3JhcHBlcnMkAAAAAAAAAAMCABFMABhEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADZMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJERpY3Rpb25hcnlXcmFwcGVyJDtMABZJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA0THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRJdGVyYWJsZVdyYXBwZXIkO0wAFkl0ZXJhdG9yV3JhcHBlciRtb2R1bGV0ADRMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEl0ZXJhdG9yV3JhcHBlciQ7TAAZSkNvbGxlY3Rpb25XcmFwcGVyJG1vZHVsZXQAN0xzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkNvbGxlY3Rpb25XcmFwcGVyJDtMABxKQ29uY3VycmVudE1hcFdyYXBwZXIkbW9kdWxldAA6THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKQ29uY3VycmVudE1hcFdyYXBwZXIkO0wAGUpEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADdMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpEaWN0aW9uYXJ5V3JhcHBlciQ7TAAaSkVudW1lcmF0aW9uV3JhcHBlciRtb2R1bGV0ADhMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpFbnVtZXJhdGlvbldyYXBwZXIkO0wAF0pJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA1THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKSXRlcmFibGVXcmFwcGVyJDtMABdKSXRlcmF0b3JXcmFwcGVyJG1vZHVsZXQANUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkl0ZXJhdG9yV3JhcHBlciQ7TAATSkxpc3RXcmFwcGVyJG1vZHVsZXQAMUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkxpc3RXcmFwcGVyJDtMABlKUHJvcGVydGllc1dyYXBwZXIkbW9kdWxldAA3THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKUHJvcGVydGllc1dyYXBwZXIkO0wAEkpTZXRXcmFwcGVyJG1vZHVsZXQAMExzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSlNldFdyYXBwZXIkO0wAG011dGFibGVCdWZmZXJXcmFwcGVyJG1vZHVsZXQAOUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZUJ1ZmZlcldyYXBwZXIkO0wAGE11dGFibGVNYXBXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZU1hcFdyYXBwZXIkO0wAGE11dGFibGVTZXFXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNlcVdyYXBwZXIkO0wAGE11dGFibGVTZXRXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNldFdyYXBwZXIkO0wAEVNlcVdyYXBwZXIkbW9kdWxldAAvTHNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRTZXFXcmFwcGVyJDt4cHBwcHBwcHBwcHBwcHBwcHBwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyACpzY2FsYS5jb2xsZWN0aW9uLkl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAUwAB2ZhY3Rvcnl0ACJMc2NhbGEvY29sbGVjdGlvbi9JdGVyYWJsZUZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuU2V0JAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkU2V0JCR1c2VCYXNlbGluZXhweHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAfAAAAAnNxAH4AHwAAAAN4")
    check(g)(new collection.convert.Wrappers.SetWrapper(mutable.Set()))("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")
    check(g)(new collection.convert.Wrappers.SetWrapper(mutable.Set(1, 2, 3)))("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")
    check(g)(new collection.convert.Wrappers.MutableSetWrapper(mutable.Set()))("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")
    check(g)(new collection.convert.Wrappers.MutableSetWrapper(mutable.Set(1, 2, 3)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map()))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map[Int, Int](1 -> 2)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5, 5 -> 6)))("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")
    check(g)(new collection.convert.Wrappers.MapWrapper(mutable.Map()))("rO0ABXNyACxzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuV3JhcHBlcnMkTWFwV3JhcHBlcgAAAAAAAAADAgACTAAGJG91dGVydAAjTHNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycztMADhzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkV3JhcHBlcnMkTWFwV3JhcHBlciQkdW5kZXJseWluZ3QAFkxzY2FsYS9jb2xsZWN0aW9uL01hcDt4cHNyACJzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuV3JhcHBlcnMkAAAAAAAAAAMCABFMABhEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADZMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJERpY3Rpb25hcnlXcmFwcGVyJDtMABZJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA0THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRJdGVyYWJsZVdyYXBwZXIkO0wAFkl0ZXJhdG9yV3JhcHBlciRtb2R1bGV0ADRMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEl0ZXJhdG9yV3JhcHBlciQ7TAAZSkNvbGxlY3Rpb25XcmFwcGVyJG1vZHVsZXQAN0xzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkNvbGxlY3Rpb25XcmFwcGVyJDtMABxKQ29uY3VycmVudE1hcFdyYXBwZXIkbW9kdWxldAA6THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKQ29uY3VycmVudE1hcFdyYXBwZXIkO0wAGUpEaWN0aW9uYXJ5V3JhcHBlciRtb2R1bGV0ADdMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpEaWN0aW9uYXJ5V3JhcHBlciQ7TAAaSkVudW1lcmF0aW9uV3JhcHBlciRtb2R1bGV0ADhMc2NhbGEvY29sbGVjdGlvbi9jb252ZXJ0L1dyYXBwZXJzJEpFbnVtZXJhdGlvbldyYXBwZXIkO0wAF0pJdGVyYWJsZVdyYXBwZXIkbW9kdWxldAA1THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKSXRlcmFibGVXcmFwcGVyJDtMABdKSXRlcmF0b3JXcmFwcGVyJG1vZHVsZXQANUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkl0ZXJhdG9yV3JhcHBlciQ7TAATSkxpc3RXcmFwcGVyJG1vZHVsZXQAMUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSkxpc3RXcmFwcGVyJDtMABlKUHJvcGVydGllc1dyYXBwZXIkbW9kdWxldAA3THNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRKUHJvcGVydGllc1dyYXBwZXIkO0wAEkpTZXRXcmFwcGVyJG1vZHVsZXQAMExzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkSlNldFdyYXBwZXIkO0wAG011dGFibGVCdWZmZXJXcmFwcGVyJG1vZHVsZXQAOUxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZUJ1ZmZlcldyYXBwZXIkO0wAGE11dGFibGVNYXBXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZU1hcFdyYXBwZXIkO0wAGE11dGFibGVTZXFXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNlcVdyYXBwZXIkO0wAGE11dGFibGVTZXRXcmFwcGVyJG1vZHVsZXQANkxzY2FsYS9jb2xsZWN0aW9uL2NvbnZlcnQvV3JhcHBlcnMkTXV0YWJsZVNldFdyYXBwZXIkO0wAEVNlcVdyYXBwZXIkbW9kdWxldAAvTHNjYWxhL2NvbGxlY3Rpb24vY29udmVydC9XcmFwcGVycyRTZXFXcmFwcGVyJDt4cHBwcHBwcHBwcHBwcHBwcHBwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyACVzY2FsYS5jb2xsZWN0aW9uLk1hcEZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAdTHNjYWxhL2NvbGxlY3Rpb24vTWFwRmFjdG9yeTt4cHNyACFzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaE1hcCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(new collection.convert.Wrappers.MapWrapper(mutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("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")
    check(g)(new collection.convert.Wrappers.MutableMapWrapper(mutable.Map()))("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")
    check(g)(new collection.convert.Wrappers.MutableMapWrapper(mutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("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")

    check(g)(immutable.BitSet(1, 2, 3))( "rO0ABXNyADRzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5CaXRTZXQkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMCAAB4cgAqc2NhbGEuY29sbGVjdGlvbi5CaXRTZXQkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAB4cHcMAAAAAQAAAAAAAAAOeA==")
    check(g)(immutable.HashMap())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaE1hcCQAAAAAAAAAAwMAAUwACEVtcHR5TWFwdAAkTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0hhc2hNYXA7eHB4dwQAAAAAeA==")
    check(g)(immutable.HashMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaE1hcCQAAAAAAAAAAwMAAUwACEVtcHR5TWFwdAAkTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0hhc2hNYXA7eHB4dwQAAAABc3IADHNjYWxhLlR1cGxlMhDcBtcOREuEAgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAKeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ADAAAAAJ4")
    check(g)(immutable.HashMap(1 -> 2, 3 -> 4))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaE1hcCQAAAAAAAAAAwMAAUwACEVtcHR5TWFwdAAkTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0hhc2hNYXA7eHB4dwQAAAACc3IADHNjYWxhLlR1cGxlMhDcBtcOREuEAgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAKeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ADAAAAAJzcQB+AAlzcQB+AAwAAAADc3EAfgAMAAAABHg=")
    // TODO provoke HashMapCollision1

    check(g)(immutable.HashSet())(        "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hTZXQkAAAAAAAAAAMDAAFMAAhFbXB0eVNldHQAJExzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9IYXNoU2V0O3hweHcEAAAAAHg=")
    check(g)(immutable.HashSet(1))(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hTZXQkAAAAAAAAAAMDAAFMAAhFbXB0eVNldHQAJExzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9IYXNoU2V0O3hweHcEAAAAAXNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABeA==")
    check(g)(immutable.HashSet(1, 2))(    "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hTZXQkAAAAAAAAAAMDAAFMAAhFbXB0eVNldHQAJExzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9IYXNoU2V0O3hweHcEAAAAAnNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAJAAAAAng=")
    check(g)(immutable.HashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hTZXQkAAAAAAAAAAMDAAFMAAhFbXB0eVNldHQAJExzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9IYXNoU2V0O3hweHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAJAAAAAnNxAH4ACQAAAAN4")
    // TODO provoke HashSetCollision1

    check(g)(immutable.ListSet())(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3RTZXQkAAAAAAAAAAMDAAB4cHh3BAAAAAB4")
    check(g)(immutable.ListSet(1))(      "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3RTZXQkAAAAAAAAAAMDAAB4cHh3BP////9zcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(immutable.ListMap())(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdE1hcCQAAAAAAAAAAwMAAHhweHcEAAAAAHg=")
    check(g)(immutable.ListMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdE1hcCQAAAAAAAAAAwMAAHhweHcE/////3NyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3IAJnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5TZXJpYWxpemVFbmQkAAAAAAAAAAMCAAB4cHg=")
    //check(g)(immutable.Queue())(         "rO0ABXNyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5RdWV1ZZY146W3qSuhAgACTAACaW50ACFMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvTGlzdDtMAANvdXRxAH4AAXhwc3IAMnNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3QkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAEDAAB4cHNyACxzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0U2VyaWFsaXplRW5kJIpcY1v3UwttAgAAeHB4cQB+AAQ=")
    //check(g)(immutable.Queue(1, 2, 3))(  "rO0ABXNyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5RdWV1ZZY146W3qSuhAgACTAACaW50ACFMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvTGlzdDtMAANvdXRxAH4AAXhwc3IAMnNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3QkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAEDAAB4cHNyACxzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0U2VyaWFsaXplRW5kJIpcY1v3UwttAgAAeHB4c3EAfgADc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAgAAAACc3EAfgAIAAAAA3EAfgAGeA==")

    // TODO scala/bug#8576 throws scala.UnitializedFieldError under -Xcheckinit
    // check(g)(new immutable.Range(0, 1, 1))( "rO0ABXNyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5SYW5nZWm7o1SrFTINAgAHSQADZW5kWgAHaXNFbXB0eUkAC2xhc3RFbGVtZW50SQAQbnVtUmFuZ2VFbGVtZW50c0kABXN0YXJ0SQAEc3RlcEkAD3Rlcm1pbmFsRWxlbWVudHhwAAAAAQAAAAAAAAAAAQAAAAAAAAABAAAAAQ==")

    check(g)(immutable.Set())(              "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAH3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCQAAAAAAAAAAwMAAVoAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFNldCQkdXNlQmFzZWxpbmV4cHh3BAAAAAB4")
    check(g)(immutable.Set(1))(             "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAH3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCQAAAAAAAAAAwMAAVoAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFNldCQkdXNlQmFzZWxpbmV4cHh3BAAAAAFzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXg=")
    check(g)(immutable.Set(1, 2))(          "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAH3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCQAAAAAAAAAAwMAAVoAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFNldCQkdXNlQmFzZWxpbmV4cHh3BAAAAAJzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACAAAAAJ4")
    check(g)(immutable.Set(1, 2, 3))(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAH3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCQAAAAAAAAAAwMAAVoAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFNldCQkdXNlQmFzZWxpbmV4cHh3BAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACAAAAAJzcQB+AAgAAAADeA==")
    check(g)(immutable.Set(1, 2, 3, 4))(    "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAH3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCQAAAAAAAAAAwMAAVoAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFNldCQkdXNlQmFzZWxpbmV4cHh3BAAAAARzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACAAAAAJzcQB+AAgAAAADc3EAfgAIAAAABHg=")
    check(g)(immutable.Set(1, 2, 3, 4, 5))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hTZXQkAAAAAAAAAAMDAAFMAAhFbXB0eVNldHQAJExzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9IYXNoU2V0O3hweHcEAAAABXNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAFc3EAfgAJAAAAAXNxAH4ACQAAAAJzcQB+AAkAAAADc3EAfgAJAAAABHg=")

    check(g)(immutable.Map())(                                       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJCR1c2VCYXNlbGluZXhweHcEAAAAAHg=")
    check(g)(immutable.Map(1 -> 2))(                                 "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJCR1c2VCYXNlbGluZXhweHcEAAAAAXNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACeA==")
    check(g)(immutable.Map(1 -> 2, 2 -> 3))(                         "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJCR1c2VCYXNlbGluZXhweHcEAAAAAnNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgAIcQB+AA5zcQB+AAsAAAADeA==")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4))(                 "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJCR1c2VCYXNlbGluZXhweHcEAAAAA3NyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgAIcQB+AA5zcQB+AAsAAAADc3EAfgAIcQB+ABBzcQB+AAsAAAAEeA==")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5))(         "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAfc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJAAAAAAAAAADAwABWgArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJCR1c2VCYXNlbGluZXhweHcEAAAABHNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgAIcQB+AA5zcQB+AAsAAAADc3EAfgAIcQB+ABBzcQB+AAsAAAAEc3EAfgAIcQB+ABJzcQB+AAsAAAAFeA==")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5, 5 -> 6))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaE1hcCQAAAAAAAAAAwMAAUwACEVtcHR5TWFwdAAkTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0hhc2hNYXA7eHB4dwQAAAAFc3IADHNjYWxhLlR1cGxlMhDcBtcOREuEAgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAKeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAABXNxAH4ADAAAAAZzcQB+AAlzcQB+AAwAAAABc3EAfgAMAAAAAnNxAH4ACXEAfgASc3EAfgAMAAAAA3NxAH4ACXEAfgAUc3EAfgAMAAAABHNxAH4ACXEAfgAWcQB+AA54")

    //check(g)(immutable.Stack(1, 2, 3))( "rO0ABXNyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TdGFjaxtt3qEbMvq+AgABTAAFZWxlbXN0ACFMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvTGlzdDt4cHNyADJzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0JFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAwAAeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABQAAAAJzcQB+AAUAAAADc3IALHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3RTZXJpYWxpemVFbmQkilxjW/dTC20CAAB4cHg=")

    // TODO scala/bug#8576 Uninitialized field: IndexedSeqLike.scala: 56
    // check(g)(immutable.Stream(1, 2, 3))( "rO0ABXNyACZzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TdHJlYW0kQ29uc/ekjBXM3TlFAgADTAACaGR0ABJMamF2YS9sYW5nL09iamVjdDtMAAV0bEdlbnQAEUxzY2FsYS9GdW5jdGlvbjA7TAAFdGxWYWx0ACNMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvU3RyZWFtO3hyACFzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TdHJlYW0552RDntM42gIAAHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcgAtc2NhbGEuY29sbGVjdGlvbi5JdGVyYXRvciQkYW5vbmZ1biR0b1N0cmVhbSQxRWR4We0SX0UCAAFMAAYkb3V0ZXJ0ABtMc2NhbGEvY29sbGVjdGlvbi9JdGVyYXRvcjt4cHNyAChzY2FsYS5jb2xsZWN0aW9uLkluZGV4ZWRTZXFMaWtlJEVsZW1lbnRzGF+1cBwmcx0CAANJAANlbmRJAAVpbmRleEwABiRvdXRlcnQAIUxzY2FsYS9jb2xsZWN0aW9uL0luZGV4ZWRTZXFMaWtlO3hwAAAAAwAAAAFzcgArc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLldyYXBwZWRBcnJheSRvZkludMmRLBcI15VjAgABWwAFYXJyYXl0AAJbSXhwdXIAAltJTbpgJnbqsqUCAAB4cAAAAAMAAAABAAAAAgAAAANw")

    check(g)(immutable.TreeSet[Int]())(   "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JFKsUqd1equjAgAAeHBzcgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVHJlZVNldCQAAAAAAAAAAwIAAHhwdwT/////c3IAJnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5TZXJpYWxpemVFbmQkAAAAAAAAAAMCAAB4cHg=")

    // TODO scala/bug#8576 unstable under -Xcheckinit
    // check(g)(immutable.TreeSet(1, 2, 3))( "rO0ABXNyACJzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5UcmVlU2V0sRdVIDjbWAsCAAJMAAhvcmRlcmluZ3QAFUxzY2FsYS9tYXRoL09yZGVyaW5nO0wABHRyZWV0AC5Mc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvUmVkQmxhY2tUcmVlJFRyZWU7eHBzcgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkC4BMdr1Z51wCAAB4cHNyADFzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5SZWRCbGFja1RyZWUkQmxhY2tUcmVlzRxnCKenVAECAAB4cgAsc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuUmVkQmxhY2tUcmVlJFRyZWVrqCSyHJbsMgIABUkABWNvdW50TAADa2V5dAASTGphdmEvbGFuZy9PYmplY3Q7TAAEbGVmdHEAfgACTAAFcmlnaHRxAH4AAkwABXZhbHVlcQB+AAh4cAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAnNxAH4ABgAAAAFzcQB+AAoAAAABcHBzcgAXc2NhbGEucnVudGltZS5Cb3hlZFVuaXR0pn1HHezLmgIAAHhwc3EAfgAGAAAAAXNxAH4ACgAAAANwcHEAfgAQcQB+ABA=")

    // TODO scala/bug#8576 Uninitialized field under -Xcheckinit
    // check(g)(mutable.ArrayBuffer(1, 2, 3))(      "rO0ABXNyACRzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuQXJyYXlCdWZmZXIVOLBTg4KOcwIAA0kAC2luaXRpYWxTaXplSQAFc2l6ZTBbAAVhcnJheXQAE1tMamF2YS9sYW5nL09iamVjdDt4cAAAABAAAAADdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAEHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAFAAAAAnNxAH4ABQAAAANwcHBwcHBwcHBwcHBw")
    // TODO scala/bug#8576 Uninitialized field under -Xcheckinit
    // check(g)(mutable.ArraySeq(1, 2, 3))(         "rO0ABXNyACFzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuQXJyYXlTZXEVPD3SKEkOcwIAAkkABmxlbmd0aFsABWFycmF5dAATW0xqYXZhL2xhbmcvT2JqZWN0O3hwAAAAA3VyABNbTGphdmEubGFuZy5PYmplY3Q7kM5YnxBzKWwCAAB4cAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABQAAAAJzcQB+AAUAAAAD")
    check(g)(mutable.AnyRefMap("a" -> "A"))(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAtc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkFueVJlZk1hcCRUb0ZhY3RvcnkkAAAAAAAAAAMCAAB4cHcEAAAAAXNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ABnhwdAABYXQAAUF4")
    //check(g)(mutable.ArrayStack(1, 2, 3))(       "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuQXJyYXlTdGFja3bdxXbcnLBeAgACSQAqc2NhbGEkY29sbGVjdGlvbiRtdXRhYmxlJEFycmF5U3RhY2skJGluZGV4WwAqc2NhbGEkY29sbGVjdGlvbiRtdXRhYmxlJEFycmF5U3RhY2skJHRhYmxldAATW0xqYXZhL2xhbmcvT2JqZWN0O3hwAAAAA3VyABNbTGphdmEubGFuZy5PYmplY3Q7kM5YnxBzKWwCAAB4cAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAA3NxAH4ABQAAAAJzcQB+AAUAAAAB")
    //check(g)(mutable.DoubleLinkedList(1, 2, 3))( "rO0ABXNyAClzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuRG91YmxlTGlua2VkTGlzdI73LKsKRr1RAgADTAAEZWxlbXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wABG5leHR0ACtMc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL0RvdWJsZUxpbmtlZExpc3Q7TAAEcHJldnEAfgACeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AAHNxAH4ABAAAAAJzcQB+AABzcQB+AAQAAAADc3EAfgAAcHEAfgALcQB+AAlxAH4AB3EAfgADcA==")

    check(g)(mutable.HashMap())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAkAAAAAAAAAAMCAAB4cHcEAAAAAHg=")
    check(g)(mutable.HashMap(1 -> 1))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAkAAAAAAAAAAMCAAB4cHcEAAAAAXNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFxAH4ADXg=")
    check(g)(mutable.HashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAIXNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5IYXNoU2V0JAAAAAAAAAADAgAAeHB3BAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACAAAAAJzcQB+AAgAAAADeA==")
    check(g)(mutable.TreeMap[Int, Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDF0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyABhzY2FsYS5tYXRoLk9yZGVyaW5nJEludCRSrFKndXqrowIAAHhwc3IAIXNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5UcmVlTWFwJAAAAAAAAAADAgAAeHB3BAAAAAB4")
    check(g)(mutable.TreeMap(1 -> 1, 3 -> 6))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDF0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyABhzY2FsYS5tYXRoLk9yZGVyaW5nJEludCRSrFKndXqrowIAAHhwc3IAIXNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5UcmVlTWFwJAAAAAAAAAADAgAAeHB3BAAAAAJzcgAMc2NhbGEuVHVwbGUyENwG1w5ES4QCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABcQB+ABBzcQB+AAtzcQB+AA4AAAADc3EAfgAOAAAABng=")
    check(g)(mutable.TreeMap(1 -> 1, 3 -> 6).range(1, 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDF0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyABhzY2FsYS5tYXRoLk9yZGVyaW5nJEludCRSrFKndXqrowIAAHhwc3IAIXNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5UcmVlTWFwJAAAAAAAAAADAgAAeHB3BP////9zcgAMc2NhbGEuVHVwbGUyENwG1w5ES4QCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABcQB+ABBzcgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")
    check(g)(mutable.TreeSet[Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JFKsUqd1equjAgAAeHBzcgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVTZXQkAAAAAAAAAAMCAAB4cHcEAAAAAHg=")
    check(g)(mutable.TreeSet(1, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JFKsUqd1equjAgAAeHBzcgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVTZXQkAAAAAAAAAAMCAAB4cHcEAAAAAnNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgALAAAAA3g=")
    check(g)(mutable.TreeSet(1, 3).range(1, 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JFKsUqd1equjAgAAeHBzcgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVTZXQkAAAAAAAAAAMCAAB4cHcE/////3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3IAJnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5TZXJpYWxpemVFbmQkAAAAAAAAAAMCAAB4cHg=")
    // TODO scala/bug#8576 Uninitialized field under -Xcheckinit
    // check(g)(new mutable.History())( "rO0ABXNyACBzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGlzdG9yeUhuXxDIFJrsAgACSQAKbWF4SGlzdG9yeUwAA2xvZ3QAIExzY2FsYS9jb2xsZWN0aW9uL211dGFibGUvUXVldWU7eHAAAAPoc3IAHnNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5RdWV1ZbjMURVfOuHHAgAAeHIAJHNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5NdXRhYmxlTGlzdFJpnjJ+gFbAAgADSQADbGVuTAAGZmlyc3QwdAAlTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9MaW5rZWRMaXN0O0wABWxhc3QwcQB+AAV4cAAAAABzcgAjc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkxpbmtlZExpc3Sak+nGCZHaUQIAAkwABGVsZW10ABJMamF2YS9sYW5nL09iamVjdDtMAARuZXh0dAAeTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9TZXE7eHBwcQB+AApxAH4ACg==")
    check(g)(mutable.LinkedHashMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAnc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkxpbmtlZEhhc2hNYXAkAAAAAAAAAAMCAAB4cHcEAAAAAXNyAAxzY2FsYS5UdXBsZTIQ3AbXDkRLhAIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACeA==")
    check(g)(mutable.LinkedHashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJ3NjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5MaW5rZWRIYXNoU2V0JAAAAAAAAAADAgAAeHB3BAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACAAAAAJzcQB+AAgAAAADeA==")
    //check(g)(mutable.LinkedList(1, 2, 3))( "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuTGlua2VkTGlzdJqT6cYJkdpRAgACTAAEZWxlbXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wABG5leHR0ACVMc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL0xpbmtlZExpc3Q7eHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AAHNxAH4ABAAAAAJzcQB+AABzcQB+AAQAAAADc3EAfgAAcHEAfgAL")

    // TODO scala/bug#8576 unstable under -Xcheckinit
    // check(g)(mutable.ListBuffer(1, 2, 3))( "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuTGlzdEJ1ZmZlci9y9I7QyWzGAwAEWgAIZXhwb3J0ZWRJAANsZW5MAAVsYXN0MHQAKUxzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS8kY29sb24kY29sb247TAAqc2NhbGEkY29sbGVjdGlvbiRtdXRhYmxlJExpc3RCdWZmZXIkJHN0YXJ0dAAhTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0xpc3Q7eHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABAAAAAJzcQB+AAQAAAADc3IALHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3RTZXJpYWxpemVFbmQkilxjW/dTC20CAAB4cHcFAAAAAAN4")
    check(g)(new mutable.StringBuilder(new java.lang.StringBuilder("123")))( "rO0ABXNyACZzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuU3RyaW5nQnVpbGRlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAGUxqYXZhL2xhbmcvU3RyaW5nQnVpbGRlcjt4cgAkc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkFic3RyYWN0U2VxAAAAAAAAAAMCAAB4cgAcc2NhbGEuY29sbGVjdGlvbi5BYnN0cmFjdFNlcQAAAAAAAAADAgAAeHIAIXNjYWxhLmNvbGxlY3Rpb24uQWJzdHJhY3RJdGVyYWJsZQAAAAAAAAADAgAAeHBzcgAXamF2YS5sYW5nLlN0cmluZ0J1aWxkZXI81fsUWkxqywMAAHhwdwQAAAADdXIAAltDsCZmsOJdhKwCAAB4cAAAABMAMQAyADMAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAHg=")
    check(g)(mutable.UnrolledBuffer[Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uQ2xhc3NUYWdTZXFGYWN0b3J5JEFueVNlcURlbGVnYXRlAAAAAAAAAAMCAAB4cgA8c2NhbGEuY29sbGVjdGlvbi5DbGFzc1RhZ0l0ZXJhYmxlRmFjdG9yeSRBbnlJdGVyYWJsZURlbGVnYXRlAAAAAAAAAAMCAAFMAAhkZWxlZ2F0ZXQAKkxzY2FsYS9jb2xsZWN0aW9uL0NsYXNzVGFnSXRlcmFibGVGYWN0b3J5O3hwc3IAKHNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5VbnJvbGxlZEJ1ZmZlciQAAAAAAAAAAwMABEkADnVucm9sbGVkbGVuZ3RoSQAJd2F0ZXJsaW5lSQAOd2F0ZXJsaW5lRGVsaW1MAAh1bnRhZ2dlZHQAHUxzY2FsYS9jb2xsZWN0aW9uL1NlcUZhY3Rvcnk7eHB4dwT/////c3IAJnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5TZXJpYWxpemVFbmQkAAAAAAAAAAMCAAB4cHg=")

    check(g)("...".r)("rO0ABXNyABlzY2FsYS51dGlsLm1hdGNoaW5nLlJlZ2V44u3Vap7wIb8CAAJMAAdwYXR0ZXJudAAZTGphdmEvdXRpbC9yZWdleC9QYXR0ZXJuO0wAJXNjYWxhJHV0aWwkbWF0Y2hpbmckUmVnZXgkJGdyb3VwTmFtZXN0ACBMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvU2VxO3hwc3IAF2phdmEudXRpbC5yZWdleC5QYXR0ZXJuRmfVa25JAg0CAAJJAAVmbGFnc0wAB3BhdHRlcm50ABJMamF2YS9sYW5nL1N0cmluZzt4cAAAAAB0AAMuLi5zcgAyc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLkRlZmF1bHRTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAwMAAUwAB2ZhY3Rvcnl0ABpMc2NhbGEvY29sbGVjdGlvbi9GYWN0b3J5O3hwc3IAKnNjYWxhLmNvbGxlY3Rpb24uSXRlcmFibGVGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAIkxzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhYmxlRmFjdG9yeTt4cHNyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0JAAAAAAAAAADAwABTAARcGFydGlhbE5vdEFwcGxpZWR0ABFMc2NhbGEvRnVuY3Rpb24xO3hweHcEAAAAAHg=",
      r => (r.toString))
  }
}

class SerializationStabilityTest {
  @Test
  def testAll: Unit = SerializationStability.main(new Array[String](0))
}