/*
 * reserved comment block
 * DO NOT REMOVE OR ALTER!
 */
/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sun.org.apache.bcel.internal.classfile;

import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * represents one parameter annotation in the parameter annotation table
 *
 * @version $Id: ParameterAnnotationEntry
 * @since 6.0
 */
public class ParameterAnnotationEntry implements Node {

    private final AnnotationEntry[] annotation_table;


    /**
     * Construct object from input stream.
     *
     * @param input Input stream
     * @throws IOException
     */
    ParameterAnnotationEntry(final DataInput input, final ConstantPool constant_pool) throws IOException {
        final int annotation_table_length = input.readUnsignedShort();
        annotation_table = new AnnotationEntry[annotation_table_length];
        for (int i = 0; i < annotation_table_length; i++) {
            // TODO isRuntimeVisible
            annotation_table[i] = AnnotationEntry.read(input, constant_pool, false);
        }
    }


    /**
     * Called by objects that are traversing the nodes of the tree implicitely
     * defined by the contents of a Java class. I.e., the hierarchy of methods,
     * fields, attributes, etc. spawns a tree of objects.
     *
     * @param v Visitor object
     */
    @Override
    public void accept( final Visitor v ) {
        v.visitParameterAnnotationEntry(this);
    }

    /**
     * returns the array of annotation entries in this annotation
     */
    public AnnotationEntry[] getAnnotationEntries() {
        return annotation_table;
    }

    public void dump(final DataOutputStream dos) throws IOException {
        dos.writeShort(annotation_table.length);
        for (final AnnotationEntry entry : annotation_table) {
            entry.dump(dos);
        }
    }

  public static ParameterAnnotationEntry[] createParameterAnnotationEntries(final Attribute[] attrs) {
      // Find attributes that contain parameter annotation data
      final List<ParameterAnnotationEntry> accumulatedAnnotations = new ArrayList<>(attrs.length);
      for (final Attribute attribute : attrs) {
          if (attribute instanceof ParameterAnnotations) {
              final ParameterAnnotations runtimeAnnotations = (ParameterAnnotations)attribute;
              Collections.addAll(accumulatedAnnotations, runtimeAnnotations.getParameterAnnotationEntries());
          }
      }
      return accumulatedAnnotations.toArray(new ParameterAnnotationEntry[accumulatedAnnotations.size()]);
  }
}
