/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __RTSP_ENGINE_H__
#define __RTSP_ENGINE_H__

/**
 * @file rtsp_engine.h
 * @brief RTSP Engine.
 */ 

#include "apt.h"
#include "apt_unique_id.h"
#include "rtsp_message.h"

APT_BEGIN_EXTERN_C

typedef struct apt_unique_id rtsp_session_id;
typedef struct rtsp_session_t rtsp_session_t;
typedef struct rtsp_engine_t rtsp_engine_t;
typedef struct rtsp_engine_event_handler_t rtsp_engine_event_handler_t;

/** RTSP engine event handler */
struct rtsp_engine_event_handler_t {
	apt_bool_t (*on_session_create)(rtsp_engine_event_handler_t *handler, rtsp_session_t *session);
	apt_bool_t (*on_session_terminate)(rtsp_engine_event_handler_t *handler, rtsp_session_t *session);

	apt_bool_t (*on_request_receive)(rtsp_engine_event_handler_t *handler, rtsp_session_t *session, rtsp_message_t *request);
	apt_bool_t (*on_response_receive)(rtsp_engine_event_handler_t *handler, rtsp_session_t *session, rtsp_message_t *response, rtsp_message_t *request);

	void *object;
};


/** 
 * Start RTSP client engine.
 */
rtsp_engine_t* rtsp_client_engine_start(rtsp_engine_event_handler_t *handler, const char *server_ip, apr_port_t server_port, const char *resource_location);

/** 
 * Start RTSP server engine.
 */
rtsp_engine_t* rtsp_server_engine_start(rtsp_engine_event_handler_t *handler, const char *listen_ip, apr_port_t listen_port, const char *resource_location);

/** 
 * Shutdown RTSP engine.
 * @param engine the engine to destroy
 */
apt_bool_t rtsp_engine_shutdown(rtsp_engine_t *engine);

/** 
 * Create RTSP session (communication handle).
 * @param engine the engine
 * @param message the optional message to send
 */
rtsp_session_t* rtsp_engine_session_create(rtsp_engine_t *engine, rtsp_message_t *message);

/** 
 * Start to terminate RTSP session.
 * @param session the session to terminate
 */
apt_bool_t rtsp_engine_session_terminate(rtsp_session_t *session);

/** 
 * Finally destroy RTSP session.
 * @param session the session to destroy.
 */
apt_bool_t rtsp_engine_session_destroy(rtsp_session_t *session);

/** 
 * Send RTSP message.
 * @param engine the engine
 * @param message the message to send
 */
apt_bool_t rtsp_engine_message_send(rtsp_session_t *session, rtsp_message_t *message);

/** 
 * Associate external context specific object with RTSP session.
 * @param session the session to associate with
 * @param object the object to associate.
 */
void rtsp_engine_session_object_set(rtsp_session_t *session, void *object);

/** 
 * Get external context specific object associated with RTSP session.
 * @param session the session to get associate object from
 */
void* rtsp_engine_session_object_get(rtsp_session_t *session);

/** 
 * Get RTSP session identifier.
 * @param session the session to identifier from
 */
rtsp_session_id* rtsp_engine_session_id_get(rtsp_session_t *session);


APT_END_EXTERN_C

#endif /*__RTSP_ENGINE_H__*/
