/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_SERVER_SESSION_H__
#define __MRCP_SERVER_SESSION_H__

/**
 * @file mrcp_server_session.h
 * @brief MRCP Server Session Definition.
 */ 

#include <apr_pools.h>
#include "mrcp_server_types.h"
#include "mrcp_server_resource.h"
#include "mrcp_description.h"

MRCP_BEGIN_EXTERN_C


/** MRCP session */
struct mrcp_session_t {
	/** Unique session identifier */
	mrcp_session_id           id;
	
	/** Number of channels and actual type of channels (control,audio,...) 
	 *  should be referenced by signaling_channel->local_descriptor 
	 */
	void                     *channels[MRCP_MAX_MEDIA_COUNT];

	/** Signaling channel */
	mrcp_signaling_channel_t *signaling_channel;

	mrcp_status_t             terminating;

	/** Memory pool to allocate memory from */
	apr_pool_t               *pool;
};


/** Create MRCP server session */
mrcp_session_t* mrcp_server_session_create(mrcp_session_id *session_id);

/** Destroy MRCP server session */
mrcp_status_t mrcp_server_session_destroy(mrcp_session_t *session);

/** Create MRCP channel */
mrcp_server_channel_t* mrcp_server_channel_create(mrcp_session_t *session, mrcp_server_resource_t *resource);

/** Destroy MRCP channel */
mrcp_server_channel_state_t mrcp_server_channel_destroy(mrcp_server_channel_t *channel);


/** Handle MRCP channel events */
mrcp_status_t mrcp_server_channel_event_handler(mrcp_server_channel_t *channel, mrcp_message_t *event_message);
/** Handle MRCP channel responses */
mrcp_status_t mrcp_server_channel_response_handler(mrcp_server_channel_t *channel, mrcp_message_t *response_message);
/** Process MRCP message */
mrcp_status_t mrcp_server_message_process(mrcp_session_t *session, mrcp_message_t *mrcp_message, mrcp_message_t *response_message);

/** Set the associations between control and audio channels */
mrcp_status_t mrcp_server_channel_assocciations_set(mrcp_session_t *session);

/** Set Channel to MRCP Server Session */
mrcp_status_t mrcp_server_session_channel_set(mrcp_session_t *session, void *channel, size_t index);

/** Get Channel from MRCP Server Session */
void* mrcp_server_session_channel_get(mrcp_session_t *session, size_t index);


MRCP_END_EXTERN_C

#endif /*__MRCP_SERVER_SESSION_H__*/
