/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_resource_manager.h"
#include "mrcp_resource_set.h"

#include "mrcp_synthesizer.h"
#include "mrcp_recognizer.h"

#include "mrcp_synthesizer_channel.h"
#include "mrcp_recognizer_channel.h"


/* string table of MRCPv1 resources (mrcp_resource_set_t) */
static const apt_string_table_item_t mrcp_v1_resource_string_table[] = {
	{"speechsynthesizer",17,6},
	{"speechrecognizer", 16,6}
};

/* string table of MRCPv2 resources (mrcp_resource_set_t) */
static const apt_string_table_item_t mrcp_v2_resource_string_table[] = {
	{"speechsynth",11,6},
	{"speechrecog",11,6}
};

static const apt_string_table_item_t* mrcp_resource_string_table_get(mrcp_version_t version,
																	 const apt_string_table_item_t *resource_string_table)
{
	if(resource_string_table) {
		return resource_string_table;
	}
	
	if(version == MRCP_VERSION_1) {
		return mrcp_v1_resource_string_table;
	}
	return mrcp_v2_resource_string_table;
}

mrcp_resource_container_t* mrcp_parser_resource_container_create(mrcp_version_t version, 
																 const apt_string_table_item_t *resource_string_table,
																 apr_pool_t *pool)
{
	mrcp_resource_t *mrcp_resource;
	mrcp_resource_container_t *resource_container = mrcp_resource_container_create(
														MRCP_RESOURCE_COUNT,
														mrcp_resource_string_table_get(version,resource_string_table),
														pool);
	if(!resource_container) {
		return NULL;
	}

	mrcp_resource = apr_palloc(pool,sizeof(mrcp_resource_t));
	mrcp_synthesizer_init(mrcp_resource,version);
	mrcp_resource->id = MRCP_RESOURCE_SYNTHESIZER;
	mrcp_resource_register(resource_container,mrcp_resource);

	mrcp_resource = apr_palloc(pool,sizeof(mrcp_resource_t));
	mrcp_recognizer_init(mrcp_resource,version);
	mrcp_resource->id = MRCP_RESOURCE_RECOGNIZER;
	mrcp_resource_register(resource_container,mrcp_resource);

	return resource_container;
}

mrcp_resource_container_t* mrcp_server_resource_container_create(mrcp_version_t version, 
																 const apt_string_table_item_t *resource_string_table,
																 apr_pool_t *pool)
{
	mrcp_resource_t *resource;
	mrcp_resource_container_t *resource_container = mrcp_resource_container_create(
														MRCP_RESOURCE_COUNT,
														mrcp_resource_string_table_get(version,resource_string_table),
														pool);
	if(!resource_container) {
		return NULL;
	}

	resource = mrcp_synthesizer_server_resource_create(version,pool);
	if(resource) {
		resource->id = MRCP_RESOURCE_SYNTHESIZER;
		mrcp_resource_register(resource_container,resource);
	}

	resource = mrcp_recognizer_server_resource_create(version,pool);
	if(resource) {
		resource->id = MRCP_RESOURCE_RECOGNIZER;
		mrcp_resource_register(resource_container,resource);
	}
	
	return resource_container;
}

mrcp_resource_container_t* mrcp_client_resource_container_create(mrcp_version_t version, 
																 const apt_string_table_item_t *resource_string_table,
																 apr_pool_t *pool)
{
	return mrcp_parser_resource_container_create(version,resource_string_table,pool);
}
