/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_recognizer_channel.h"
#include "mrcp_recognizer.h"
#include "mrcp_generic_header.h"
#include "apt_ptr_list.h"

typedef struct mrcp_recognizer_channel_t mrcp_recognizer_channel_t;

struct mrcp_recognizer_channel_t {
	mrcp_recognizer_state_t state;
	apt_ptr_list_t         *queue;
};

static mrcp_status_t recognizer_queue_process(mrcp_server_channel_t *channel, mrcp_server_method_plugin_t method_plugin, mrcp_message_t *response_message);
static mrcp_status_t recognizer_stop_process(mrcp_server_channel_t *channel, mrcp_server_method_plugin_t method_plugin, mrcp_message_t *request_message, mrcp_message_t *response_message);


static mrcp_server_channel_state_t recognizer_channel_open(mrcp_server_channel_t *channel, apr_pool_t *pool)
{
	mrcp_server_resource_t *resource = channel->resource;
	mrcp_recognizer_channel_t *recog_channel = apr_palloc(pool,sizeof(mrcp_recognizer_channel_t));
	recog_channel->state = RECOGNIZER_STATE_IDLE;
	recog_channel->queue = apt_list_create(pool);

	apt_log(APT_PRIO_INFO,"Recognizer Channel Open\n");
	channel->data = recog_channel;
	channel->state = MRCP_CHANNEL_STATE_OPENED;
	if(resource->plugin->channel_open) {
		if(resource->plugin->channel_open(resource->plugin,channel,pool) != MRCP_STATUS_SUCCESS) {
			channel->state = MRCP_CHANNEL_STATE_NONE;
		}
	}
	return channel->state;
}

static mrcp_server_channel_state_t recognizer_channel_close(mrcp_server_channel_t *channel)
{
	mrcp_server_resource_t *resource = channel->resource;
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	if(channel->state != MRCP_CHANNEL_STATE_OPENED) {
		return channel->state;
	}

	apt_log(APT_PRIO_INFO,"Recognizer Channel Close\n");
	if(apt_list_head(recog_channel->queue) == NULL) {
		if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZING) {
			apt_log(APT_PRIO_ERROR,"Channel State Error: expected to be idle\n");
		}
		if(resource->plugin->channel_close) {
			resource->plugin->channel_close(resource->plugin,channel);
		}
		channel->state = MRCP_CHANNEL_STATE_CLOSED;
	}
	else {
		if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZING) {
			mrcp_server_method_plugin_t method_plugin = resource->plugin->method_plugin_array[RECOGNIZER_STOP];
			recognizer_stop_process(channel,method_plugin,NULL,NULL);
		}
		channel->state = MRCP_CHANNEL_STATE_CLOSE_REQUESTED;
	}
	return channel->state;
}

static mrcp_status_t recognizer_queue_process(mrcp_server_channel_t *channel, mrcp_server_method_plugin_t method_plugin, mrcp_message_t *response_message)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	mrcp_message_t *request_message = apt_list_head(recog_channel->queue);
	if(!request_message || !method_plugin) {
		return MRCP_STATUS_FAILURE;
	}

	apt_log(APT_PRIO_DEBUG,"Process Recognition Request [%d]\n",request_message->start_line.request_id);
	if(method_plugin(channel,request_message,response_message) != MRCP_STATUS_SUCCESS) {
		apt_log(APT_PRIO_WARNING,"Failed to Process Recognition Request [%d]\n",request_message->start_line.request_id);
		apt_list_pop_front(recog_channel->queue);
		recog_channel->state = RECOGNIZER_STATE_IDLE;
		return MRCP_STATUS_FAILURE;
	}
	recog_channel->state = RECOGNIZER_STATE_RECOGNIZING;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_stop_process(mrcp_server_channel_t *channel, mrcp_server_method_plugin_t method_plugin, mrcp_message_t *request_message, mrcp_message_t *response_message)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	apt_log(APT_PRIO_DEBUG,"Stop Active Recognition Request\n");
	if(method_plugin) {
		method_plugin(channel,request_message,response_message);
	}
	recog_channel->state = RECOGNIZER_STATE_IDLE;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_definegrammar(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	if(recog_channel->state != RECOGNIZER_STATE_RECOGNIZING) {
		if(!method_plugin || method_plugin(channel,request_message,response_message) != MRCP_STATUS_SUCCESS) {
			response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_FAILED;
		}
	}
	else {
		response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_NOT_VALID;
	}
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_recognize(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	apt_log(APT_PRIO_DEBUG,"Add Recognition Request [%d]\n",request_message->start_line.request_id);
	mrcp_header_inherit(&request_message->header,&channel->properties,request_message->pool);
	apt_list_push_back(recog_channel->queue,request_message);

	if(apt_list_head(recog_channel->queue) == request_message) {
		if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZING) {
			apt_log(APT_PRIO_ERROR,"Channel State Error: expected to be idle\n");
		}
		if(recognizer_queue_process(channel,method_plugin,response_message) == MRCP_STATUS_SUCCESS) {
			response_message->start_line.request_state = MRCP_REQUEST_STATE_INPROGRESS;
		}
		else {
			response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_FAILED;
		}
	}
	else {
		response_message->start_line.request_state = MRCP_REQUEST_STATE_PENDING;
	}

	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_get_result(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZED) {
		if(!method_plugin || method_plugin(channel,request_message,response_message) != MRCP_STATUS_SUCCESS) {
			response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_FAILED;
		}
	}
	else {
		response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_NOT_VALID;
	}
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_recognition_start_timers(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZING) {
		if(!method_plugin || method_plugin(channel,request_message,response_message) != MRCP_STATUS_SUCCESS) {
			response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_FAILED;
		}
	}
	else {
		response_message->start_line.status_code = MRCP_STATUS_CODE_METHOD_NOT_VALID;
	}
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recog_request_id_find(mrcp_request_id_list_t *request_id_list, mrcp_request_id request_id)
{
	size_t i;
	for(i=0; i<request_id_list->count; i++) {
		if(request_id_list->ids[i] == request_id) {
			return MRCP_STATUS_SUCCESS;
		}
	}
	return MRCP_STATUS_FAILURE;
}

static mrcp_status_t recognizer_stop(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	mrcp_request_id_list_t *request_id_list = NULL;
	mrcp_message_t *message;
	apt_list_elem_t *elem;
	if(recog_channel->state != RECOGNIZER_STATE_RECOGNIZING) {
		/* send response with empty complete */
		return MRCP_STATUS_SUCCESS;
	}
	
	mrcp_generic_header_prepare(response_message);

	if(mrcp_generic_header_property_check(request_message,GENERIC_HEADER_ACTIVE_REQUEST_ID_LIST) == MRCP_STATUS_SUCCESS) {
		mrcp_generic_header_t *generic_header = request_message->header.generic_header.data;
		if(generic_header->active_request_id_list.ids && generic_header->active_request_id_list.count) {
			/* selective stop request */
			request_id_list = &generic_header->active_request_id_list;
		}
	}

	elem = apt_list_first_elem_get(recog_channel->queue);
	while(elem) {
		message = apt_list_elem_data_get(elem);
		if(!request_id_list || recog_request_id_find(request_id_list,message->start_line.request_id) == MRCP_STATUS_SUCCESS) {
			apt_log(APT_PRIO_DEBUG,"Remove Recognition Request [%d]\n",message->start_line.request_id);
			/* append active id list */
			active_request_id_list_append(response_message->header.generic_header.data,message->start_line.request_id);
			if(recog_channel->state == RECOGNIZER_STATE_RECOGNIZING) {
				/* recognition request remains in the queue until stop complete response */
				recognizer_stop_process(channel,method_plugin,request_message,response_message);
				elem = apt_list_next_elem_get(recog_channel->queue,elem);
			}
			else {
				/* remove recognition request */
				elem = apt_list_elem_remove(recog_channel->queue,elem);
			}
		}
		else {
			/* recognition request remains, just proceed to the next one */
			elem = apt_list_next_elem_get(recog_channel->queue,elem);
		}
	}

	mrcp_generic_header_property_add(response_message,GENERIC_HEADER_ACTIVE_REQUEST_ID_LIST);
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_start_of_speech(mrcp_server_channel_t *channel, mrcp_message_t *event_message)
{
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	mrcp_message_t *recog = apt_list_head(recog_channel->queue);
	if(recog_channel->state != RECOGNIZER_STATE_RECOGNIZING || !recog || !event_message) {
		/* unexpected event */
		return MRCP_STATUS_FAILURE;
	}

	if(recog->start_line.request_id != event_message->start_line.request_id) {
		/* unexpected event */
		return MRCP_STATUS_FAILURE;
	}
	
	apt_log(APT_PRIO_DEBUG,"Start of Speech [%d]\n",recog->start_line.request_id);
	event_message->start_line.request_state = MRCP_REQUEST_STATE_INPROGRESS;
	event_message->start_line.status_code = MRCP_STATUS_CODE_SUCCESS;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_recognition_complete(mrcp_server_channel_t *channel, mrcp_message_t *event_message)
{
	mrcp_server_resource_t *resource = channel->resource;
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	mrcp_message_t *recog = apt_list_head(recog_channel->queue);
	if(recog_channel->state != RECOGNIZER_STATE_RECOGNIZING || !recog || !event_message) {
		/* unexpected event */
		return MRCP_STATUS_FAILURE;
	}

	if(recog->start_line.request_id != event_message->start_line.request_id) {
		/* unexpected event */
		return MRCP_STATUS_FAILURE;
	}

	if(!resource) {
		return MRCP_STATUS_FAILURE;
	}

	apt_log(APT_PRIO_DEBUG,"Recognition Complete [%d]\n",recog->start_line.request_id);
	event_message->start_line.request_state = MRCP_REQUEST_STATE_COMPLETE;

	recog_channel->state = RECOGNIZER_STATE_RECOGNIZED;
	apt_list_pop_front(recog_channel->queue);
	/* process remaining requests */
	recognizer_queue_process(channel,resource->plugin->method_plugin_array[RECOGNIZER_RECOGNIZE],NULL);
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_response_handler(mrcp_server_channel_t *channel, mrcp_message_t *response_message)
{
	/* processing async response to stop request */
	mrcp_server_resource_t *resource = channel->resource;
	mrcp_recognizer_channel_t *recog_channel = channel->data;
	mrcp_message_t *recog = apt_list_head(recog_channel->queue);
	if(recog_channel->state != RECOGNIZER_STATE_IDLE || !recog || !response_message) {
		/* unexpected response */
		return MRCP_STATUS_FAILURE;
	}
	if(recog->start_line.request_id != response_message->start_line.request_id) {
		/* unexpected response */
		return MRCP_STATUS_FAILURE;
	}

	if(!resource) {
		return MRCP_STATUS_FAILURE;
	}

	apt_log(APT_PRIO_DEBUG,"Stop Complete [%d]\n",recog->start_line.request_id);
	apt_list_pop_front(recog_channel->queue);

	if(channel->state == MRCP_CHANNEL_STATE_CLOSE_REQUESTED) {
		if(resource->plugin->channel_close) {
			resource->plugin->channel_close(resource->plugin,channel);
		}
		channel->state = MRCP_CHANNEL_STATE_CLOSED;
		return MRCP_STATUS_SUCCESS;
	}

	/* process remaining requests */
	recognizer_queue_process(channel,resource->plugin->method_plugin_array[RECOGNIZER_RECOGNIZE],NULL);
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_setparams(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_header_set(&channel->properties,&request_message->header,request_message->pool);
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t recognizer_getparams(mrcp_server_channel_t *channel, mrcp_message_t *request_message, mrcp_message_t *response_message, mrcp_server_method_plugin_t method_plugin)
{
	mrcp_header_set(&response_message->header,&request_message->header,response_message->pool);
	mrcp_header_get(&response_message->header,&channel->properties,response_message->pool);
	return MRCP_STATUS_SUCCESS;
}


static mrcp_server_method_handler_t recognizer_server_method_array[RECOGNIZER_METHOD_COUNT] = {
	recognizer_setparams,
	recognizer_getparams,
	recognizer_definegrammar,
	recognizer_recognize,
	recognizer_get_result,
	recognizer_recognition_start_timers,
	recognizer_stop
};

static mrcp_server_event_handler_t recognizer_server_event_array[RECOGNIZER_EVENT_COUNT] = {
	recognizer_start_of_speech,
	recognizer_recognition_complete
};


/* creates mrcp recognizer server resource */
mrcp_resource_t* mrcp_recognizer_server_resource_create(mrcp_version_t version, apr_pool_t *pool)
{
	mrcp_server_resource_t *server_resource = apr_palloc(pool,sizeof(mrcp_server_resource_t));
	mrcp_recognizer_init(&server_resource->base,version);

	server_resource->channel_open = recognizer_channel_open;
	server_resource->channel_close = recognizer_channel_close;

	server_resource->method_array = recognizer_server_method_array;
	server_resource->event_array = recognizer_server_event_array;
	server_resource->response_handler = recognizer_response_handler;

	server_resource->plugin = NULL;
	return &server_resource->base;
}
