/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_RECOGNIZER_H__
#define __MRCP_RECOGNIZER_H__

/**
 * @file mrcp_recognizer.h
 * @brief MRCP Recognizer Resource Definition
 */ 

#include "mrcp_resource.h"

MRCP_BEGIN_EXTERN_C

/** MRCP recognizer methods */
typedef enum {
	RECOGNIZER_SET_PARAMS,
	RECOGNIZER_GET_PARAMS,
	RECOGNIZER_DEFINE_GRAMMAR,
	RECOGNIZER_RECOGNIZE,
	RECOGNIZER_GET_RESULT,
	RECOGNIZER_START_INPUT_TIMERS,
	RECOGNIZER_STOP,

	RECOGNIZER_METHOD_COUNT
} mrcp_recognizer_method_id;

/** MRCP recognizer events */
typedef enum {
	RECOGNIZER_START_OF_INPUT,
	RECOGNIZER_RECOGNITION_COMPLETE,

	RECOGNIZER_EVENT_COUNT
} mrcp_recognizer_event_id;

/** MRCP recognizer headers */
typedef enum {
	RECOGNIZER_HEADER_CONFIDENCE_THRESHOLD,
	RECOGNIZER_HEADER_SENSITIVITY_LEVEL,
	RECOGNIZER_HEADER_SPEED_VS_ACCURACY,
	RECOGNIZER_HEADER_N_BEST_LIST_LENGTH,
	RECOGNIZER_HEADER_NO_INPUT_TIMEOUT,
	RECOGNIZER_HEADER_RECOGNITION_TIMEOUT,
	RECOGNIZER_HEADER_WAVEFORM_URI,
	RECOGNIZER_HEADER_INPUT_WAVEFORM_URI,
	RECOGNIZER_HEADER_COMPLETION_CAUSE,
	RECOGNIZER_HEADER_RECOGNIZER_CONTEXT_BLOCK,
	RECOGNIZER_HEADER_START_INPUT_TIMERS,
	RECOGNIZER_HEADER_VENDOR_SPECIFIC,
	RECOGNIZER_HEADER_SPEECH_COMPLETE_TIMEOUT,
	RECOGNIZER_HEADER_SPEECH_INCOMPLETE_TIMEOUT,
	RECOGNIZER_HEADER_DTMF_INTERDIGIT_TIMEOUT,
	RECOGNIZER_HEADER_DTMF_TERM_TIMEOUT,
	RECOGNIZER_HEADER_DTMF_TERM_CHAR,
	RECOGNIZER_HEADER_FETCH_TIMEOUT,
	RECOGNIZER_HEADER_FAILED_URI,
	RECOGNIZER_HEADER_FAILED_URI_CAUSE,
	RECOGNIZER_HEADER_SAVE_WAVEFORM,
	RECOGNIZER_HEADER_NEW_AUDIO_CHANNEL,
	RECOGNIZER_HEADER_SPEECH_LANGUAGE,
	RECOGNIZER_HEADER_VER_BUFFER_UTTERANCE,
	RECOGNIZER_HEADER_RECOGNITION_MODE,
	RECOGNIZER_HEADER_CANCEL_IF_QUEUE,
	RECOGNIZER_HEADER_HOTWORD_MAX_DURATION,
	RECOGNIZER_HEADER_HOTWORD_MIN_DURATION,
	RECOGNIZER_HEADER_DTMF_BUFFER_TIME,
	RECOGNIZER_HEADER_CLEAR_DTMF_BUFFER,
	RECOGNIZER_HEADER_EARLY_NO_MATCH,

	RECOGNIZER_HEADER_COUNT
} mrcp_recognizer_header_id;


/** MRCP recognizer completion-cause  */
typedef enum {
	RECOGNIZER_COMPLETION_CAUSE_SUCCESS                 = 0,
	RECOGNIZER_COMPLETION_CAUSE_NO_MATCH                = 1,
	RECOGNIZER_COMPLETION_CAUSE_NO_INPUT_TIMEOUT        = 2,
	RECOGNIZER_COMPLETION_CAUSE_RECOGNITION_TIMEOUT     = 3,
	RECOGNIZER_COMPLETION_CAUSE_GRAM_LOAD_FAILURE       = 4,
	RECOGNIZER_COMPLETION_CAUSE_GRAM_COMP_FAILURE       = 5,
	RECOGNIZER_COMPLETION_CAUSE_ERROR                   = 6,
	RECOGNIZER_COMPLETION_CAUSE_SPEECH_TOO_EARLY        = 7,
	RECOGNIZER_COMPLETION_CAUSE_TOO_MUCH_SPEECH_TIMEOUT = 8,
	RECOGNIZER_COMPLETION_CAUSE_URI_FAILURE             = 9,
	RECOGNIZER_COMPLETION_CAUSE_LANGUAGE_UNSUPPORTED    = 10,
	RECOGNIZER_COMPLETION_CAUSE_CANCELLED               = 11,
	RECOGNIZER_COMPLETION_CAUSE_SEMANTICS_FAILURE       = 12,
	
	RECOGNIZER_COMPLETION_CAUSE_COUNT                   = 13,
	RECOGNIZER_COMPLETION_CAUSE_UNKNOWN                 = RECOGNIZER_COMPLETION_CAUSE_COUNT
} mrcp_recognizer_completion_cause_t;




typedef struct mrcp_recognizer_header_t mrcp_recognizer_header_t;

/** MRCP recognizer-header */
struct mrcp_recognizer_header_t {
	float                              confidence_threshold;
	size_t                             sensitivity_level;
	size_t                             speed_vs_accuracy;
	size_t                             n_best_list_length;
	size_t                             no_input_timeout;
	size_t                             recognition_timeout;
	char                              *waveform_uri;
	char                              *input_waveform_uri;
	mrcp_recognizer_completion_cause_t completion_cause;
	char                              *recognizer_context_block;
	apt_bool_t                         start_input_timers;
	char                              *vendor_specific;
	size_t                             speech_complete_timeout;
	size_t                             speech_incomplete_timeout;
	size_t                             dtmf_interdigit_timeout;
	size_t                             dtmf_term_timeout;
	char                               dtmf_term_char;
	size_t                             fetch_timeout;
	char                              *failed_uri;
	char                              *failed_uri_cause;
	apt_bool_t                         save_waveform;
	apt_bool_t                         new_audio_channel;
	char                              *speech_language;
	apt_bool_t                         ver_buffer_utterance;
	char                              *recognition_mode;
	apt_bool_t                         cancel_if_queue;
	size_t                             hotword_max_duration;
	size_t                             hotword_min_duration;
	size_t                             dtmf_buffer_time;
	apt_bool_t                         clear_dtmf_buffer;
	apt_bool_t                         early_no_match;
};

/** Initialize recognizer header */
void mrcp_recognizer_header_init(mrcp_recognizer_header_t *recognizer_header);

/** Destroy recognizer header */
void mrcp_recognizer_header_destroy(mrcp_recognizer_header_t *recognizer_header);


/** Initialize mrcp recognizer resource */
mrcp_status_t mrcp_recognizer_init(mrcp_resource_t *resource, mrcp_version_t version);

MRCP_END_EXTERN_C

#endif /*__MRCP_RECOGNIZER_H__*/
