/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_parser.h"
#include "mrcp_generic_header.h"
#include "mrcp_resource.h"


/* validates mrcp message for correctness */
static mrcp_status_t mrcp_message_validate(mrcp_message_t *message)
{
	if(message->body) {
		/* content length must be specified */
		mrcp_generic_header_t *generic_header = mrcp_generic_header_prepare(message);
		if(!generic_header) {
			return MRCP_STATUS_FAILURE;
		}
		if(mrcp_generic_header_property_check(message,GENERIC_HEADER_CONTENT_LENGTH) != MRCP_STATUS_SUCCESS ||
		  !generic_header->content_length) {
			generic_header->content_length = strlen(message->body);
			mrcp_generic_header_property_add(message,GENERIC_HEADER_CONTENT_LENGTH);
		}
	}

	return MRCP_STATUS_SUCCESS;
}

/* associate mrcp resource specific data by resource identifier */
mrcp_status_t mrcp_message_associate_resource_by_id(mrcp_resource_container_t *resource_container, mrcp_message_t *message)
{
	mrcp_resource_t *mrcp_resource;

	/* associates resource_name and resource_id */
	message->channel_id.resource_name = mrcp_resource_name_get(resource_container,message->channel_id.resource_id);
	if(!message->channel_id.resource_name) {
		return MRCP_STATUS_FAILURE;
	}
	mrcp_resource = resource_container->resource_array[message->channel_id.resource_id];
	if(!mrcp_resource) {
		return MRCP_STATUS_FAILURE;
	}

	/* associates method_name and method_id */
	if(message->start_line.message_type == MRCP_MESSAGE_TYPE_REQUEST) {
		message->start_line.method_name = apt_string_table_get(mrcp_resource->method_table,mrcp_resource->method_count,message->start_line.method_id);
		if(!message->start_line.method_name) {
			return MRCP_STATUS_FAILURE;
		}
	}
	else if(message->start_line.message_type == MRCP_MESSAGE_TYPE_EVENT) {
		message->start_line.method_name = apt_string_table_get(mrcp_resource->event_table,mrcp_resource->event_count,message->start_line.method_id);
		if(!message->start_line.method_name) {
			return MRCP_STATUS_FAILURE;
		}
	}

	/* sets header accessors for the entire message */
	message->header.resource_header.accessor = &mrcp_resource->header_accessor;
	message->header.generic_header.accessor = &resource_container->generic_header_accessor;
	return MRCP_STATUS_SUCCESS;
}

/* associate mrcp resource specific data by resource name */
mrcp_status_t mrcp_message_associate_resource_by_name(mrcp_resource_container_t *resource_container, mrcp_message_t *message)
{
	mrcp_resource_t *mrcp_resource;

	if(!message->channel_id.resource_name) {
		return MRCP_STATUS_FAILURE;
	}

	/* associates resource_name and resource_id */
	message->channel_id.resource_id = mrcp_resource_id_find(resource_container,message->channel_id.resource_name);
	if(message->channel_id.resource_id >= resource_container->resource_count) {
		return MRCP_STATUS_FAILURE;
	}
	mrcp_resource = resource_container->resource_array[message->channel_id.resource_id];
	if(!mrcp_resource) {
		return MRCP_STATUS_FAILURE;
	}

	/* associates method_name and method_id */
	if(message->start_line.message_type == MRCP_MESSAGE_TYPE_REQUEST) {
		if(!message->start_line.method_name) {
			return MRCP_STATUS_FAILURE;
		}
		message->start_line.method_id = apt_string_table_find(mrcp_resource->method_table,mrcp_resource->method_count,message->start_line.method_name);
		if(message->start_line.method_id >= mrcp_resource->method_count) {
			return MRCP_STATUS_FAILURE;
		}
	}
	else if(message->start_line.message_type == MRCP_MESSAGE_TYPE_EVENT) {
		if(!message->start_line.method_name) {
			return MRCP_STATUS_FAILURE;
		}
		message->start_line.method_id = apt_string_table_find(mrcp_resource->event_table,mrcp_resource->event_count,message->start_line.method_name);
		if(message->start_line.method_id >= mrcp_resource->event_count) {
			return MRCP_STATUS_FAILURE;
		}
	}

	/* sets header accessors for the entire message */
	message->header.resource_header.accessor = &mrcp_resource->header_accessor;
	message->header.generic_header.accessor = &resource_container->generic_header_accessor;
	return MRCP_STATUS_SUCCESS;
}


/* parses mrcp message */
mrcp_status_t mrcp_message_parse(mrcp_resource_container_t *resource_container, mrcp_message_t *message, apt_text_stream_t *text_stream)
{
	if(mrcp_start_line_parse(&message->start_line,text_stream,message->pool) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	if(message->start_line.version == MRCP_VERSION_2) {
		mrcp_channel_id_parse(&message->channel_id,text_stream,message->pool);
	}

	if(mrcp_message_associate_resource_by_name(resource_container,message) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	if(mrcp_header_parse(&message->header,text_stream,message->pool) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	mrcp_body_parse(message,text_stream,message->pool);

	return MRCP_STATUS_SUCCESS;
}


/* generates mrcp message */
mrcp_status_t mrcp_message_generate(mrcp_resource_container_t *resource_container, mrcp_message_t *message, apt_text_stream_t *text_stream)
{
	if(mrcp_message_associate_resource_by_id(resource_container,message) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	if(mrcp_message_validate(message) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}
	
	if(mrcp_start_line_generate(&message->start_line,text_stream) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	if(message->start_line.version == MRCP_VERSION_2) {
		mrcp_channel_id_generate(&message->channel_id,text_stream);
	}

	if(mrcp_header_generate(&message->header,text_stream) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}

	mrcp_body_generate(message,text_stream);
	
	text_stream->size = text_stream->pos - text_stream->buffer;
	mrcp_start_line_finalize(&message->start_line,text_stream);
	return MRCP_STATUS_SUCCESS;
}
