/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_CLIENT_SIGNALING_AGENT_H__
#define __MRCP_CLIENT_SIGNALING_AGENT_H__

/**
 * @file mrcp_client_signaling_agent.h
 * @brief MRCP Client Signaling Agent Definition.
 */ 

#include "mrcp_client_types.h"
#include "mrcp_module.h"
#include "mrcp_signaling_channel.h"

MRCP_BEGIN_EXTERN_C

typedef struct mrcp_signaling_agent_method_set_t mrcp_signaling_agent_method_set_t;
typedef struct mrcp_signaling_agent_event_set_t mrcp_signaling_agent_event_set_t;

/** Signaling agent base */
struct mrcp_signaling_agent_t {
	/** Base module */
	mrcp_module_t                             module;

	/** Set of signaling agent virtual methods */
	const mrcp_signaling_agent_method_set_t  *method_set;
	/** Set of signaling agent virtual events */
	const mrcp_signaling_agent_event_set_t   *event_set;
	/** Actual data set by inherited object */
	void                                     *object;

	/** Client IP address */
	char                                     *client_ip;
	/** Client SIP port */
	unsigned short                            client_sip_port;
	/** Server IP address */
	char                                     *server_ip;
	/** Server SIP port */
	unsigned short                            server_sip_port;

	/** SIP Contact string */
	char                                     *sip_contact_str;
	/** SIP To string */
	char                                     *sip_to_str;
	/** SIP From string */
	char                                     *sip_from_str;

	/** Resource location used as SIP username or RTSP URI prefix */
	char                                     *resource_location;

	/** Media descriptor, which stores server media capabilities */
	mrcp_descriptor_t                         server_capabilities;
};


/** Set of signaling agent virtual methods */
struct mrcp_signaling_agent_method_set_t {
	/** Resource discovery request */
	mrcp_status_t (*resource_discover)(mrcp_signaling_agent_t *agent);

	/** Create new signaling channel */
	mrcp_signaling_channel_t* (*channel_create)(mrcp_signaling_agent_t *agent, apr_pool_t *pool);
};

/** Set of signaling agent event handlers */
struct mrcp_signaling_agent_event_set_t {
	/** Resource discovery response */
	mrcp_status_t (*on_resource_discover)(mrcp_signaling_agent_t *agent);
};


/**
 * Create signaling agent object.
 * @param method_set the implemented set of signaling agent methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the signaling agent base object
 */
mrcp_signaling_agent_t* mrcp_client_signaling_agent_create(
					const mrcp_signaling_agent_method_set_t *agent_method_set,
					const mrcp_module_method_set_t *module_method_set,
					void *object,
					apr_pool_t *pool);

/**
 * Retrieve the object associated with signaling agent upon creation.
 * @param agent the agent to retrieve from
 * @return the object
 */
void* mrcp_client_signaling_agent_object_get(mrcp_signaling_agent_t *agent);

/**
 * Create signaling channel object.
 * @param agent the signaling agent (creator of the channel)
 * @param method_set the implemented set of signaling channel methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the signaling channel base object
 */
mrcp_signaling_channel_t* mrcp_client_signaling_channel_create(
									mrcp_signaling_agent_t *agent,
									const mrcp_signaling_channel_method_set_t *method_set, 
									void *object,
									apr_pool_t *pool);

/**
 * Retrieve the object associated with signaling channel upon creation.
 * @param channel the channel to retrieve from
 * @return the object
 */
void* mrcp_client_signaling_channel_object_get(mrcp_signaling_channel_t *channel);

/**
 * Retrieve the signaling agent.
 * @param channel the channel to retrieve from
 * @return the signaling agent
 */
mrcp_signaling_agent_t* mrcp_client_signaling_channel_agent_get(mrcp_signaling_channel_t *channel);

/**
 * Signal task message to client task.
 * @param agent the agent to signal message from
 * @param task_msg the message to signal
 */
mrcp_status_t mrcp_client_signaling_agent_signal(mrcp_signaling_agent_t *agent, apt_task_msg_t *task_msg);

/** Perform validity check for specified agent */
static APR_INLINE mrcp_status_t mrcp_client_signaling_agent_validity_check(mrcp_signaling_agent_t *agent)
{
	return mrcp_module_validity_check(&agent->module);
}

MRCP_END_EXTERN_C

#endif /*__MRCP_CLIENT_SIGNALING_AGENT_H__*/
