/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_TEXT_STREAM_H__
#define __APT_TEXT_STREAM_H__

/**
 * @file apt_text_stream.h
 * @brief Text Stream Routine
 */ 

#include <stdlib.h>
#include <stdio.h>
#include "apt.h"

APT_BEGIN_EXTERN_C

/** Named tokens */

/** Space */
#define APT_TOKEN_SP ' '
/** Carrige return */
#define APT_TOKEN_CR 0x0D
/** Line feed */
#define APT_TOKEN_LF 0x0A

typedef struct apt_text_stream_t apt_text_stream_t;

/** Text stream definition, used in message parsing and generation */
struct apt_text_stream_t {
	/** Actual message buffer */
	char      *buffer;
	/** Size of the buffer */
	apr_size_t size;

	/** Current position in the buffer */
	char      *pos;
};

/** 
 * Navigate through the lines of the text stream (message).
 * @param text_stream the text stream to navigate
 * @return the parsed line
 */
APT_DECLARE(char*) apt_read_line(apt_text_stream_t *text_stream);

/**
 * Navigate through the fields of the line.
 * @param line the line to navigate
 * @param separator the field separator
 * @param skip_spaces indicate whether spaces should be skipped
 * @return the parsed field
 */
APT_DECLARE(char*) apt_read_field(char **line, char separator, char skip_spaces);

/** Insert end of the line symbol(s). */
static APR_INLINE void apt_insert_end_of_line(apt_text_stream_t *text_stream)
{
	*text_stream->pos = APT_TOKEN_CR;
	text_stream->pos++;
	*text_stream->pos = APT_TOKEN_LF;
	text_stream->pos++;
}

/** Parse boolean-value. */
APT_DECLARE(apt_bool_t) apt_boolean_value_parse(const char *str, apt_bool_t *value);

/** Generate boolean-value. */
APT_DECLARE(apr_size_t) apt_boolean_value_generate(apt_bool_t value, char *str);

/** Parse size_t value. */
static APR_INLINE apr_size_t apt_size_value_parse(const char *str)
{
	return atol(str);
}

/** Generate apr_size_t value. */
static APR_INLINE apr_size_t apt_size_value_generate(apr_size_t value, char *str)
{
	int length = sprintf(str, "%"APR_SIZE_T_FMT, value);
	if(length < 0) {
		length = 0;
	}
	return length;
}

/** Parse float value. */
static APR_INLINE float apt_float_value_parse(const char *str)
{
	return (float)atof(str);
}

/** Generate float value. */
static APR_INLINE apr_size_t apt_float_value_generate(float value, char *str)
{
	int length = sprintf(str,"%.1f",value);
	if(length < 0) {
		length = 0;
	}
	return length;
}

/** Generate string value. */
static APR_INLINE apr_size_t apt_string_value_generate(const char *value, char *str)
{
	apr_size_t length = strlen(value);
	memcpy(str,value,length);
	return length;
}

/** Generate value plus the length (number of digits) of the value itself. */
APT_DECLARE(apr_size_t) apt_var_length_value_generate(apr_size_t *value, apr_size_t max_count, char *str);


typedef struct apt_name_value_pair_t apt_name_value_pair_t;
/* Generic name-value pair definition */
struct apt_name_value_pair_t {
	char *name;
	char *value;
};

/** Parse generic name-value pair */
APT_DECLARE(apt_bool_t) apt_name_value_pair_parse(apt_name_value_pair_t *pair, char *line);

/** Generate generic name-value pair */
APT_DECLARE(apr_size_t) apt_name_value_pair_generate(const apt_name_value_pair_t *pair, char *line);

/** Generate generic name-value pair */
APT_DECLARE(apr_size_t) apt_name_value_generate(const char *name, const char *value, char *line);

/** Generate only the name part ("name":) of the name-value pair */
APT_DECLARE(apr_size_t) apt_name_value_generate_name(const char *name, char *line);


APT_END_EXTERN_C

#endif /*__APT_TEXT_STREAM_H__*/
