/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __RTSP_START_LINE_H__
#define __RTSP_START_LINE_H__

/**
 * @file rtsp_start_line.h
 * @brief RTSP Start Line (request-line/status-line)
 */ 

#include <apr_strings.h>
#include <apr_pools.h>
#include "apt.h"
#include "apt_text_stream.h"

APT_BEGIN_EXTERN_C

/** Protocol version */
typedef enum {
	/** Unknown version */
	RTSP_VERSION_UNKNOWN = 0, 
	/** RTSP 1.0 */
	RTSP_VERSION_1 = 1, 
} rtsp_version_t;

/* RTSP message types */
typedef enum {
	RTSP_MESSAGE_TYPE_UNKNOWN,
	RTSP_MESSAGE_TYPE_REQUEST,
	RTSP_MESSAGE_TYPE_RESPONSE
} rtsp_message_type;

/** RTSP methods */
typedef enum{
	RTSP_METHOD_SETUP,
	RTSP_METHOD_ANNOUNCE,
	RTSP_METHOD_TEARDOWN,
	RTSP_METHOD_DESCRIBE,

	RTSP_METHOD_COUNT,
	RTSP_METHOD_UNKNOWN = RTSP_METHOD_COUNT
} rtsp_method_id;

/* Status codes */
typedef enum {
	RTSP_STATUS_CODE_UNKNOWN                   = 0,
	/* success codes (2xx) */
	RTSP_STATUS_CODE_OK                        = 200,
	RTSP_STATUS_CODE_CREATED                   = 201,
	/* failure codec (4xx) */
	RTSP_STATUS_CODE_BAD_REQUEST               = 400,
	RTSP_STATUS_CODE_UNAUTHORIZED              = 401,
	RTSP_STATUS_CODE_NOT_FOUND                 = 404,
	RTSP_STATUS_CODE_METHOD_NOT_ALLOWED        = 405,

	RTSP_STATUS_CODE_INTERNAL_SERVER_ERROR     = 500,
	RTSP_STATUS_CODE_NOT_IMPLEMENTED           = 501,
} rtsp_status_code_t;

/** Reason phrases (rtsp_status_code_t) */
#define RTSP_REASON_PHRASE_OK                    "OK"
#define RTSP_REASON_PHRASE_CREATED               "Created"
#define RTSP_REASON_PHRASE_BAD_REQUEST           "Bad Request"
#define RTSP_REASON_PHRASE_UNAUTHORIZED          "Unauthorized"
#define RTSP_REASON_PHRASE_NOT_FOUND             "Not Found"
#define RTSP_REASON_PHRASE_METHOD_NOT_ALLOWED    "Method not Allowed"
#define RTSP_REASON_PHRASE_INTERNAL_SERVER_ERROR "Internal Server Error"
#define RTSP_REASON_PHRASE_NOT_IMPLEMENTED       "Not Implemented"

typedef struct rtsp_request_line_t rtsp_request_line_t;
/** Request-line of RTSP request */
struct rtsp_request_line_t {
	/** method name */
	const char      *method_name;
	/** method id */
	rtsp_method_id   method_id;
	/** RTSP URL */
	char            *url;
	/** resource named parsed from RTSP URL */
	const char      *resource_name;
	/** version of protocol in use */
	rtsp_version_t   version;
};

static APR_INLINE void rtsp_request_line_init(rtsp_request_line_t *request_line)
{
	request_line->method_name = NULL;
	request_line->method_id = RTSP_METHOD_UNKNOWN;
	request_line->url = NULL;
	request_line->resource_name = NULL;
	request_line->version = RTSP_VERSION_1;
}

typedef struct rtsp_status_line_t rtsp_status_line_t;
/** Status-line of RTSP response */
struct rtsp_status_line_t {
	/** version of protocol in use */
	rtsp_version_t       version;
	/** success or failure or other status of the request */
	rtsp_status_code_t   status_code;
	/** reason phrase */
	const char          *reason;
};

static APR_INLINE void rtsp_status_line_init(rtsp_status_line_t *status_line)
{
	status_line->version = RTSP_VERSION_1;
	status_line->status_code = RTSP_STATUS_CODE_OK;
	status_line->reason = NULL;
}

typedef struct rtsp_start_line_t rtsp_start_line_t;
/** RTSP start-line */
struct rtsp_start_line_t {
	/** RTSP message type */
	rtsp_message_type       message_type;
	/** RTSP start-line */
	union {
		rtsp_request_line_t request_line;
		rtsp_status_line_t  status_line;
	} common;
};

/** Initialize RTSP start-line */
static APR_INLINE void rtsp_start_line_init(rtsp_start_line_t *start_line, rtsp_message_type message_type)
{
	start_line->message_type = message_type;
	if(message_type == RTSP_MESSAGE_TYPE_REQUEST) {
		rtsp_request_line_init(&start_line->common.request_line);
	}
	else if(message_type == RTSP_MESSAGE_TYPE_RESPONSE) {
		rtsp_status_line_init(&start_line->common.status_line);
	}
}

/** Parse RTSP start-line */
apt_bool_t rtsp_start_line_parse(rtsp_start_line_t *start_line, apt_text_stream_t *text_stream, apr_pool_t *pool);

/** Generate RTSP start-line */
apt_bool_t rtsp_start_line_generate(rtsp_start_line_t *start_line, apt_text_stream_t *text_stream);


APT_END_EXTERN_C

#endif /*__RTSP_START_LINE_H__*/
