/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include <assert.h>
#include <apr_hash.h>
#include "mrcp_server.h"
#include "mrcp_server_signaling_agent.h"
#include "mrcp_server_proto_agent.h"
#include "mrcp_media_agent.h"
#include "mrcp_server_session.h"
#include "apt_consumer_task.h"
#include "mrcp_engine.h"

struct mrcp_server_t {
	/* base */
	mrcp_engine_t                engine;

	mrcp_server_module_loader_t *module_loader;

	/* container of mrcp server resources */
	mrcp_resource_container_t   *resource_container;

	/* signaling agent registered to mrcp server */
	mrcp_signaling_agent_t      *signaling_agent;
	/* mrcp protocol agent registered to mrcp server */
	mrcp_proto_agent_t          *proto_agent;
	/* mrcp media agent registered to mrcp server */
	mrcp_media_agent_t          *media_agent;

	/* memory pool */
	apr_pool_t                  *pool;
	/* table of mrcp sessions */
	apr_hash_t                  *session_table;
};

typedef enum {
	MODULE_TYPE_SIGNALING_AGENT,
	MODULE_TYPE_MRCP_AGENT,
	MODULE_TYPE_MEDIA_AGENT,
	MODULE_TYPE_PLUGIN
} mrcp_module_type;


static mrcp_status_t mrcp_server_on_module_open(mrcp_module_t *module);
static mrcp_status_t mrcp_server_on_module_close(mrcp_module_t *module);

static const mrcp_module_event_set_t module_event_set = {
	mrcp_server_on_module_open,
	mrcp_server_on_module_close,
};


static mrcp_session_t* mrcp_server_on_session_create(mrcp_signaling_agent_t *agent, mrcp_session_id *session_id, apr_pool_t **pool);
static mrcp_status_t mrcp_server_on_signaling_channel_attach(mrcp_session_t *session, mrcp_signaling_channel_t *channel);

static const mrcp_signaling_agent_event_set_t signaling_agent_event_set = {
	mrcp_server_on_session_create,
	mrcp_server_on_signaling_channel_attach
};

static mrcp_status_t mrcp_server_on_signaling_channel_offer(mrcp_signaling_channel_t *channel);
static mrcp_status_t mrcp_server_on_signaling_channel_terminate(mrcp_signaling_channel_t *channel);

static const mrcp_signaling_channel_event_set_t signaling_channel_event_set = {
	mrcp_server_on_signaling_channel_offer,
	NULL, /*on_answer*/
	mrcp_server_on_signaling_channel_terminate
};


static mrcp_status_t mrcp_server_on_client_disconnect(mrcp_connection_t *connection);
static mrcp_status_t mrcp_server_on_message_receive(mrcp_connection_t *connection, mrcp_message_t *mrcp_message);

static const mrcp_connection_event_set_t connection_event_set = {
	mrcp_server_on_client_disconnect,
	mrcp_server_on_message_receive
};


static APR_INLINE mrcp_server_t * mrcp_server_get(mrcp_module_t *module)
{
	return (mrcp_server_t *)module->engine;
}

static APR_INLINE mrcp_server_resource_t* mrcp_server_resource_get(mrcp_server_t *mrcp_server, mrcp_resource_id resource_id)
{
	if(resource_id >= mrcp_server->resource_container->resource_count) {
		return NULL;
	}
	return (mrcp_server_resource_t*)mrcp_server->resource_container->resource_array[resource_id];
}

static APR_INLINE mrcp_session_t* mrcp_server_session_find(mrcp_server_t *mrcp_server, const mrcp_session_id *session_id)
{
	return apr_hash_get(mrcp_server->session_table,session_id->hex_str,session_id->length);
}


/* adds mrcp session, returns created mrcp session instance */
static mrcp_session_t* mrcp_server_session_add(mrcp_server_t *mrcp_server, mrcp_session_id *session_id)
{
	mrcp_session_t* session;
	session	= mrcp_server_session_create(session_id);
	if(!session) {
		apt_log(APT_PRIO_WARNING,"Failed to Create MRCP Session\n");
		return NULL;
	}
	apr_hash_set(mrcp_server->session_table,session->id.hex_str,session->id.length,session);

	return session;
}

/* removes mrcp session */
static mrcp_status_t mrcp_server_session_remove(mrcp_server_t *mrcp_server, mrcp_session_t *session)
{
	if(!session) {
		apt_log(APT_PRIO_WARNING,"Failed to Remove NULL Session\n");
		return MRCP_STATUS_FAILURE;
	}

	if(apr_hash_get(mrcp_server->session_table,session->id.hex_str,session->id.length) == NULL) {
		apt_log(APT_PRIO_WARNING,"Failed to Remove Session: no such session <%s>\n",session->id.hex_str);
		return MRCP_STATUS_FAILURE;
	}
	apr_hash_set(mrcp_server->session_table,session->id.hex_str,session->id.length,NULL);

	return mrcp_server_session_destroy(session);
}

static mrcp_status_t mrcp_server_on_client_disconnect(mrcp_connection_t *connection)
{
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_on_message_receive(mrcp_connection_t *connection, mrcp_message_t *mrcp_message)
{
	mrcp_server_t *mrcp_server = mrcp_server_get(&connection->agent->module);
	mrcp_session_t *session;
	mrcp_message_t response_message;
	mrcp_message_init(&response_message,mrcp_message->pool);
	mrcp_response_init_by_request(&response_message,mrcp_message);
	
	apt_log(APT_PRIO_INFO,"Process Message <%s@%s>\n",mrcp_message->channel_id.session_id.hex_str,mrcp_message->channel_id.resource_name);
	session = mrcp_server_session_find(mrcp_server,&mrcp_message->channel_id.session_id);
	if(session) {
		mrcp_server_message_process(session,mrcp_message,&response_message);
	}
	else {
		apt_log(APT_PRIO_WARNING,"Failed to Process Message: no such session <%s>\n",mrcp_message->channel_id.session_id.hex_str);
		response_message.start_line.status_code = MRCP_STATUS_CODE_NOT_FOUND;
		return MRCP_STATUS_FAILURE;
	}

	return connection->method_set->send(connection,&response_message);
}

static mrcp_session_t* mrcp_server_on_session_create(mrcp_signaling_agent_t *agent, mrcp_session_id *session_id, apr_pool_t **pool)
{
	mrcp_server_t *mrcp_server = mrcp_server_get(&agent->module);
	mrcp_session_t *session = mrcp_server_session_add(mrcp_server,session_id);
	if(!session) {
		return NULL;
	}
	if(pool) {
		*pool = session->pool;
	}
	return session;
}

static mrcp_status_t mrcp_server_on_signaling_channel_attach(mrcp_session_t *session, mrcp_signaling_channel_t *channel)
{
	session->signaling_channel = channel;
	channel->session = session;
	return MRCP_STATUS_SUCCESS;
}


static mrcp_status_t mrcp_server_control_media_offer_process(
								mrcp_server_t *mrcp_server, 
								mrcp_session_t *session, 
								mrcp_media_control_t *local_media, 
								mrcp_media_control_t *remote_media)
{
	mrcp_server_channel_t *channel = mrcp_server_session_channel_get(session,local_media->base.id);
	
	if(remote_media->base.state == MRCP_MEDIA_ENABLED) {
		if(local_media->base.state == MRCP_MEDIA_ENABLED) {
			/* nothing to do (update) */
		}
		else {
			mrcp_resource_id resource_id;
			mrcp_server_resource_t *resource;

			resource_id = mrcp_resource_id_find(mrcp_server->resource_container,local_media->resource_name);
			resource = mrcp_server_resource_get(mrcp_server,resource_id);
			if(!resource || !resource->plugin) {
				apt_log(APT_PRIO_WARNING,"Failed to Add Channel: no such resource <%s@%s>\n",session->id.hex_str,local_media->resource_name);
				return MRCP_STATUS_FAILURE;
			}
			
			if(channel && channel->resource && channel->resource->base.id == resource_id) {
				/* just going to reuse the same channel */
			}
			else {
				/* create new channel */
				channel = mrcp_server_channel_create(session,resource);
				if(!channel) {
					apt_log(APT_PRIO_WARNING,"Failed to Create Channel <%s@%s>\n",session->id.hex_str,local_media->resource_name);
					return MRCP_STATUS_FAILURE;
				}

				mrcp_server_session_channel_set(session,channel,local_media->base.id);

				channel->connection = mrcp_server->proto_agent->agent_method_set->connect(
																		mrcp_server->proto_agent,
																		channel->session->signaling_channel,
																		remote_media->base.ip);
			}
			local_media->base.state = MRCP_MEDIA_ENABLED;
		}
	}
	else if(remote_media->base.state == MRCP_MEDIA_DISABLED) {
		if(local_media->base.state == MRCP_MEDIA_DISABLED) {
			/* nothing to do (update) */
		}
		else {
			if(channel) {
				mrcp_server_channel_destroy(channel);
			}
			local_media->base.state = MRCP_MEDIA_DISABLED;
		}
	}

	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_audio_media_offer_process(
								mrcp_server_t *mrcp_server, 
								mrcp_session_t *session, 
								mrcp_media_audio_t *local_media, 
								mrcp_media_audio_t *remote_media)
{
	mrcp_audio_channel_t *channel = mrcp_server_session_channel_get(session,local_media->base.id);

	if(remote_media->base.state == MRCP_MEDIA_ENABLED) {
		if(local_media->base.state == MRCP_MEDIA_ENABLED) {
			/* nothing to do (update) */
		}
		else {
			if(!channel) {
				/* create new channel */
				channel = mrcp_audio_channel_create(
										MRCP_AUDIO_CHANNEL_PASSIVE,
										local_media,
										remote_media,
										session->pool);
				if(mrcp_server->media_agent->method_set->channel_create(
										mrcp_server->media_agent,
										channel,
										session->pool) != MRCP_STATUS_SUCCESS) {
					apt_log(APT_PRIO_WARNING,"Failed to Create Audio Channel\n");
					return MRCP_STATUS_FAILURE;
				}
				
				mrcp_server_session_channel_set(session,channel,local_media->base.id);
			}
			local_media->base.state = MRCP_MEDIA_ENABLED;
		}
	}
	else if(remote_media->base.state == MRCP_MEDIA_DISABLED) {
		if(local_media->base.state == MRCP_MEDIA_DISABLED) {
			/* nothing to do (update) */
		}
		else {
			local_media->base.state = MRCP_MEDIA_DISABLED;
		}
	}

	if(channel) {
		channel->method_set->modify(channel,MRCP_AUDIO_REQUEST_RECEIVE_OFFER);
	}
	return MRCP_STATUS_SUCCESS;
}


static mrcp_status_t mrcp_server_on_signaling_channel_offer(mrcp_signaling_channel_t *channel)
{
	mrcp_signaling_agent_t *agent = channel->agent;
	mrcp_server_t *mrcp_server = mrcp_server_get(&agent->module);
	mrcp_media_audio_t *local_audio_media;
	mrcp_media_audio_t *remote_audio_media;
	mrcp_media_control_t *local_control_media;
	mrcp_media_control_t *remote_control_media;
	size_t i;
	size_t count = 0;

	count = channel->local_descriptor.media_count;
	if(count > channel->remote_descriptor.media_count) {
		apt_log(APT_PRIO_WARNING,"Local Media Count [%d] > Remote Media Count [%d]\n",
			channel->local_descriptor.media_count,
			channel->remote_descriptor.media_count);
		count = channel->remote_descriptor.media_count;
	}

	/* remove/update existing media */
	for(i=0; i<count; i++) {
		if(!channel->local_descriptor.media[i] || !channel->remote_descriptor.media[i]) {
			continue;
		}
		if(channel->local_descriptor.media[i]->type != channel->remote_descriptor.media[i]->type) {
			continue;
		}
		switch(channel->remote_descriptor.media[i]->type) {
			case MRCP_MEDIA_TYPE_CONTROL:
				local_control_media = (mrcp_media_control_t*)channel->local_descriptor.media[i];
				remote_control_media = (mrcp_media_control_t*)channel->remote_descriptor.media[i];
				mrcp_server_control_media_offer_process(mrcp_server,channel->session,local_control_media,remote_control_media);
				break;
			case MRCP_MEDIA_TYPE_AUDIO:
				local_audio_media = (mrcp_media_audio_t*)channel->local_descriptor.media[i];
				remote_audio_media = (mrcp_media_audio_t*)channel->remote_descriptor.media[i];
				mrcp_server_audio_media_offer_process(mrcp_server,channel->session,local_audio_media,remote_audio_media);
				break;
		}
	}

	/* add new media */
	for(; i<channel->remote_descriptor.media_count; i++) {
		if(!channel->remote_descriptor.media[i]) {
			continue;
		}
		switch(channel->remote_descriptor.media[i]->type) {
			case MRCP_MEDIA_TYPE_CONTROL:
				remote_control_media = (mrcp_media_control_t*)channel->remote_descriptor.media[i];
				local_control_media = mrcp_control_media_create(channel->session->pool);
				*local_control_media = *remote_control_media;
				local_control_media->base.ip = channel->local_descriptor.ip;
				local_control_media->base.port = mrcp_server->signaling_agent->mrcp_port;
				local_control_media->setup_type = MRCP_CONTROL_SETUP_TYPE_PASSIVE;
				local_control_media->session_id = channel->session->id;
				local_control_media->base.state = MRCP_MEDIA_DISABLED;
				mrcp_descriptor_media_add(&channel->local_descriptor,&local_control_media->base);
				
				if(local_control_media->base.id != remote_control_media->base.id) {
					continue;
				}
				
				mrcp_server_control_media_offer_process(mrcp_server,channel->session,local_control_media,remote_control_media);
				break;
			case MRCP_MEDIA_TYPE_AUDIO:
				remote_audio_media = (mrcp_media_audio_t*)channel->remote_descriptor.media[i];
				local_audio_media = mrcp_audio_media_create(channel->session->pool);
				local_audio_media->mid = remote_audio_media->mid;
				if(remote_audio_media->ptime) {
					local_audio_media->ptime = remote_audio_media->ptime;
				}
				mrcp_descriptor_media_add(&channel->local_descriptor,&local_audio_media->base);

				if(local_audio_media->base.id != remote_audio_media->base.id) {
					continue;
				}

				mrcp_server_audio_media_offer_process(mrcp_server,channel->session,local_audio_media,remote_audio_media);
				break;
		}
	}

	mrcp_server_channel_assocciations_set(channel->session);

	return channel->method_set->answer(channel);
}

static mrcp_status_t mrcp_server_on_signaling_channel_terminate(mrcp_signaling_channel_t *channel)
{
	mrcp_signaling_agent_t *agent = channel->agent;
	mrcp_server_t *mrcp_server = mrcp_server_get(&agent->module);
	return mrcp_server_session_remove(mrcp_server,channel->session);
}

#if 0
static void mrcp_active_session_destructor(mrcp_server_t *mrcp_server)
{
	apr_hash_index_t *hi;
	void *val;
	mrcp_session_t* session;
	for(hi=apr_hash_first(NULL,mrcp_server->session_table);hi;hi = apr_hash_next(hi)) {
		apr_hash_this(hi, NULL, NULL, &val);
		if(val) {
			session = val;
			apt_log(APT_PRIO_NOTICE,"Delete Session <%s>\n",session->id.hex_str);
			mrcp_server_session_destroy(session);
		}
	}
}
#endif

static mrcp_status_t mrcp_server_msg_handler(void *object, apt_task_msg_t *task_msg)
{
	mrcp_server_t *mrcp_server = object;
	mrcp_module_t *module = task_msg->msg_handle;
	if(!mrcp_server) {
		return MRCP_STATUS_FAILURE;
	}
	switch(module->type) {
		case MODULE_TYPE_SIGNALING_AGENT:
		case MODULE_TYPE_MRCP_AGENT:
		case MODULE_TYPE_MEDIA_AGENT:
			module->method_set->process(module,task_msg);
			break;
		case MODULE_TYPE_PLUGIN:
		{
			mrcp_server_plugin_msg_t *plugin_msg = (mrcp_server_plugin_msg_t *)task_msg->data;
			if(!plugin_msg || !plugin_msg->channel || !plugin_msg->mrcp_message) {
				return MRCP_STATUS_FAILURE;
			}

			if(plugin_msg->mrcp_message->start_line.message_type == MRCP_MESSAGE_TYPE_EVENT) {
				mrcp_server_channel_event_handler(plugin_msg->channel,plugin_msg->mrcp_message);
			}
			else if(plugin_msg->mrcp_message->start_line.message_type == MRCP_MESSAGE_TYPE_RESPONSE) {
				mrcp_server_channel_response_handler(plugin_msg->channel,plugin_msg->mrcp_message);
			}
			break;
		}
	}
	return MRCP_STATUS_SUCCESS;
}

mrcp_status_t mrcp_server_channel_signal(mrcp_server_channel_t *channel, mrcp_message_t *message)
{
	mrcp_server_plugin_t *plugin = channel->resource->plugin;
	mrcp_server_plugin_msg_t *plugin_msg;
	apt_task_msg_t *task_msg = apt_task_msg_acquire(plugin->msg_pool);
	plugin_msg = (mrcp_server_plugin_msg_t*)task_msg->data;
	plugin_msg->channel = channel;
	plugin_msg->mrcp_message = message;
	
	return plugin->module.signal(&plugin->module,task_msg);
}

static mrcp_status_t mrcp_server_resource_setup(mrcp_server_t *mrcp_server, mrcp_server_resource_t *resource, apr_pool_t *pool)
{
	mrcp_resource_id resource_id;
	const char *resource_name;
	if(!resource) {
		apt_log(APT_PRIO_WARNING,"Null Server Resource Found\n");
		return MRCP_STATUS_FAILURE;
	}

	resource_id = resource->base.id;
	resource_name = resource->base.name;

	if(!resource->plugin) {
		apt_log(APT_PRIO_NOTICE,"Register Plugin %lu - [%s] missing\n",resource_id,resource_name);
		return MRCP_STATUS_FAILURE;
	}
	if(resource->plugin->method_count != resource->base.method_count) {
		apt_log(APT_PRIO_WARNING,"Failed to Register Plugin: invalid method_count=%lu\n",resource->plugin->method_count);
		return MRCP_STATUS_FAILURE;
	}

	apt_log(APT_PRIO_NOTICE,"Register Plugin %lu - [%s]\n",resource_id,resource_name);
	resource->plugin->msg_pool = apt_task_msg_pool_create_dynamic(sizeof(mrcp_server_plugin_msg_t),pool);
	resource->plugin->module.type = MODULE_TYPE_PLUGIN;
	resource->plugin->module.event_set = &module_event_set;
	mrcp_engine_module_register(&mrcp_server->engine,&resource->plugin->module);
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_resources_setup(mrcp_server_t *mrcp_server)
{
	size_t i;
	apt_log(APT_PRIO_INFO,"Setup MRCP Server Resources\n");
	mrcp_server->resource_container = mrcp_server->module_loader->resource_container_create(mrcp_server->module_loader,mrcp_server->pool);
	if(!mrcp_server->resource_container) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Server Resources\n");
		return MRCP_STATUS_FAILURE;
	}

	for(i=0; i<mrcp_server->resource_container->resource_count; i++) {
		if(mrcp_server_resource_setup(mrcp_server,(mrcp_server_resource_t*)mrcp_server->resource_container->resource_array[i],mrcp_server->pool) != MRCP_STATUS_SUCCESS) {
			mrcp_server->resource_container->resource_array[i] = NULL;
		}
	}

	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_signaling_agent_setup(mrcp_server_t *mrcp_server)
{
	mrcp_signaling_agent_t *agent;
	apt_log(APT_PRIO_INFO,"Setup MRCP Signaling Agent\n");
	agent = mrcp_server->module_loader->signaling_agent_create(mrcp_server->module_loader,mrcp_server->pool);
	if(!agent) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Signaling Agent\n");
		return MRCP_STATUS_FAILURE;
	}
	agent->agent_event_set = &signaling_agent_event_set;
	agent->channel_event_set = &signaling_channel_event_set;
	if(mrcp_server_signaling_agent_validity_check(agent) != MRCP_STATUS_SUCCESS) {
		apt_log(APT_PRIO_DEBUG,"Failed to Pass Validity Check for Signaling Agent\n");
		return MRCP_STATUS_FAILURE;
	}
	
	agent->module.type = MODULE_TYPE_SIGNALING_AGENT;
	agent->module.event_set = &module_event_set;
	mrcp_engine_module_register(&mrcp_server->engine,&agent->module);

	mrcp_server->signaling_agent = agent;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_proto_agent_setup(mrcp_server_t *mrcp_server)
{
	mrcp_proto_agent_t *agent;
	apt_log(APT_PRIO_INFO,"Setup MRCP Message Agent\n");
	agent = mrcp_server->module_loader->proto_agent_create(mrcp_server->module_loader,mrcp_server->signaling_agent,mrcp_server->pool);
	if(!agent) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Message Agent\n");
		return MRCP_STATUS_FAILURE;
	}
	agent->connection_event_set = &connection_event_set;
	if(mrcp_server_proto_agent_validity_check(agent) != MRCP_STATUS_SUCCESS) {
		apt_log(APT_PRIO_DEBUG,"Failed to Pass Validity Check for Message Agent\n");
		return MRCP_STATUS_FAILURE;
	}

	agent->module.type = MODULE_TYPE_MRCP_AGENT;
	agent->module.event_set = &module_event_set;
	mrcp_engine_module_register(&mrcp_server->engine,&agent->module);

	agent->resource_container = mrcp_server->resource_container;

	mrcp_server->proto_agent = agent;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_media_agent_setup(mrcp_server_t *mrcp_server)
{
	mrcp_media_agent_t *agent;
	apt_log(APT_PRIO_INFO,"Setup Media Agent\n");
	agent = mrcp_server->module_loader->media_agent_create(mrcp_server->module_loader,mrcp_server->pool);
	if(!agent) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Media Agent\n");
		return MRCP_STATUS_FAILURE;
	}
	if(mrcp_media_agent_validity_check(agent) != MRCP_STATUS_SUCCESS) {
		apt_log(APT_PRIO_DEBUG,"Failed to Pass Validity Check for Media Agent\n");
		return MRCP_STATUS_FAILURE;
	}

	agent->module.type = MODULE_TYPE_MEDIA_AGENT;
	agent->module.event_set = &module_event_set;
	mrcp_engine_module_register(&mrcp_server->engine,&agent->module);

	mrcp_server->media_agent = agent;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_capabilities_setup(mrcp_server_t *mrcp_server)
{
	size_t i;
	mrcp_descriptor_t *descripor;
	mrcp_media_control_t *control_media;
	mrcp_resource_t *resource;
	if(!mrcp_server->signaling_agent) {
		return MRCP_STATUS_FAILURE;
	}
	descripor = &mrcp_server->signaling_agent->capabilities;
	for(i=0; i<mrcp_server->resource_container->resource_count; i++) {
		resource = mrcp_server->resource_container->resource_array[i];
		if(!resource) {
			continue;
		}

		control_media = mrcp_control_media_create(mrcp_server->pool);
		control_media->proto = MRCP_PROTO_TCP;
		control_media->resource_name = resource->name;
	}
	return 	MRCP_STATUS_SUCCESS;
}



static void mrcp_server_on_start_in_progress(void *data)
{
	mrcp_server_t *mrcp_server = data;
	assert(mrcp_server);
	apt_log(APT_PRIO_INFO,"Initialize MRCP Server\n");

	if(mrcp_server_resources_setup(mrcp_server) != MRCP_STATUS_SUCCESS) {
		apt_consumer_task_running_set(mrcp_server->engine.consumer_task,FALSE);
		apt_log(APT_PRIO_WARNING,"Failed to Setup Server Resources\n");
		return;
	}
	if(mrcp_server_signaling_agent_setup(mrcp_server) != MRCP_STATUS_SUCCESS) {
		apt_consumer_task_running_set(mrcp_server->engine.consumer_task,FALSE);
		apt_log(APT_PRIO_WARNING,"Failed to Setup Signaling Agent\n");
		return;
	}
	if(mrcp_server_proto_agent_setup(mrcp_server) != MRCP_STATUS_SUCCESS) {
		apt_consumer_task_running_set(mrcp_server->engine.consumer_task,FALSE);
		apt_log(APT_PRIO_WARNING,"Failed to Setup MRCP Proto Agent\n");
		return;
	}
	if(mrcp_server_media_agent_setup(mrcp_server) != MRCP_STATUS_SUCCESS) {
		apt_consumer_task_running_set(mrcp_server->engine.consumer_task,FALSE);
		apt_log(APT_PRIO_WARNING,"Failed to Setup Media Agent\n");
		return;
	}

	mrcp_server_capabilities_setup(mrcp_server);

	mrcp_server->session_table = apr_hash_make(mrcp_server->pool);

	mrcp_engine_open(&mrcp_server->engine);
}

static void mrcp_server_on_terminate_in_progress(void *data)
{
	mrcp_server_t *mrcp_server = data;
	assert(mrcp_server);

	apt_consumer_task_running_set(mrcp_server->engine.consumer_task,TRUE);

	if(mrcp_engine_close(&mrcp_server->engine) == MODULE_STATE_CLOSED) {
		apt_consumer_task_running_set(mrcp_server->engine.consumer_task,FALSE);
	}
}

static void mrcp_server_on_terminate_complete(void *data)
{
	mrcp_server_t *mrcp_server = data;
	assert(mrcp_server);
	apt_log(APT_PRIO_INFO,"Destroy MRCP Server\n");

	if(mrcp_server->resource_container) {
		size_t i;
		mrcp_server_resource_t* resource;
		for(i=0; i<mrcp_server->resource_container->resource_count; i++) {
			resource = mrcp_server_resource_get(mrcp_server,i);
			if(resource && resource->plugin && resource->plugin->msg_pool) {
				apt_task_msg_pool_destroy(resource->plugin->msg_pool);
				resource->plugin->msg_pool = NULL;
			}
		}

		mrcp_resource_container_destroy(mrcp_server->resource_container);
		mrcp_server->resource_container = NULL;
	}

	mrcp_engine_destroy(&mrcp_server->engine);
}

static mrcp_status_t mrcp_server_on_module_open(mrcp_module_t *module)
{
	module->state = MODULE_STATE_OPENED;
	return MRCP_STATUS_SUCCESS;
}

static mrcp_status_t mrcp_server_on_module_close(mrcp_module_t *module)
{
	module->state = MODULE_STATE_CLOSED;
	if(mrcp_engine_is_closed(module->engine) == MRCP_STATUS_SUCCESS) {
		apt_consumer_task_running_set(module->engine->consumer_task,FALSE);
	}
	return MRCP_STATUS_SUCCESS;
}


/* starts message processing loop */
mrcp_server_t* mrcp_server_start(mrcp_server_module_loader_t *module_loader)
{
	mrcp_server_t* mrcp_server;
	apt_task_t *task;
	apr_pool_t *pool;
	
	if(!module_loader || !module_loader->signaling_agent_create || !module_loader->proto_agent_create || !module_loader->resource_container_create) {
		apt_log(APT_PRIO_DEBUG,"Invalid Module Loader\n");
		return NULL;
	}

	if(apr_pool_create(&pool,NULL) != APR_SUCCESS) {
		return NULL;
	}

	apt_log(APT_PRIO_NOTICE,"Starting MRCP Server\n");
	mrcp_server = apr_palloc(pool,sizeof(mrcp_server_t));
	mrcp_server->pool = pool;
	mrcp_server->resource_container = NULL;
	mrcp_server->signaling_agent = NULL;
	mrcp_server->proto_agent = NULL;
	mrcp_server->session_table = NULL;
	mrcp_server->module_loader = module_loader;

	mrcp_engine_init(&mrcp_server->engine);
	mrcp_server->engine.consumer_task = apt_consumer_task_create(mrcp_server, mrcp_server_msg_handler, pool);
	if(!mrcp_server->engine.consumer_task) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Server Consumer Task\n");
		apr_pool_destroy(pool);
		return NULL;
	}
	
	task = apt_consumer_task_get(mrcp_server->engine.consumer_task);
	apt_task_event_handler_set(task,TASK_STATE_START_IN_PROGRESS,mrcp_server,mrcp_server_on_start_in_progress);
	apt_task_event_handler_set(task,TASK_STATE_TERMINATE_IN_PROGRESS,mrcp_server,mrcp_server_on_terminate_in_progress);
	apt_task_event_handler_set(task,TASK_STATE_TERMINATE_COMPLETED,mrcp_server,mrcp_server_on_terminate_complete);
	apt_consumer_task_start(mrcp_server->engine.consumer_task);
	return mrcp_server;
}

/* terminates message processing loop */
mrcp_status_t mrcp_server_shutdown(mrcp_server_t* mrcp_server)
{
	assert(mrcp_server);
	apt_log(APT_PRIO_NOTICE,"Shutting Down MRCP Server\n");
	if(apt_consumer_task_terminate(mrcp_server->engine.consumer_task) != TRUE) {
		apt_log(APT_PRIO_DEBUG,"Failed to Terminate Server Consumer Task\n");
		return MRCP_STATUS_FAILURE;
	}
	apt_consumer_task_destroy(mrcp_server->engine.consumer_task);
	mrcp_server->engine.consumer_task = NULL;

	apr_pool_destroy(mrcp_server->pool);
	return MRCP_STATUS_SUCCESS;
}
