/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_SERVER_SIGNALING_AGENT_H__
#define __MRCP_SERVER_SIGNALING_AGENT_H__

/**
 * @file mrcp_server_signaling_agent.h
 * @brief MRCP Server Signaling Agent Definition.
 */ 

#include "mrcp_server_types.h"
#include "mrcp_module.h"
#include "mrcp_signaling_channel.h"

MRCP_BEGIN_EXTERN_C

typedef struct mrcp_signaling_agent_method_set_t mrcp_signaling_agent_method_set_t;
typedef struct mrcp_signaling_agent_event_set_t mrcp_signaling_agent_event_set_t;


/** Signaling agent base */
struct mrcp_signaling_agent_t {
	/** Base module */
	mrcp_module_t                             module;

	/** Set of signaling agent virtual events */
	const mrcp_signaling_agent_event_set_t   *agent_event_set;
	/** Set of signaling channel virtual events */
	const mrcp_signaling_channel_event_set_t *channel_event_set;

	/** Actual data set by inherited object */
	void                                     *object;

	/** Agent capabilities */
	mrcp_descriptor_t                         capabilities;
	/** Server SIP port */
	unsigned short                            sip_port;
	/** Server MRCP port */
	unsigned short                            mrcp_port;

	/** SIP Contact string */
	char                                     *sip_contact_str;

	/** Resource location used as SIP username or RTSP URI prefix */
	char                                     *resource_location;
};


/** Set of signaling agent event handlers */
struct mrcp_signaling_agent_event_set_t {
	/** Create new session */
	mrcp_session_t* (*on_session_create)(mrcp_signaling_agent_t *agent, mrcp_session_id *session_id, apr_pool_t **pool);
	/** Attach signaling channel to session */
	mrcp_status_t (*on_channel_attach)(mrcp_session_t *session, mrcp_signaling_channel_t *channel);
};


/**
 * Create signaling agent object.
 * @param method_set the implemented set of signaling agent methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the signaling agent base object
 */
mrcp_signaling_agent_t* mrcp_server_signaling_agent_create(
							const mrcp_module_method_set_t *method_set,
							void *object,
							apr_pool_t *pool);

/**
 * Create signaling channel object.
 * @param agent the signaling agent (creator of the channel)
 * @param method_set the implemented set of signaling channel methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the signaling channel base object
 */
mrcp_signaling_channel_t* mrcp_server_signaling_channel_create(
									mrcp_signaling_agent_t *agent,
									const mrcp_signaling_channel_method_set_t *method_set, 
									void *object,
									apr_pool_t *pool);


/** Perfom validity check for specified agent */
static APR_INLINE mrcp_status_t mrcp_server_signaling_agent_validity_check(mrcp_signaling_agent_t *agent)
{
	return mrcp_module_validity_check(&agent->module);
}

MRCP_END_EXTERN_C

#endif /*__MRCP_SERVER_SIGNALING_AGENT_H__*/
