/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_synthesizer.h"
#include "apt_string_table.h"
#include "mrcp_header_base.h"

/* string table of mrcp synthesizer methods (mrcp_synthesizer_method_id) */
static const apt_string_table_item_t synthesizer_method_string_table[] = {
	{"SET-PARAMS",       10,10},
	{"GET-PARAMS",       10,0},
	{"SPEAK",             5,1},
	{"STOP",              4,1},
	{"PAUSE",             5,0},
	{"RESUME",            6,0},
	{"BARGE-IN-OCCURRED",17,0},
	{"CONTROL",           7,0}
};

/* string table of mrcp synthesizer events (mrcp_synthesizer_event_id) */
static const apt_string_table_item_t synthesizer_event_string_table[] = {
	{"SPEECH-MARKER", 13,3},
	{"SPEAK-COMPLETE",14,3}
};

/* string table of mrcp synthesizer headers (mrcp_synthesizer_header_id) */
static const apt_string_table_item_t synthesizer_header_string_table[] = {
	{"Jump-Size",            9,0},
	{"Kill-On-Barge-In",    16,0},
	{"Speaker=Profile",     15,6},
	{"Completion-Cause",    16,16},
	{"Completion-Reason",   17,11},
	{"Voice-Gender",        12,6},
	{"Voice-Age",            9,6},
	{"Voice-Variant",       13,6},
	{"Voice-Name",          10,6},
	{"Prosody-Volume",      14,8},
	{"Prosody-Rate",        12,8},
	{"Speech-Marker",       13,7},
	{"Speech-Language",     15,7},
	{"Fetch-Hint",          10,6},
	{"Fetch-Timeout",       13,6},
	{"Audio-Fetch-Hint",    16,0},
	{"Failed-Uri",          10,10},
	{"Failed-Uri_Cause",    16,10},
	{"Speak-Restart",       13,6},
	{"Speak-Length",        12,6},
	{"Load-Lexicon",        12,2},
	{"Lexicon-Search-Order",20,2}
};

/* string table of mrcp speech-unit fields (mrcp_speech_unit_t) */
static const apt_string_table_item_t speech_unit_string_table[] = {
	{"Second",   6,2},
	{"Word",     4,0},
	{"Sentence", 8,2},
	{"Paragraph",9,0}
};

/* string table of mrcp voice-gender fields (mrcp_voice_gender_t) */
static const apt_string_table_item_t voice_gender_string_table[] = {
	{"male",   4,0},
	{"female", 6,0},
	{"neutral",7,0}
};

/* string table of mrcp prosody-volume fields (mrcp_prosody_volume_t) */
static const apt_string_table_item_t prosody_volume_string_table[] = {
	{"silent", 6,1},
	{"x-soft", 6,2},
	{"soft",   4,3},
	{"medium", 6,0},
	{"loud",   4,0},
	{"x-loud", 6,5},
	{"default",7,0} 
};

/* string table of mrcp prosody-rate fields (mrcp_prosody_rate_t) */
static const apt_string_table_item_t prosody_rate_string_table[] = {
	{"x-slow", 6,3},
	{"slow",   4,0},
	{"medium", 6,0},
	{"fast",   4,0},
	{"x-fast", 6,4},
	{"default",7,0}
};

/* string table of mrcp synthesizer completion-cause fields (mrcp_synthesizer_completion_cause_t) */
static const apt_string_table_item_t completion_cause_string_table[] = {
	{"normal",               6,0},
	{"barge-in",             8,0},
	{"parse-failure",       13,0},
	{"uri-failure",         11,0},
	{"error",                5,0},
	{"language-unsupported",20,4},
	{"lexicon-load-failure",20,1},
	{"cancelled",            9,0}
};


static APR_INLINE size_t apt_string_table_value_parse(const apt_string_table_item_t *string_table, size_t count, const char *field)
{
	return apt_string_table_find(string_table,count,field);
}

static size_t apt_string_table_value_generate(const apt_string_table_item_t *string_table, size_t count, size_t id, char *field)
{
	size_t length = 0;
	const char *name;
	name = apt_string_table_get(string_table,count,id);
	if(name) {
		length = strlen(name);
		memcpy(field,name,length);
	}
	return length;
}

/* parses mrcp speech-length value */
static mrcp_status_t mrcp_speech_length_value_parse(mrcp_speech_length_value_t *speech_length, char *value)
{
	if(!value) {
		return MRCP_STATUS_FAILURE;
	}
	switch(*value) {
		case '+': speech_length->type = SPEECH_LENGTH_TYPE_NUMERIC_POSITIVE; break;
		case '-': speech_length->type = SPEECH_LENGTH_TYPE_NUMERIC_NEGATIVE; break;
		default : speech_length->type = SPEECH_LENGTH_TYPE_TEXT;
	}

	if(speech_length->type == SPEECH_LENGTH_TYPE_TEXT) {
		char *str;
		str = apt_read_field(&value,APT_TOKEN_SP,1);
		if(!str) {
			return MRCP_STATUS_FAILURE;
		}
		if(value - str < MAX_TEXT_SPEECH_TAG_LENGTH) {
			strcpy(speech_length->value.tag,str);
		}
	}
	else {
		char *str;
		value++;
		str = apt_read_field(&value,APT_TOKEN_SP,1);
		if(!str) {
			return MRCP_STATUS_FAILURE;
		}
		speech_length->value.numeric.length = apt_size_value_parse(str);

		str = apt_read_field(&value,APT_TOKEN_SP,1);
		if(!str) {
			return MRCP_STATUS_FAILURE;
		}
		speech_length->value.numeric.unit = apt_string_table_value_parse(speech_unit_string_table,SPEECH_UNIT_COUNT,str);
	}
	return MRCP_STATUS_SUCCESS;
}

/* generates mrcp speech-length value */
static size_t mrcp_speech_length_generate(mrcp_speech_length_value_t *speech_length, char *value)
{
	size_t length = 0;
	if(!value) {
		return MRCP_STATUS_FAILURE;
	}

	if(speech_length->type == SPEECH_LENGTH_TYPE_TEXT) {
		length = strlen(speech_length->value.tag);
		memcpy(value,speech_length->value.tag,length);
	}
	else {
		if(speech_length->type == SPEECH_LENGTH_TYPE_NUMERIC_POSITIVE) {
			*value = '+';
		}
		else {
			*value = '-';
		}
		length++;
		length += apt_size_value_generate(speech_length->value.numeric.length,value+length);
		value[length++] = ' ';
		length += apt_string_table_value_generate(speech_unit_string_table,SPEECH_UNIT_COUNT,speech_length->value.numeric.unit,value+length);
	}
	return length;
}

/* generates mrcp synthesizer completion-cause */
static size_t mrcp_completion_cause_generate(mrcp_synthesizer_completion_cause_t completion_cause, char *field)
{
	int length = 0;
	const char *name = apt_string_table_get(completion_cause_string_table,SYNTHESIZER_COMPLETION_CAUSE_COUNT,completion_cause);
	if(!name) {
		return 0;
	}
	length = sprintf(field,"%03"APR_SIZE_T_FMT" %s", completion_cause,name);
	if(length < 0) {
		length = 0;
	}
	return length;
}



/* initializes synthesizer header */
void mrcp_synthesizer_header_init(mrcp_synthesizer_header_t *synthesizer_header)
{
	synthesizer_header->jump_size.type = SPEECH_LENGTH_TYPE_UNKNOWN;
	synthesizer_header->kill_on_barge_in = FALSE;
	synthesizer_header->speaker_profile = NULL;
	synthesizer_header->completion_cause = SYNTHESIZER_COMPLETION_CAUSE_UNKNOWN;
	synthesizer_header->completion_reason = NULL;
	synthesizer_header->voice_param.gender = VOICE_GENDER_UNKNOWN;
	synthesizer_header->voice_param.age = 0;
	synthesizer_header->voice_param.variant = 0;
	synthesizer_header->voice_param.name = NULL;
	synthesizer_header->prosody_param.volume = PROSODY_VOLUME_UNKNOWN;
	synthesizer_header->prosody_param.rate = PROSODY_RATE_UNKNOWN;
	synthesizer_header->speech_marker = NULL;
	synthesizer_header->speech_language = NULL;
	synthesizer_header->fetch_hint = NULL;
	synthesizer_header->audio_fetch_hint = NULL;
	synthesizer_header->fetch_timeout = 0;
	synthesizer_header->failed_uri = NULL;
	synthesizer_header->failed_uri_cause = NULL;
	synthesizer_header->speak_restart = FALSE;
	synthesizer_header->speak_length.type = SPEECH_LENGTH_TYPE_UNKNOWN;
	synthesizer_header->load_lexicon = FALSE;
	synthesizer_header->lexicon_search_order = NULL;
}

/* destroys synthesizer header */
void mrcp_synthesizer_header_destroy(mrcp_synthesizer_header_t *synthesizer_header)
{
	/* nothing to do as data is allocated from the memory pool and
	will be destroyed with memory pool */
	mrcp_synthesizer_header_init(synthesizer_header);
}


/* allocates mrcp synthesizer header */
static void* mrcp_synthesizer_header_accessor_allocate(mrcp_header_base_t *header, apr_pool_t *pool)
{
	mrcp_synthesizer_header_t *synthesizer_header = apr_palloc(pool,sizeof(mrcp_synthesizer_header_t));
	mrcp_synthesizer_header_init(synthesizer_header);
	header->data = synthesizer_header;
	return header->data;
}

/* destroys mrcp synthesizer header */
static void mrcp_synthesizer_header_accessor_destroy(mrcp_header_base_t *header)
{
	if(header->data) {
		mrcp_synthesizer_header_destroy(header->data);
		header->data = NULL;
	}
}


/* parses mrcp synthesizer header */
static mrcp_status_t mrcp_synthesizer_header_parse(mrcp_header_base_t *header, size_t id, char *value, apr_pool_t *pool)
{
	mrcp_status_t status;
	mrcp_synthesizer_header_t *synthesizer_header = header->data;
	if(!synthesizer_header) {
		return MRCP_STATUS_FAILURE;
	}

	status = MRCP_STATUS_SUCCESS;
	switch(id)
	{
		case SYNTHESIZER_HEADER_JUMP_SIZE:
			mrcp_speech_length_value_parse(&synthesizer_header->jump_size,value);
			break;
		case SYNTHESIZER_HEADER_KILL_ON_BARGE_IN:
			apt_boolean_value_parse(value,&synthesizer_header->kill_on_barge_in);
			break;
		case SYNTHESIZER_HEADER_SPEAKER_PROFILE:
			synthesizer_header->speaker_profile = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_COMPLETION_CAUSE:
			synthesizer_header->completion_cause = apt_size_value_parse(value);
			break;
		case SYNTHESIZER_HEADER_COMPLETION_REASON:
			synthesizer_header->completion_reason = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_VOICE_GENDER:
			synthesizer_header->voice_param.gender = apt_string_table_value_parse(voice_gender_string_table,VOICE_GENDER_COUNT,value);
			break;
		case SYNTHESIZER_HEADER_VOICE_AGE:
			synthesizer_header->voice_param.age = apt_size_value_parse(value);
			break;
		case SYNTHESIZER_HEADER_VOICE_VARIANT:
			synthesizer_header->voice_param.variant = apt_size_value_parse(value);
			break;
		case SYNTHESIZER_HEADER_VOICE_NAME:
			synthesizer_header->voice_param.name = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_PROSODY_VOLUME:
			synthesizer_header->prosody_param.volume = apt_string_table_value_parse(prosody_volume_string_table,PROSODY_VOLUME_COUNT,value);
			break;
		case SYNTHESIZER_HEADER_PROSODY_RATE:
			synthesizer_header->prosody_param.rate = apt_string_table_value_parse(prosody_rate_string_table,PROSODY_RATE_COUNT,value);
			break;
		case SYNTHESIZER_HEADER_SPEECH_MARKER:
			synthesizer_header->speech_marker = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_SPEECH_LANGUAGE:
			synthesizer_header->speech_language = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_FETCH_HINT:
			synthesizer_header->fetch_hint = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_AUDIO_FETCH_HINT:
			synthesizer_header->audio_fetch_hint = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_FETCH_TIMEOUT:
			synthesizer_header->fetch_timeout = apt_size_value_parse(value);
			break;
		case SYNTHESIZER_HEADER_FAILED_URI:
			synthesizer_header->failed_uri = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_FAILED_URI_CAUSE:
			synthesizer_header->failed_uri_cause = apr_pstrdup(pool,value);
			break;
		case SYNTHESIZER_HEADER_SPEAK_RESTART:
			apt_boolean_value_parse(value,&synthesizer_header->speak_restart);
			break;
		case SYNTHESIZER_HEADER_SPEAK_LENGTH:
			mrcp_speech_length_value_parse(&synthesizer_header->speak_length,value);
			break;
		case SYNTHESIZER_HEADER_LOAD_LEXICON:
			apt_boolean_value_parse(value,&synthesizer_header->load_lexicon);
			break;
		case SYNTHESIZER_HEADER_LEXICON_SEARCH_ORDER:
			synthesizer_header->lexicon_search_order = apr_pstrdup(pool,value);
			break;
		default:
			status = MRCP_STATUS_FAILURE;
	}
	return status;
}

/* generates mrcp synthesizer header */
static size_t mrcp_synthesizer_header_generate(mrcp_header_base_t *header, size_t id, char *value)
{
	size_t length = 0;
	mrcp_synthesizer_header_t *synthesizer_header = header->data;
	if(!synthesizer_header) {
		return MRCP_STATUS_FAILURE;
	}

	switch(id)
	{
		case SYNTHESIZER_HEADER_JUMP_SIZE:
			if(synthesizer_header->jump_size.type != SPEECH_LENGTH_TYPE_UNKNOWN) {
				length = mrcp_speech_length_generate(&synthesizer_header->jump_size,value);
			}
			break;
		case SYNTHESIZER_HEADER_KILL_ON_BARGE_IN:
			length = apt_boolean_value_generate(synthesizer_header->kill_on_barge_in,value);
			break;
		case SYNTHESIZER_HEADER_SPEAKER_PROFILE:
			if(synthesizer_header->speaker_profile) {
				length = apt_string_value_generate(synthesizer_header->speaker_profile,value);
			}
			break;
		case SYNTHESIZER_HEADER_COMPLETION_CAUSE:
			if(synthesizer_header->completion_cause != SYNTHESIZER_COMPLETION_CAUSE_UNKNOWN) {
				length = mrcp_completion_cause_generate(synthesizer_header->completion_cause,value);
			}
			break;
		case SYNTHESIZER_HEADER_COMPLETION_REASON:
			if(synthesizer_header->completion_reason) {
				length = apt_string_value_generate(synthesizer_header->completion_reason,value);
			}
			break;
		case SYNTHESIZER_HEADER_VOICE_GENDER:
			if(synthesizer_header->voice_param.gender != VOICE_GENDER_UNKNOWN) {
				length = apt_string_table_value_generate(voice_gender_string_table,VOICE_GENDER_COUNT,synthesizer_header->voice_param.gender,value);
			}
			break;
		case SYNTHESIZER_HEADER_VOICE_AGE:
			length = apt_size_value_generate(synthesizer_header->voice_param.age,value);
			break;
		case SYNTHESIZER_HEADER_VOICE_VARIANT:
			length = apt_size_value_generate(synthesizer_header->voice_param.variant,value);
			break;
		case SYNTHESIZER_HEADER_VOICE_NAME:
			if(synthesizer_header->voice_param.name) {
				length = apt_string_value_generate(synthesizer_header->voice_param.name,value);
			}
			break;
		case SYNTHESIZER_HEADER_PROSODY_VOLUME:
			if(synthesizer_header->prosody_param.volume != PROSODY_VOLUME_UNKNOWN) {
				length = apt_string_table_value_generate(prosody_volume_string_table,PROSODY_VOLUME_COUNT,synthesizer_header->prosody_param.volume,value);
			}
			break;
		case SYNTHESIZER_HEADER_PROSODY_RATE:
			if(synthesizer_header->prosody_param.rate != PROSODY_RATE_UNKNOWN) {
				length = apt_string_table_value_generate(prosody_rate_string_table,PROSODY_RATE_COUNT,synthesizer_header->prosody_param.rate,value);
			}
			break;
		case SYNTHESIZER_HEADER_SPEECH_MARKER:
			if(synthesizer_header->speech_marker) {
				length = apt_string_value_generate(synthesizer_header->speech_marker,value);
			}
			break;
		case SYNTHESIZER_HEADER_SPEECH_LANGUAGE:
			if(synthesizer_header->speech_language) {
				length = apt_string_value_generate(synthesizer_header->speech_language,value);
			}
			break;
		case SYNTHESIZER_HEADER_FETCH_HINT:
			if(synthesizer_header->fetch_hint) {
				length = apt_string_value_generate(synthesizer_header->fetch_hint,value);
			}
			break;
		case SYNTHESIZER_HEADER_AUDIO_FETCH_HINT:
			if(synthesizer_header->audio_fetch_hint) {
				length = apt_string_value_generate(synthesizer_header->audio_fetch_hint,value);
			}
			break;
		case SYNTHESIZER_HEADER_FETCH_TIMEOUT:
			length = apt_size_value_generate(synthesizer_header->fetch_timeout,value);
			break;
		case SYNTHESIZER_HEADER_FAILED_URI:
			if(synthesizer_header->failed_uri) {
				length = apt_string_value_generate(synthesizer_header->failed_uri,value);
			}
			break;
		case SYNTHESIZER_HEADER_FAILED_URI_CAUSE:
			if(synthesizer_header->failed_uri_cause) {
				length = apt_string_value_generate(synthesizer_header->failed_uri_cause,value);
			}
			break;
		case SYNTHESIZER_HEADER_SPEAK_RESTART:
			length = apt_boolean_value_generate(synthesizer_header->speak_restart,value);
			break;
		case SYNTHESIZER_HEADER_SPEAK_LENGTH:
			if(synthesizer_header->speak_length.type != SPEECH_LENGTH_TYPE_UNKNOWN) {
				length = mrcp_speech_length_generate(&synthesizer_header->speak_length,value);
			}
			break;
		case SYNTHESIZER_HEADER_LOAD_LEXICON:
			length = apt_boolean_value_generate(synthesizer_header->load_lexicon,value);
			break;
		case SYNTHESIZER_HEADER_LEXICON_SEARCH_ORDER:
			if(synthesizer_header->lexicon_search_order) {
				length = apt_string_value_generate(synthesizer_header->lexicon_search_order,value);
			}
			break;
		default:
			length = 0;
	}
	return length;
}

/* duplicates mrcp synthesizer header */
static mrcp_status_t mrcp_synthesizer_header_duplicate(mrcp_header_base_t *header, const mrcp_header_base_t *src, size_t id, apr_pool_t *pool)
{
	mrcp_status_t status;
	mrcp_synthesizer_header_t *synthesizer_header = header->data;
	const mrcp_synthesizer_header_t *src_synthesizer_header = src->data;
	if(!synthesizer_header || !src_synthesizer_header) {
		return MRCP_STATUS_FAILURE;
	}

	status = MRCP_STATUS_SUCCESS;
	switch(id)
	{
		case SYNTHESIZER_HEADER_JUMP_SIZE:
			synthesizer_header->jump_size = src_synthesizer_header->jump_size;
			break;
		case SYNTHESIZER_HEADER_KILL_ON_BARGE_IN:
			synthesizer_header->kill_on_barge_in = src_synthesizer_header->kill_on_barge_in;
			break;
		case SYNTHESIZER_HEADER_SPEAKER_PROFILE:
			synthesizer_header->speaker_profile = apr_pstrdup(pool,src_synthesizer_header->speaker_profile);
			break;
		case SYNTHESIZER_HEADER_COMPLETION_CAUSE:
			synthesizer_header->completion_cause = src_synthesizer_header->completion_cause;
			break;
		case SYNTHESIZER_HEADER_COMPLETION_REASON:
			synthesizer_header->completion_reason = apr_pstrdup(pool,src_synthesizer_header->completion_reason);
			break;
		case SYNTHESIZER_HEADER_VOICE_GENDER:
			synthesizer_header->voice_param.gender = src_synthesizer_header->voice_param.gender;
			break;
		case SYNTHESIZER_HEADER_VOICE_AGE:
			synthesizer_header->voice_param.age = src_synthesizer_header->voice_param.age;
			break;
		case SYNTHESIZER_HEADER_VOICE_VARIANT:
			synthesizer_header->voice_param.variant = src_synthesizer_header->voice_param.variant;
			break;
		case SYNTHESIZER_HEADER_VOICE_NAME:
			synthesizer_header->voice_param.name = apr_pstrdup(pool,src_synthesizer_header->voice_param.name);
			break;
		case SYNTHESIZER_HEADER_PROSODY_VOLUME:
			synthesizer_header->prosody_param.volume = src_synthesizer_header->prosody_param.volume;
			break;
		case SYNTHESIZER_HEADER_PROSODY_RATE:
			synthesizer_header->prosody_param.rate = src_synthesizer_header->prosody_param.rate;
			break;
		case SYNTHESIZER_HEADER_SPEECH_MARKER:
			synthesizer_header->speech_marker = apr_pstrdup(pool,src_synthesizer_header->speech_marker);
			break;
		case SYNTHESIZER_HEADER_SPEECH_LANGUAGE:
			synthesizer_header->speech_language = apr_pstrdup(pool,src_synthesizer_header->speech_language);
			break;
		case SYNTHESIZER_HEADER_FETCH_HINT:
			synthesizer_header->fetch_hint = apr_pstrdup(pool,src_synthesizer_header->fetch_hint);
			break;
		case SYNTHESIZER_HEADER_AUDIO_FETCH_HINT:
			synthesizer_header->audio_fetch_hint = apr_pstrdup(pool,src_synthesizer_header->audio_fetch_hint);
			break;
		case SYNTHESIZER_HEADER_FETCH_TIMEOUT:
			synthesizer_header->fetch_timeout = src_synthesizer_header->fetch_timeout;
			break;
		case SYNTHESIZER_HEADER_FAILED_URI:
			synthesizer_header->failed_uri = apr_pstrdup(pool,src_synthesizer_header->failed_uri);
			break;
		case SYNTHESIZER_HEADER_FAILED_URI_CAUSE:
			synthesizer_header->failed_uri_cause = apr_pstrdup(pool,src_synthesizer_header->failed_uri_cause);
			break;
		case SYNTHESIZER_HEADER_SPEAK_RESTART:
			synthesizer_header->speak_restart = src_synthesizer_header->speak_restart;
			break;
		case SYNTHESIZER_HEADER_SPEAK_LENGTH:
			synthesizer_header->speak_length = src_synthesizer_header->speak_length;
			break;
		case SYNTHESIZER_HEADER_LOAD_LEXICON:
			synthesizer_header->load_lexicon = src_synthesizer_header->load_lexicon;
			break;
		case SYNTHESIZER_HEADER_LEXICON_SEARCH_ORDER:
			synthesizer_header->lexicon_search_order = apr_pstrdup(pool,src_synthesizer_header->lexicon_search_order);
			break;
		default:
			status = MRCP_STATUS_FAILURE;
	}
	return status;
}



static APR_INLINE void mrcp_resource_header_accessor_set(mrcp_header_accessor_t *header_accessor)
{
	header_accessor->allocator = mrcp_synthesizer_header_accessor_allocate;
	header_accessor->destructor = mrcp_synthesizer_header_accessor_destroy;
	header_accessor->field_parser = mrcp_synthesizer_header_parse;
	header_accessor->field_generator = mrcp_synthesizer_header_generate;
	header_accessor->field_duplicator = mrcp_synthesizer_header_duplicate;
	header_accessor->field_table = synthesizer_header_string_table;
	header_accessor->field_count = SYNTHESIZER_HEADER_COUNT;
}


/* initializes mrcp synthesizer resource */
mrcp_status_t mrcp_synthesizer_init(mrcp_resource_t *resource, mrcp_version_t version)
{
	resource->method_table = synthesizer_method_string_table;
	resource->method_count = SYNTHESIZER_METHOD_COUNT;

	resource->event_table = synthesizer_event_string_table;
	resource->event_count = SYNTHESIZER_EVENT_COUNT;

	resource->audio_stream_type = MRCP_AUDIO_STREAM_SOURCE;

	mrcp_resource_header_accessor_set(&resource->header_accessor);
	return MRCP_STATUS_SUCCESS;
}
