/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_header_base.h"

/* parses header base */
mrcp_status_t mrcp_header_base_parse(mrcp_header_base_t *header, apt_name_value_pair_t *name_value_pair, apr_pool_t *pool)
{
	size_t id;
	
	if(!header->accessor) {
		return MRCP_STATUS_FAILURE;
	}

	id = apt_string_table_find(header->accessor->field_table,header->accessor->field_count,name_value_pair->name);
	if(id >= header->accessor->field_count) {
		return MRCP_STATUS_FAILURE;
	}

	if(header->accessor->field_parser(header,id,name_value_pair->value,pool) == MRCP_STATUS_FAILURE) {
		return MRCP_STATUS_FAILURE;
	}
	
	mrcp_header_property_add(&header->property_set,id);
	return MRCP_STATUS_SUCCESS;
}


/* generates header base */
mrcp_status_t mrcp_header_base_generate(mrcp_header_base_t *header, apt_text_stream_t *text_stream)
{
	const char *name;
	size_t i;
	mrcp_header_property_set_t property_set;

	if(!header->accessor) {
		return MRCP_STATUS_FAILURE;
	}

	property_set = header->property_set;
	for(i=0; i<header->accessor->field_count && property_set != 0; i++) {
		if(mrcp_header_property_check(&property_set,i) == MRCP_STATUS_SUCCESS) {
			name = apt_string_table_get(header->accessor->field_table,header->accessor->field_count,i);
			if(!name) {
				continue;
			}
			
			text_stream->pos += apt_name_value_generate_name(name,text_stream->pos);
			text_stream->pos += header->accessor->field_generator(header,i,text_stream->pos);
			apt_insert_end_of_line(text_stream);
			
			mrcp_header_property_remove(&property_set,i);
		}
	}
	return MRCP_STATUS_SUCCESS;
}

/* sets header base */
mrcp_status_t mrcp_header_base_set(mrcp_header_base_t *header, const mrcp_header_base_t *src, mrcp_header_property_set_t mask, apr_pool_t *pool)
{
	size_t i;
	mrcp_header_property_set_t property_set = src->property_set;

	if(!header->accessor || !src->accessor) {
		return MRCP_STATUS_FAILURE;
	}

	mrcp_header_base_allocate(header,pool);

	property_set = src->property_set;
	for(i=0; i<src->accessor->field_count && property_set != 0; i++) {
		if(mrcp_header_property_check(&property_set,i) == MRCP_STATUS_SUCCESS) {
			if(mrcp_header_property_check(&mask,i) == MRCP_STATUS_SUCCESS) {
				header->accessor->field_duplicator(header,src,i,pool);
				mrcp_header_property_add(&header->property_set,i);
			}
			
			mrcp_header_property_remove(&property_set,i);
		}
	}
	return MRCP_STATUS_SUCCESS;
}

/* inherits header base */
mrcp_status_t mrcp_header_base_inherit(mrcp_header_base_t *header, const mrcp_header_base_t *parent, apr_pool_t *pool)
{
	size_t i;
	mrcp_header_property_set_t property_set;

	if(!header->accessor || !parent->accessor) {
		return MRCP_STATUS_FAILURE;
	}

	mrcp_header_base_allocate(header,pool);

	property_set = parent->property_set;
	for(i=0; i<parent->accessor->field_count && property_set != 0; i++) {
		if(mrcp_header_property_check(&property_set,i) == MRCP_STATUS_SUCCESS) {
			if(mrcp_header_property_check(&header->property_set,i) != MRCP_STATUS_SUCCESS) {
				header->accessor->field_duplicator(header,parent,i,pool);
				mrcp_header_property_add(&header->property_set,i);
			}
			
			mrcp_header_property_remove(&property_set,i);
		}
	}
	return MRCP_STATUS_SUCCESS;
}
