/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "codec_manager.h"
#include "g711/g711.h"

static apr_status_t g711_open(codec_handle_t *handle)
{
	handle->frame_size = CODEC_FRAME_TIME_BASE * handle->descriptor->sampling_rate / 1000;
	return APR_SUCCESS;
}

static apr_status_t g711_close(codec_handle_t *handle)
{
	return APR_SUCCESS;
}

static apr_status_t g711u_encode(codec_handle_t *handle, const codec_frame_t *frame_in, codec_frame_t *frame_out)
{
	const short *decode_buf;
	unsigned char *encode_buf;
	apr_uint32_t i;

	decode_buf = frame_in->buffer;
	encode_buf = frame_out->buffer;

	frame_out->size = frame_in->size / sizeof(short);

	for(i=0; i<frame_out->size; i++) {
		encode_buf[i] = linear_to_ulaw(decode_buf[i]);
	}

	return APR_SUCCESS;
}

static apr_status_t g711u_decode(codec_handle_t *handle, const codec_frame_t *frame_in, codec_frame_t *frame_out)
{
	short *decode_buf;
	const unsigned char *encode_buf;
	apr_uint32_t i;

	decode_buf = frame_out->buffer;
	encode_buf = frame_in->buffer;

	frame_out->size = frame_in->size * sizeof(short);

	for(i=0; i<frame_in->size; i++) {
		decode_buf[i] = ulaw_to_linear(encode_buf[i]);
	}

	return APR_SUCCESS;
}

static apr_status_t g711a_encode(codec_handle_t *handle, const codec_frame_t *frame_in, codec_frame_t *frame_out)
{
	const short *decode_buf;
	unsigned char *encode_buf;
	apr_uint32_t i;

	decode_buf = frame_in->buffer;
	encode_buf = frame_out->buffer;

	frame_out->size = frame_in->size / sizeof(short);

	for(i=0; i<frame_out->size; i++) {
		encode_buf[i] = linear_to_alaw(decode_buf[i]);
	}

	return APR_SUCCESS;
}

static apr_status_t g711a_decode(codec_handle_t *handle, const codec_frame_t *frame_in, codec_frame_t *frame_out)
{
	short *decode_buf;
	const unsigned char *encode_buf;
	apr_uint32_t i;

	decode_buf = frame_out->buffer;
	encode_buf = frame_in->buffer;

	frame_out->size = frame_in->size * sizeof(short);

	for(i=0; i<frame_in->size; i++) {
		decode_buf[i] = alaw_to_linear(encode_buf[i]);
	}

	return APR_SUCCESS;
}

static apr_size_t g711_dissect(codec_handle_t *handle, void *buffer, apr_size_t size, codec_frame_t *frames, apr_size_t max_frames)
{
	apr_size_t frame_count = 0;

	while(size >= handle->frame_size && frame_count < max_frames) {
		frames[frame_count].buffer = buffer;
		frames[frame_count].size = handle->frame_size;
		frame_count++;
		
		buffer = (char*)buffer + handle->frame_size;
		size -= handle->frame_size;
	}

	return frame_count;
}

static codec_manipulator_t g711u_manipulator = {
	g711_open,
	g711_close,
	g711u_encode,
	g711u_decode,
	g711_dissect
};

static codec_manipulator_t g711a_manipulator = {
	g711_open,
	g711_close,
	g711a_encode,
	g711a_decode,
	g711_dissect
};

static codec_descriptor_t g711u_descriptor = {
	0,
	"PCMU",
	8000,
	1,
	NULL
};

static codec_descriptor_t g711a_descriptor = {
	8,
	"PCMA",
	8000,
	1,
	NULL
};

apr_status_t g711u_codec_init(codec_manager_t *codec_manager)
{
	return codec_manager_codec_register(codec_manager,&g711u_descriptor,&g711u_manipulator);
}

apr_status_t g711a_codec_init(codec_manager_t *codec_manager)
{
	return codec_manager_codec_register(codec_manager,&g711a_descriptor,&g711a_manipulator);
}
