/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifdef WIN32
#pragma warning(disable: 4127)
#endif
#include <apr_ring.h>
#include "apt_ptr_list.h"

struct apt_list_elem_t {
	APR_RING_ENTRY(apt_list_elem_t) link;
	void                           *data;
};

struct apt_ptr_list_t {
	APR_RING_HEAD(apt_list_head_t, apt_list_elem_t) head;
	apr_pool_t                                     *pool;
};



APT_DECLARE(apt_ptr_list_t*) apt_list_create(apr_pool_t *pool)
{
	apt_ptr_list_t *list = apr_palloc(pool, sizeof(apt_ptr_list_t));
	list->pool = pool;
	APR_RING_INIT(&list->head, apt_list_elem_t, link);
	return list;
}

APT_DECLARE(void) apt_list_destroy(apt_ptr_list_t *list)
{
	/* nothing to do */
}

APT_DECLARE(apt_list_elem_t*) apt_list_push_back(apt_ptr_list_t *list, void *data)
{
	apt_list_elem_t *elem = apr_palloc(list->pool,sizeof(apt_list_elem_t));
	elem->data = data;

	APR_RING_INSERT_TAIL(&list->head,elem,apt_list_elem_t,link);
	return elem;
}

APT_DECLARE(void*) apt_list_pop_front(apt_ptr_list_t *list)
{
	apt_list_elem_t *elem;
	if(APR_RING_EMPTY(&list->head,apt_list_elem_t,link)) {
		return NULL;
	}
	elem = APR_RING_FIRST(&list->head);
	APR_RING_REMOVE(elem,link);
	return elem->data;
}

APT_DECLARE(void*) apt_list_head(apt_ptr_list_t *list)
{
	apt_list_elem_t *elem;
	if(APR_RING_EMPTY(&list->head,apt_list_elem_t,link)) {
		return NULL;
	}
	elem = APR_RING_FIRST(&list->head);
	return elem->data;
}

APT_DECLARE(void*) apt_ptr_list_tail(apt_ptr_list_t *list)
{
	apt_list_elem_t *elem;
	if(APR_RING_EMPTY(&list->head,apt_list_elem_t,link)) {
		return NULL;
	}
	elem = APR_RING_LAST(&list->head);
	return elem->data;
}

APT_DECLARE(apt_list_elem_t*) apt_list_first_elem_get(apt_ptr_list_t *list)
{
	if(APR_RING_EMPTY(&list->head,apt_list_elem_t,link)) {
		return NULL;
	}
	return APR_RING_FIRST(&list->head);
}

APT_DECLARE(apt_list_elem_t*) apt_list_last_elem_get(apt_ptr_list_t *list)
{
	if(APR_RING_EMPTY(&list->head,apt_list_elem_t,link)) {
		return NULL;
	}
	return APR_RING_LAST(&list->head);
}

APT_DECLARE(apt_list_elem_t*) apt_list_next_elem_get(apt_ptr_list_t *list, apt_list_elem_t *elem)
{
	apt_list_elem_t *next_elem = APR_RING_NEXT(elem,link);
	if(next_elem == APR_RING_SENTINEL(list,apt_list_elem_t,link)) {
		next_elem = NULL;
	}
	return next_elem;
}

APT_DECLARE(apt_list_elem_t*) apt_list_prev_elem_get(apt_ptr_list_t *list, apt_list_elem_t *elem)
{
	apt_list_elem_t *prev_elem = APR_RING_PREV(elem,link);
	if(prev_elem == APR_RING_SENTINEL(list,apt_list_elem_t,link)) {
		prev_elem = NULL;
	}
	return prev_elem;
}

APT_DECLARE(apt_list_elem_t*) apt_list_elem_insert(apt_ptr_list_t *list, apt_list_elem_t *elem, void *data)
{
	apt_list_elem_t *new_elem = apr_palloc(list->pool,sizeof(apt_list_elem_t));
	new_elem->data = data;
	APR_RING_INSERT_BEFORE(elem,new_elem,link);
	return new_elem;
}

APT_DECLARE(apt_list_elem_t*) apt_list_elem_remove(apt_ptr_list_t *list, apt_list_elem_t *elem)
{
	apt_list_elem_t *next_elem = APR_RING_NEXT(elem,link);
	APR_RING_REMOVE(elem,link);
	if(next_elem == APR_RING_SENTINEL(list,apt_list_elem_t,link)) {
		next_elem = NULL;
	}
	return next_elem;
}

APT_DECLARE(void*) apt_list_elem_data_get(apt_list_elem_t *elem)
{
	return elem->data;
}
