/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_TASK_MSG_H__
#define __APT_TASK_MSG_H__

/**
 * @file apt_task_msg.h
 * @brief Task Message Definition
 *        Task message is a inter-task communication object,
 *        which is used to signal the message from producer task to consumer task.
 */ 

#include "apt_task.h"

APT_BEGIN_EXTERN_C

/** Opaque task message declaration */
typedef struct apt_task_msg_t apt_task_msg_t;
/** Opaque task message pool declaration */
typedef struct apt_task_msg_pool_t apt_task_msg_pool_t;

/** Task message is used to signal the message from producer task to consumer task */
struct apt_task_msg_t {
	/** Message pool the task message is allocated from */
	apt_task_msg_pool_t *msg_pool;
	/** Context specific message handle */
	void                *msg_handle;
	/** Context specific data */
	char                 data[1];
};

/** Create pool of task messages with dynamic allocation of messages (no actual pool is created) */
APT_DECLARE(apt_task_msg_pool_t*) apt_task_msg_pool_create_dynamic(apr_size_t msg_size, apr_pool_t *pool);

/** Create pool of task messages with static waitable allocation of messages */
APT_DECLARE(apt_task_msg_pool_t*) apt_task_msg_pool_create_waitable_static(apr_size_t msg_size, apr_pool_t *pool);

/** Create pool of task messages with static allocation of messages */
APT_DECLARE(apt_task_msg_pool_t*) apt_task_msg_pool_create_static(apr_size_t msg_size, apr_size_t msg_pool_size, apr_pool_t *pool);

/** Destroy pool of task messages */
APT_DECLARE(void) apt_task_msg_pool_destroy(apt_task_msg_pool_t *msg_pool);


/** Acquire task message from task message pool */
APT_DECLARE(apt_task_msg_t*) apt_task_msg_acquire(apt_task_msg_pool_t *task_msg_pool);

/** Inquire whether task message is blocking */
APT_DECLARE(apt_bool_t) apt_task_msg_is_blocking(apt_task_msg_t *task_msg);

/** Wait for task message to be processed (released) */
APT_DECLARE(void) apt_task_msg_wait(apt_task_msg_t *task_msg);

/** Realese task message */
APT_DECLARE(void) apt_task_msg_release(apt_task_msg_t *task_msg);

APT_END_EXTERN_C

#endif /*__APT_TASK_MSG_H__*/
