/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_SERVER_PROTO_AGENT_H__
#define __MRCP_SERVER_PROTO_AGENT_H__

/**
 * @file mrcp_server_proto_agent.h
 * @brief MRCP Server Protocol Agent Definition.
 */ 

#include "mrcp_server_types.h"
#include "mrcp_module.h"
#include "mrcp_connection.h"

MRCP_BEGIN_EXTERN_C

typedef struct mrcp_proto_agent_method_set_t mrcp_proto_agent_method_set_t;
typedef struct mrcp_proto_agent_event_set_t mrcp_proto_agent_event_set_t;

/** MRCP protocol agent */
struct mrcp_proto_agent_t {
	/** Base module */
	mrcp_module_t                              module;

	/** Set of signaling agent virtual methods */
	const mrcp_proto_agent_method_set_t *agent_method_set;
	/** Set of connection virtual events */
	const mrcp_connection_event_set_t   *connection_event_set;
	/** Actual data set by inherited object */
	void                                       *object;

	/** MRCP resource container */
	mrcp_resource_container_t                  *resource_container;
};


/** Set of protocol agent virtual methods */
struct mrcp_proto_agent_method_set_t {
	/* Create new or return existing shared connection */
	mrcp_connection_t* (*connect)(mrcp_proto_agent_t *agent, mrcp_signaling_channel_t *channel, const char *remote_ip);
};


/** Perform validity check for specified agent */
static APR_INLINE mrcp_status_t mrcp_server_proto_agent_validity_check(mrcp_proto_agent_t *agent)
{
	return mrcp_module_validity_check(&agent->module);
}

/** Initialize MRCP response message by MRCP request message */
static APR_INLINE void mrcp_response_init_by_request(mrcp_message_t *response_message, const mrcp_message_t *request_message)
{
	mrcp_message_init_by_request(response_message,request_message);

	response_message->start_line.message_type = MRCP_MESSAGE_TYPE_RESPONSE;
	response_message->start_line.request_state = MRCP_REQUEST_STATE_COMPLETE;
	response_message->start_line.status_code = MRCP_STATUS_CODE_SUCCESS;

	response_message->start_line.method_id = request_message->start_line.method_id;
	response_message->start_line.method_name = request_message->start_line.method_name;
}


MRCP_END_EXTERN_C

#endif /*__MRCP_SERVER_PROTO_AGENT_H__*/
