/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_SIGNALING_CHANNEL_H__
#define __MRCP_SIGNALING_CHANNEL_H__

/**
 * @file mrcp_signaling_channel.h
 * @brief MRCP Signaling Channel Definition.
 */ 

#include "mrcp_description.h"

MRCP_BEGIN_EXTERN_C

typedef struct mrcp_signaling_channel_method_set_t mrcp_signaling_channel_method_set_t;
typedef struct mrcp_signaling_channel_event_set_t mrcp_signaling_channel_event_set_t;

/** Signaling channel base */
struct mrcp_signaling_channel_t {
	/** Set of signaling channel virtual methods */
	const mrcp_signaling_channel_method_set_t *method_set;
	/** Set of signaling channel event handlers */
	const mrcp_signaling_channel_event_set_t  *event_set;
	/** Actual data set by inherited object */
	void                                      *object;

	/** Back-pointer to signaling agent (stored by reference) */
	mrcp_signaling_agent_t                    *agent;
	/** Back-pointer to session (stored by reference) */
	mrcp_session_t                            *session;

	/** Local descriptor */
	mrcp_descriptor_t                          local_descriptor;
	/** Remote descriptor */
	mrcp_descriptor_t                          remote_descriptor;

	apr_pool_t                                *pool;
};

/** Set of signaling channel virtual methods */
struct mrcp_signaling_channel_method_set_t {
	/** Offer local description to remote party */
	mrcp_status_t (*offer)(mrcp_signaling_channel_t *channel);
	/** Answer to offer, by setting up local description according to the remote one */
	mrcp_status_t (*answer)(mrcp_signaling_channel_t *channel);
	/** Terminate signaling channel */
	mrcp_status_t (*terminate)(mrcp_signaling_channel_t *channel);

	/** Destroy signaling channel (channel must be terminated prior to destruction) */
	mrcp_status_t (*destroy)(mrcp_signaling_channel_t *channel);
};

/** Set of signaling channel event handlers */
struct mrcp_signaling_channel_event_set_t {
	/** Received offer from remote party */
	mrcp_status_t (*on_offer)(mrcp_signaling_channel_t *channel);
	/** Received answer from remote party */
	mrcp_status_t (*on_answer)(mrcp_signaling_channel_t *channel);
	/** Terminated signaling channel */
	mrcp_status_t (*on_terminate)(mrcp_signaling_channel_t *channel);
};

MRCP_END_EXTERN_C

#endif /*__MRCP_SIGNALING_CHANNEL_H__*/
