/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_CONSUMER_TASK_H__
#define __APT_CONSUMER_TASK_H__

/**
 * @file apt_consumer_task.h
 * @brief Consumer Task Definition
 *        Consumer task processes incoming messages from message queue.
 */ 

#include "apt_task_msg.h"

APT_BEGIN_EXTERN_C

/** Opaque consumer task declaration */
typedef struct apt_consumer_task_t apt_consumer_task_t;

/** Prototype of task message handler */
typedef apt_bool_t (*apt_task_msg_handler)(void *object, apt_task_msg_t *task_msg);

/**
 * Create consumer task.
 * @param object the object to pass to msg_handler
 * @param msg_handler the main message handler routine
 * @param pool the pool to allocate consumer task from
 * @return the created consumer task
 */
APT_DECLARE(apt_consumer_task_t*) apt_consumer_task_create(void *object, 
                    apt_task_msg_handler msg_handler, apr_pool_t *pool);

/**
 * Destroy consumer task.
 * @param consumer_task the consumer task to destroy
 */
APT_DECLARE(apt_bool_t) apt_consumer_task_destroy(apt_consumer_task_t *consumer_task);

/**
 * Start consumer task (wait for messages from incoming message queue and process them).
 * @param consumer_task the consumer task to start
 */
APT_DECLARE(apt_bool_t) apt_consumer_task_start(apt_consumer_task_t *consumer_task);

/**
 * Terminate consumer task.
 * @param consumer_task the consumer task to terminate
 */
APT_DECLARE(apt_bool_t) apt_consumer_task_terminate(apt_consumer_task_t *consumer_task);

/**
 * Set running state.
 * @param consumer_task the consumer task to operate on
 * @param running TRUE - keep running, FALSE - safe to terminate
 */
APT_DECLARE(void) apt_consumer_task_running_set(apt_consumer_task_t *consumer_task, apt_bool_t running);

/**
 * Signal (post) the message to consumer task.
 * @param consumer_task the consumer task to be signalled
 * @param task_msg the message to pass
 */
APT_DECLARE(apt_bool_t) apt_consumer_task_signal(apt_consumer_task_t *consumer_task, apt_task_msg_t *task_msg);

/**
 * Retrieve task instance.
 * @param consumer_task the consumer task to get instance from
 */
APT_DECLARE(apt_task_t*) apt_consumer_task_get(apt_consumer_task_t *consumer_task);


APT_END_EXTERN_C

#endif /*__APT_CONSUMER_TASK_H__*/
